/*****************************************************************************
 ****                                                                     ****
 **** atree.c  for Unix                                                   ****
 ****                                                                     ****
 **** atree release 2.0                                                   ****
 **** Adaptive Logic Network (ALN) simulation program.                    ****
 **** Copyright (C) A. Dwelly, R. Manderscheid, W.W. Armstrong, 1991.     ****
 ****                                                                     ****
 **** License:                                                            ****
 **** A royalty-free license is granted for the use of this software for  ****
 **** NON_COMMERCIAL PURPOSES ONLY. The software may be copied and/or     ****
 **** modified provided this notice appears in its entirety and unchanged ****
 **** in all derived source programs.  Persons modifying the code are     ****
 **** requested to state the date, the changes made and who made them     ****
 **** in the modification history.                                        ****
 ****                                                                     ****
 **** Patent License:                                                     ****
 **** The use of a digital circuit which transmits a signal indicating    ****
 **** heuristic responsibility is protected by U. S. Patent 3,934,231     ****
 **** and others assigned to Dendronic Decisions Limited of Edmonton,     ****
 **** W. W. Armstrong, President.  A royalty-free license is granted      ****
 **** by the company to use this patent for NON_COMMERCIAL PURPOSES ONLY  ****
 **** to adapt logic trees using this program and its modifications.      ****
 ****                                                                     ****
 **** Limited Warranty:                                                   ****
 **** This software is provided "as is" without warranty of any kind,     ****
 **** either expressed or implied, including, but not limited to, the     ****
 **** implied warrantees of merchantability and fitness for a particular  ****
 **** purpose.  The entire risk as to the quality and performance of the  ****
 **** program is with the user.  Neither the authors, nor the             ****
 **** University of Alberta, its officers, agents, servants or employees  ****
 **** shall be liable or responsible in any way for any damage to         ****
 **** property or direct personal or consequential injury of any nature   ****
 **** whatsoever that may be suffered or sustained by any licensee, user  ****
 **** or any other party as a consequence of the use or disposition of    ****
 **** this software.                                                      ****
 ****                                                                     ****
 **** Modification history:                                               ****
 ****                                                                     ****
 **** 90.05.09 Initial implementation, A.Dwelly                           ****
 **** 91.07.15 Release 2, Rolf Manderscheid                               ****
 ****          Thanks to Allen Supynuk for turning my compression         ****
 ****          function into something readable.                          ****
 **** 91.10.30 0.0 and 1.0 replaced by code -> low and code -> high in    ****
 ****          atree_read_code to correct bug found by Monroe Thomas      ****
 ****                                                                     ****
 *****************************************************************************/

/*****************************************************************************
 ****                                                                     ****
 **** Include Files                                                       ****
 ****                                                                     ****
 *****************************************************************************/

#include <stdio.h>
#include <math.h>
#include <assert.h>
#include <ctype.h>
#include "atree.h"

extern char *malloc();
extern int errno;

/*****************************************************************************
 ****                                                                     ****
 **** Constants                                                           ****
 ****                                                                     ****
 *****************************************************************************/

/* Node and leaf types */

#define AND 0
#define RIGHT 1
#define LEFT 2
#define OR 3
#define LEAF 4

#define LEFT_CHILD 0
#define RIGHT_CHILD 1

#define FALSE 0           /* As usual */
#define TRUE 1
#define UNEVALUATED 2     /* We complete the lattice of boolean functions */
#define ERROR 3

/* Number of retries before an error in atree_rand_walk */
#define MAX_RETRY 50

/* Number of nodes/leaves to allocate in each call to malloc in get_node() */
#define NEWMALLOC	1024

/* Initialisation values */

#define MAXSET		63
#define ABOVEMID	32
#define BELOWMID	31
#define MINSET		0


#define STACKSIZE	100		/* stack used in load_tree */
#define LEAF_TOKEN 256

#define CODING_HEADER_FORMAT	"vectors=%d, width=%d, range=%g,%g\n"
#define CODING_HEADER_ITEMS		4

/*
 * Macros
 */

/* This keeps lint quiet */

#define Printf (void) printf 

/* Public and private procedures */

#define public
#define private static

/* Infinite loops - for EVER */

#define EVER (;;)

/* Printing out the boolean lattice */

#define PRINTBOOL(b) if (b == TRUE) {Printf("1");} else if (b == FALSE)\
    {Printf("0");} else if (b == UNEVALUATED) {Printf("U");} else \
    if (b == ERROR) {Printf("E");} else {Printf("?");}

/* Verbosity */

#define VERBOSE(level,s) if (verbosity >= level) {s ;}

/*
 * Types
 */

typedef int bool_t;      /*
                          * Only TRUE, FALSE, UNEVALUATED or ERROR
                          * are used in these variables
                          */

typedef struct token_struct
{
    int token;
    int bit_no;
    int complemented;
} token_t;

/*
 * Preliminary Private Procedure Declarations
 */

private atree *get_node();
private void reclaim_node();
private atree *build_tree();
private void print_tree(); 
private bool_t train();
private void adapt();
private int node_function();
private void compress_tree();
private int count_leaves();
private int store_tree();
private int get_token();

/*****************************************************************************
 *****************************************************************************
 ****                                                                     ****
 ****                        Public Routines                              ****
 ****                                                                     ****
 *****************************************************************************
 *****************************************************************************/

/*****************************************************************************
 ****                                                                     ****
 **** void atree_init()                                                   ****
 ****                                                                     ****
 **** Synopsis:                                                           ****
 ****                                                                     ****
 **** Initialises various global variables, and makes an initial call to  ****
 **** the random number seed routine.                                     ****
 *****************************************************************************/

public void
atree_init()
{
    (void) srand(getpid());
}

/*****************************************************************************
 ****                                                                     ****
 **** bit_vec *atree_rand_walk(num,width,p)                               ****
 ****                                                                     ****
 **** int num;                                                            ****
 **** int width;                                                          ****
 **** int p;                                                              ****
 ****                                                                     ****
 **** Synopsis:                                                           ****
 ****                                                                     ****
 **** Creates an array of _num_ binary vectors, of _width_ bits where     ****
 **** each is _p_ bits away from its neighbours in Hamming space. Each    ****
 **** vector is unique.                                                   ****
 **** This routine returns NULL if it's unable to find enough vectors.    ****
 *****************************************************************************/

public bit_vec * 
atree_rand_walk(num,width,p)

int num;
int width;
int p;

{
    /*
     * An initial vector is produced with random bits, and each subsequent 
     * vector in the random walk just has _p_ bits flipped. In order to
     * guarantee that each vector is unique, it is checked against
     * a chained list of vectors of the same bit sum. If it is not already
     * in the chain it is appended to the end, or else the vector is discarded
     * and the process repeats itself.
     */

    bit_vec *walk;              /* An array of bit vectors */
    bit_vec *pckd_vec;          /* The packed unique ? vector */ 
    bool_t unique;              /* Uniqueness flag set during testing */
    int *bit_sum_chain;         /* Pointers to vectors of equivalent bit sums */
    int *next_vec;              /* Chain array */
    char *unpckd_vec;           /* An unpacked vector */
    char *this_vec;             /* An unpacked vector */
    char *mark_vec;             /* TRUE where a bit has been flipped */
    int i;
    int j;
    int old_bit_sum;            /* Last number of TRUE bits in vector */
    int bit_sum;                /* Current number of TRUE bits in vector */
    int retrys;                 /* Number of attempts to find a unique vec */
    int victim;                 /* The bit just twiddled */
    int current_vec;            /* Part of the chain */

    assert(num > 0);
    assert(width > 0);
    assert(p > 0 && p <= width);

    /* Assign space in memory */

    walk = (bit_vec *) malloc((unsigned) num * sizeof(bit_vec));
    MEMCHECK(walk);
    bit_sum_chain = (int *) malloc((unsigned) (width+1) * sizeof(int));
    MEMCHECK(bit_sum_chain);
    next_vec = (int *) malloc((unsigned) num * sizeof(int));
    MEMCHECK(next_vec);
    unpckd_vec = (char *) malloc((unsigned) width * sizeof(char));
    MEMCHECK(unpckd_vec);
    this_vec = (char *) malloc((unsigned) width * sizeof(char));
    MEMCHECK(this_vec);
    mark_vec = (char *) malloc((unsigned) width * sizeof(char));
    MEMCHECK(mark_vec); 

    /* Clear bit_sum_chain */

    for (i = 0; i <= width; i++)
    {
        bit_sum_chain[i] = -1;
    }

    /* Clear next_vec */

    for (i = 0; i < num; i++)
    {
        next_vec[i] = -1;
    }

    /* Create initial vector and bit sum */

    old_bit_sum = 0;
    for (i = 0; i < width; i++)
    {
        if (unpckd_vec[i] = RANDOM(2))
        {
           old_bit_sum++;
        }
    }

    walk[0] = *(bv_pack(unpckd_vec,width));
    bit_sum_chain[old_bit_sum] = 0;

    /* Random walk */

    for (i = 1; i < num; i++)
    {
        retrys = 0;
        unique = FALSE;
        while ((!unique) && (retrys < MAX_RETRY)) 
        {
            retrys++;

            /* Make a new vector */

            bit_sum = old_bit_sum;
            for (j = 0; j < width; j++)
            {
                this_vec[j] = unpckd_vec[j];
                mark_vec[j] = FALSE;
            }

            for (j = 0; j < p; j++)
            {
                do
                {
                    victim = RANDOM(width);
                }
                while (mark_vec[victim]);

                mark_vec[victim] = TRUE;
                this_vec[victim] = !this_vec[victim];
                if (this_vec[victim] == FALSE)
                {
                    bit_sum--;
                }
                else
                {   
                    bit_sum++;
                }
            }
            
            pckd_vec = bv_pack(this_vec,width);

            /* Compare it with the existing ones of equal bit length */

            if (bit_sum_chain[bit_sum] == -1)
            {
                /* There are no other vectors with this bit sum, so */

                unique = TRUE;
                walk[i] = *pckd_vec;
                bit_sum_chain[bit_sum] = i;     
                next_vec[i] = -1;
            }
            else
            {
                current_vec = bit_sum_chain[bit_sum];
                for EVER /* We break out of here inside the loop */ 
                { 
                    if (bv_equal(&(walk[current_vec]),pckd_vec))
                    {
                        /* This vector already exists,  unique = FALSE; */
                        break;
                    }
                    else
                    {
                        if (next_vec[current_vec] == -1)
                        { 
                            unique = TRUE;
                            walk[i] = *pckd_vec;
                            next_vec[current_vec] = i;
                            next_vec[i] = -1;
                            break;
                        }
                        else
                        {
                            current_vec = next_vec[current_vec];
                        }
                    }
                } /* for EVER */
            } /* if (bit_sum_chain...) */
        } /* while ((!unique... ))*/

        /*
         * If Unique is TRUE at this point, we have a unique
         * vector stored, we have to set up old_bit sum and unpckd_vec
         * for the next vector.
         * If unique is FALSE, we have tried to create a unique vector
         * MAX_RETRY times, and failed. We therefore signal an error.
         */
        
        if (unique)
        {
            for (j = 0; j < width; j++)
            {
                unpckd_vec[j] = this_vec[j];
            }
            old_bit_sum = bit_sum;
        }
        else
        {
            (void) free((char *) walk);
            walk = NULL;
            break;
        }
    } /* for */


    /* Free space in memory */

    (void) free((char *) bit_sum_chain);
    (void) free((char *) next_vec);
    (void) free((char *) unpckd_vec);
    (void) free((char *) this_vec);
    (void) free((char *) mark_vec);

    /* Return final vector */

    return(walk);
}

/*****************************************************************************
 ****                                                                     ****
 **** atree *atree_create(numvars,leaves)                                 ****
 ****                                                                     ****
 **** int numvars;                                                        ****
 **** int leaves;                                                         ****
 ****                                                                     ****
 **** Synopsis:                                                           ****
 ****                                                                     ****
 **** Creates an Armstrong tree with _leaves_ number of leaves connected  ****
 **** to _numvars_  variables and their complements.  The number of       ****
 **** leaves should probably be at least twice _numvars_.                 ****
 **** The node functions and the connections are picked at random.        ****
 ****                                                                     ****
 *****************************************************************************/

public atree *
atree_create(numvars,leaves)

int numvars;
int leaves;

{
    atree *tree;
    int *connection;
    bool_t *complemented;
    bool_t comp;
    int i;

    assert(leaves > 0);
    assert(numvars > 0);

    /* 
     * Create a list of bit inputs and shuffle, complemented inputs
     * are marked with a TRUE in the complemented array.
     */

    connection = (int *) malloc((unsigned) sizeof(int) * leaves);
    MEMCHECK(connection);
    complemented = (bool_t *) malloc((unsigned) sizeof(int) * leaves);
    MEMCHECK(complemented);

    comp = FALSE;
    for (i = 0; i < leaves; i++)
    {
        int numvarscount = i % numvars;
        if (numvarscount == 0)
        {
            comp = !comp;
        }
        connection[i] = numvarscount;
        complemented[i] = comp;
    }

    /* Shuffle */

    for (i = 0; i < leaves; i++)
    {
        int a;
        int b;
        int tmp;

        a = RANDOM(leaves);
        b = RANDOM(leaves);
        tmp = connection[a];
        connection[a] = connection[b];
        connection[b] = tmp;
        tmp = complemented[a];
        complemented[a] = complemented[b];
        complemented[b] = tmp;
    }

    tree = build_tree(connection, complemented, 0, leaves - 1);

    (void) free((char *) connection);
    (void) free((char *) complemented);

    return(tree);
}

/*****************************************************************************
 ****                                                                     ****
 **** void atree_free(tree)                                               ****
 ****                                                                     ****
 **** atree *tree;                                                        ****
 ****                                                                     ****
 **** Synopsis:                                                           ****
 ****                                                                     ****
 **** Returns memory occupied by _tree_ to the freelists.                 ****
 *****************************************************************************/

public void
atree_free(tree)
atree *tree;
{
    if (tree -> c_tag != LEAF)
    {
        atree_free(tree -> n_child[LEFT_CHILD]);
        atree_free(tree -> n_child[RIGHT_CHILD]);
    }
    reclaim_node(tree);
}

/*****************************************************************************
 ****                                                                     ****
 **** (int) atree_eval(tree,vec)                                          ****
 ****                                                                     ****
 **** atree *tree;                                                        ****
 **** bit_vec *vec;                                                       ****
 ****                                                                     ****
 **** Synopsis:                                                           ****
 ****                                                                     ****
 **** Applies the _tree_ to the bit vector _vec_ and returns the result,  ****
 **** 1 for true, and 0 for false.                                        ****
 *****************************************************************************/

public bool_t
atree_eval(tree,vec)

atree *tree;
bit_vec *vec;

{
    register bool_t result; 

    switch (tree -> c_tag)
    {
    case AND:
        tree -> n_sig_right = UNEVALUATED;
        result = (tree -> n_sig_left =
                     atree_eval(tree -> n_child[LEFT_CHILD], vec))
              && (tree -> n_sig_right =
                     atree_eval(tree -> n_child[RIGHT_CHILD], vec));
        break;

    case RIGHT:
        tree -> n_sig_left = UNEVALUATED;
        result = (tree -> n_sig_right =
                     atree_eval(tree -> n_child[RIGHT_CHILD], vec));
        break;

    case LEFT:
        tree -> n_sig_right = UNEVALUATED;
        result = (tree -> n_sig_left =
                     atree_eval(tree -> n_child[LEFT_CHILD], vec));
        break;

    case OR:
        tree -> n_sig_right = UNEVALUATED;
        result = (tree -> n_sig_left =
                     atree_eval(tree -> n_child[LEFT_CHILD], vec))
              || (tree -> n_sig_right =
                     atree_eval(tree -> n_child[RIGHT_CHILD], vec));
        break;

    case LEAF:
        result = bv_extract((int) tree -> l_bit_no, vec) != tree -> l_comp;
        break;

    }

    return(result);
}

/*****************************************************************************
 ****                                                                     ****
 **** bool_t atree_train(tree,tset,correct_result,bit_col,tset_size,      ****
 ****                    no_correct,epochs,verbosity)                     ****
 ****                                                                     ****
 **** atree *tree;                                                        ****
 **** bit_vec *tset;                                                      ****
 **** bit_vec *correct_result;                                            ****
 **** int bit_col;                                                        ****
 **** int tset_size;                                                      ****
 **** int no_correct;                                                     ****
 **** int epochs;                                                         ****
 **** int verbosity;                                                      ****
 ****                                                                     ****
 **** Synopsis:                                                           ****
 ****                                                                     ****
 **** This routine is the heart of the library. It takes an atree _tree_  ****
 **** to be trained, an array of input vectors _tset_, an array of        ****
 **** vectors, _correct_result_ where each bit column holds a correct bit ****
 **** result for each vector in _tset_. [Note: Only a single vector is    ****
 **** actually required here, but doing it this way make life convenient  ****
 **** for training collections of trees for real numbers] An integer      ****
 **** _bit_col_ denoting the column to be trained on. Another integer     ****
 **** _test_size gives the size of both the _tset_ and _correct_result_   ****
 **** arrays.                                                             ****
 ****                                                                     ****
 **** The _tree_ is trained until the number of vectors presented in      ****
 **** _tset_ equals _epoch_ epochs, or the last presentation of the number****
 **** in an epoch had at least _no_correct_ presentations.                ****
 **** The _verbosity_ is currently 0 or 1.                                ****
 **** The routine returns TRUE if it stopped because it succeeded         ****
 **** _no_correct_ times.                                                 ****
 ****                                                                     ****
 *****************************************************************************/

public bool_t
atree_train(tree,tset,correct_result,bit_col,tset_size,no_correct,
            epochs,verbosity)

atree *tree;
bit_vec *tset;
bit_vec *correct_result;
int bit_col;
int tset_size;
int no_correct;
int epochs;
int verbosity;

{
    bool_t no_correct_flag = FALSE;
    int *vec_list;
    int i; 
    int j;
    int cor_this_epoch;
    int vec_num;
    bit_vec *vec;

    vec_list = (int *) malloc((unsigned) sizeof(int) * tset_size);
    MEMCHECK(vec_list);

    for (i = 0; i < tset_size; i++)
    {
        vec_list[i] = i;
    }

    /* For the specified number of epochs */

    for (i = 0; i < epochs; i++)
    {
        cor_this_epoch = 0;

        VERBOSE(1,Printf("Epoch : %d\n",i));

        /* Shuffle */

        for (j = 0; j < tset_size; j++)
        {
            int a;
            int b;
            int tmp;

            a = RANDOM(tset_size);
            do
            {
                b = RANDOM(tset_size);
            }
            while (a == b);

            tmp = vec_list[a];
            vec_list[a] = vec_list[b];
            vec_list[b] = tmp;
        }

        /* For the elements of the test set */

        for (j = 0; j < tset_size; j++)
        {
            /* Pick a random vector */

            vec_num = vec_list[j];
            vec = tset + vec_num;

            /* Train the tree */

            if (train(tree,vec,bv_extract(bit_col,correct_result + vec_num)))
            {
                /* The tree succeeded */

                cor_this_epoch++;
                if (cor_this_epoch == no_correct)
                {
                    no_correct_flag = TRUE;
                    break; /* out of this epoch */

                } /* if (cor_this...) */
            } /* if (train..) */
        } /* for (j = ..) */

        VERBOSE(1,Printf("Estimated number correct: %d\n",cor_this_epoch));

        /* If no_correct_flag is TRUE here, its time to stop */

        if (no_correct_flag || i == epochs - 1)
        {
            int correct = 0;

            for (j = 0; j < tset_size; j++)
            {
                if (atree_eval(tree, tset + j)
                    == bv_extract(bit_col, correct_result + j))
                {
                    correct++;
                }
            }

            VERBOSE(1,Printf("Actual number correct: %d\n", correct));

            if (correct >= no_correct)
            {
                break; /* out of the epoch loop */
            }
        }
    } /* for (i = ...) */

    (void) free((char *) vec_list);
    return(no_correct_flag);
}

/*****************************************************************************
 ****                                                                     ****
 **** (void) atree_print(tree,verbosity)                                  ****
 ****                                                                     ****
 **** atree *tree;                                                        ****
 ****                                                                     ****
 **** Synopsis:                                                           ****
 ****                                                                     ****
 **** Prints out a tree for diagnostic and explanation purposes, the      ****
 **** verbosity levels are 0 or above.                                    ****
 *****************************************************************************/

public void
atree_print(tree,verbosity)

atree *tree;
int verbosity;

{
    /*
     * This routine makes an immediate call to the private
     * print tree routine that takes an extra parameter
     * controlling the indentation.
     */

    print_tree(tree,0,verbosity);
}

/*
 * function: atree_fold
 *
 * This function removes all LEFT and RIGHT nodes from _tree_
 * and returns a pointer to the resulting tree.
 */
public atree *
atree_fold(tree)
atree *tree;
{
    atree *folded_tree;
    int keep;

    switch (tree -> c_tag)
    {
    case LEAF:
        folded_tree = tree;
        break;
    case OR:
    case AND:
        tree -> n_child[LEFT_CHILD] = atree_fold(tree -> n_child[LEFT_CHILD]);
        tree -> n_child[RIGHT_CHILD] = atree_fold(tree -> n_child[RIGHT_CHILD]);
        folded_tree = tree;
        break;
    case LEFT:
    case RIGHT:
        keep = (tree -> c_tag == LEFT) ? LEFT_CHILD : RIGHT_CHILD;
        folded_tree = atree_fold(tree -> n_child[keep]);
        atree_free(tree -> n_child[!keep]);
        reclaim_node(tree);
        break;
    default:
        assert(0);
    }

    return(folded_tree);
}

/*
 * function: atree_compress
 *
 * This function converts an atree to a fast_tree.
 * See the commentary for the private function
 * compress_tree for more information.
 */
public fast_tree *
atree_compress(tree)
atree *tree;
{
    fast_tree *ftree;
    int leaf_num;
    int leaf_count = count_leaves(tree, TRUE);

    ftree = (fast_tree *)
            malloc((unsigned) (leaf_count + 1) * sizeof(fast_tree));
    MEMCHECK(ftree);
    ftree[leaf_count].bit_no = -1;  /* mark end */
    leaf_num = leaf_count - 1;
    compress_tree(tree, NULL, NULL, ftree, &leaf_num);
    return(ftree);
}

/*
 * function: atree_fast_eval
 *
 * This function is the fast_tree equivalent of atree_eval.
 */
public int
atree_fast_eval(tree, vec)
register fast_tree *tree;
bit_vec *vec;
{
    register int result;

    do
    {
        result = bv_extract(tree -> bit_no, vec) != tree -> comp;
    } while ((tree = tree -> next[result]) != NULL);

    return(result);
}

public void
atree_fast_print(ftree)
fast_tree *ftree;
{
    int i;

    for (i = 0; ftree[i].bit_no >= 0; i++)
    {
        Printf("[%3d] bit=%s%d, next[0]=%d, next[1]=%d\n",
               i, ftree[i].comp ? "!" : "", ftree[i].bit_no,
               ftree[i].next[0] ? ftree[i].next[0] - ftree : -1,
               ftree[i].next[1] ? ftree[i].next[1] - ftree : -1);
    }
}

/*
 * function: atree_set_code
 *
 * This is the constructor for type code_t.
 * Returns non-zero on failure.
 */
public int
atree_set_code(code, low, high, count, width, dist)
code_t *code;
double low;
double high;
int count;
int width;
int dist;
{
    static bit_vec *zero_vec = NULL;
    static bit_vec *one_vec = NULL;
    int error = FALSE;

    assert(low < high);
    assert(count > 1);
    assert(width > 0);
    assert(dist > 0);

    code -> width = width;
    code -> low = low;
    code -> high = high;

    if (width == 1)		/* boolean */
    {
        code -> vector = (bit_vec *) malloc((unsigned) sizeof(bit_vec) * 2);
        MEMCHECK(code -> vector);
        if (zero_vec == NULL)
        {
            zero_vec = bv_create(1);
            one_vec = bv_create(1);
            bv_set(0, zero_vec, 0);
            bv_set(0, one_vec, 1);
        }
        code -> vector[0] = *zero_vec;
        code -> vector[1] = *one_vec;
        code -> vector_count = 2;
        code -> step = high - low;
    }
    else
    {
        if ((code -> vector = atree_rand_walk(count, width, dist)) == NULL)
        {
            error = TRUE;
        }
        code -> vector_count = count;
        code -> step = (high - low) / (count - 1);
    }
    return(error);
}

/*
 * atree_encode:
 *
 * returns the quantization level corresponding to the floating point
 * value _x_.  To get the bit vector, use the expression:
 *
 *   code -> vector + atree_encode(x, code)
 */

public int
atree_encode(x, code)
double x;
code_t *code;
{
    int index;

    if (code -> width == 1)
    {
        index = x > (code -> low + code -> high) / 2;
    }
    else if (x < code -> low)
    {
        (void) fprintf(stderr,
                       "warning: atree_encode: argument out of range: %g\n", x);
        index = 0;
    }
    else if (x > code -> high)
    {
        (void) fprintf(stderr,
                       "warning: atree_encode: argument out of range: %g\n", x);
        index = code -> vector_count - 1;
    }
    else
    {
        index = (int) rint((x - code -> low) / code -> step);
    }

    return(index);
}

/*
 * atree_decode:
 *
 * returns the quantization level corresponding to the bit vector
 * _vec_.  To get the corresponding floating point value, use the
 * expression:
 *
 *   code -> low + code -> step * atree_decode(vec, code)
 */

public int
atree_decode(vec, code)
bit_vec *vec;
code_t *code;
{
    int closest = 0;

    if (code -> width == 1)		/* boolean */
    {
        closest = bv_extract(0, vec);
    }
    else
    {
        int i;
        int diff;
        int closest_bit_diff = code -> width;

        for (i = 0; i < code -> vector_count; i++)
        {
            if ((diff = bv_diff(vec, code -> vector + i)) < closest_bit_diff)
            {
                closest_bit_diff = diff;
                closest = i;
            }
        }
    }
    return(closest);
}


/*
 * atree I/O routines.
 */

public int
atree_store(tree, filename)
atree *tree;
char *filename;
{
    FILE *fp;

    if ((fp = fopen(filename, "w")) == NULL)
    {
        return(errno);
    }
    atree_write(fp, tree);
    fclose(fp);
    return(0);
}

public atree *
atree_load(filename)
char *filename;
{
    FILE *fp;
    atree *tree;

    if ((fp = fopen(filename, "r")) == NULL)
    {
        return(NULL);
    }
    tree = atree_read(fp);
    fclose(fp);
    return(tree);
}

/*
 * function: atree_write
 *
 * Stores a single _tree_ onto _stream_.
 * returns 0 for success, 1 on failure.
 */
public int
atree_write(stream, tree)
FILE *stream;
atree *tree;
{
    return store_tree(tree, stream) || fprintf(stream, ";\n") == EOF;
}

/*
 * function: atree_read
 *
 * Reads tree stored in postfix notation.  Valid symbols are:
 * '&', '|', 'L', 'R' (AND, OR, LEFT, and RIGHT respectively),
 * and numerals (representing bit numbers) optionally preceded
 * by a '!' for negation.  Returns NULL if any error or EOF is
 * encountered.  A file may store multiple trees, each tree is 
 * separated by a ';'.
 */

public atree *
atree_read(stream)
FILE *stream;
{
    token_t t;
    atree *node = NULL;
    int error = 0;

    atree *stack[STACKSIZE];
    int top = 0;
#define ITEMS		top
#define PUSH(node)	stack[top++] = (node)
#define POP			stack[--top]

    while ((error = get_token(stream, &t)) == 0)
    {
        if (t.token == EOF || t.token == ';')
        {
            break;
        }

        if (t.token == LEAF_TOKEN)
        {
            node = get_node(LEAF);
            node -> l_bit_no = t.bit_no;
            node -> l_comp = t.complemented;
        }
        else if (ITEMS < 2)
        {
            (void) fprintf(stderr,
                           "atree_read: too few arguments for %c, offset %ld\n",
                           t.token, ftell(stream));
            error++;
            break;
        }
        else
        {
            node = get_node(!LEAF);
            node -> n_cnt_left = (t.token == '|' || t.token == 'L')
                                 ? MAXSET : MINSET;
            node -> n_cnt_right = (t.token == '|' || t.token == 'R')
                                 ? MAXSET : MINSET;
            node -> c_tag = node_function(node);
            node -> n_sig_left = UNEVALUATED;
            node -> n_sig_right = UNEVALUATED;
            node -> n_child[RIGHT_CHILD] = POP;
            node -> n_child[LEFT_CHILD] = POP;
        }

        if (ITEMS >= STACKSIZE)
        {
            (void) fprintf(stderr, "atree_read: stack overflow\n");
            error++;
            break;
        }
        else
        {
            PUSH(node);
        }

    } /* while */

    if (!error && ITEMS != 1)
    {
        (void) fprintf(stderr, "atree_read: unexpected ");
        if (t.token == EOF)
        {
            (void) fprintf(stderr, "EOF\n");
        }
        else
        {
            (void) fprintf(stderr, "';', offset %ld\n", ftell(stream));
        }
        error++;
    }

    return(error ? NULL : node);

#undef ITEMS
#undef PUSH
#undef POP
}

/*
 * code I/O routines
 */

/*
 * function: atree_write_code
 *
 * writes _code_ onto _stream_.
 * returns 0 for success, 1 on failure.
 */
public int
atree_write_code(stream, code)
FILE *stream;
code_t *code;
{
    int i;
    int error;

    error = fprintf(stream, CODING_HEADER_FORMAT,
                    code -> vector_count, code -> width,
                    code -> low, code -> high) == EOF;

    if (!error && code -> width > 1)
    {
        for (i = 0; i < code -> vector_count; i++)
        {
            if (bv_print(stream, code -> vector + i)
            || fprintf(stream, "\n") == EOF)
            {
                error = TRUE;
                break;
            }
        }
    }
    return(error);
}

public code_t *
atree_read_code(stream, code)
FILE *stream;
code_t *code;
{
    char *buf;
    char *p;
    int i;
    int error = 0;

    if (fscanf(stream, CODING_HEADER_FORMAT,
               &code -> vector_count, &code -> width,
               &code -> low, &code -> high) != CODING_HEADER_ITEMS)
    {
        (void) fprintf(stderr, "atree_read_code: bad header, offset %ld\n",
                       ftell(stream));
        return(NULL);
    }
    else if (code -> vector_count < 2)
    {
        (void) fprintf(stderr,
                       "atree_read_code: vector count too low, offset %ld\n",
                       ftell(stream));
        return(NULL);
    }
    else if (code -> high <= code -> low)
    {
        (void) fprintf(stderr,
                       "atree_read_code: bad range, offset %ld\n",
                       ftell(stream));
        return(NULL);
    }
    else if (code -> width < 1)
    {
        (void) fprintf(stderr,
                      "atree_read_code: width must be at least 1, offset %ld\n",
                      ftell(stream));
        return(NULL);
    }

    if (code -> width == 1)
    {
        atree_set_code(code, code -> low, /* low */
                             code -> high, /* high */
                             2,   /* count */
                             1,   /* width */
                             1);  /* distance */
    }
    else {

        code -> step = (code->high - code->low ) / (code -> vector_count - 1);
        code -> vector = (bit_vec *)
          malloc((unsigned) sizeof(bit_vec) * code -> vector_count);
        MEMCHECK(code -> vector);

        buf = malloc((unsigned) code -> width + 2); /* room for \n and \0 */
        MEMCHECK(buf);

        for (i = 0; i < code -> vector_count; i++)
        {
            if (fgets(buf, code -> width + 2, stream) == NULL)
            {
                (void) fprintf(stderr,
                               "atree_read_code: read error on offset %ld\n",
                               ftell(stream));
                error++;
                break;
            }

            if (strlen(buf) != code -> width + 1
            || buf[code -> width] != '\n')
            {
                (void) fprintf(stderr,
                          "atree_read_code: inconsistent vector length, offset %ld\n",
                          ftell(stream));
                error++;
                break;
            }
            else
            {
                buf[code -> width] = '\0'; /* remove \n */
            }

            if (strspn(buf, "01") != code -> width)
            {
                (void) fprintf(stderr,
                               "atree_read_code: garbage at offset %ld\n",
                               ftell(stream));
                error++;
                break;
            }

            /*
             * prepare the vector for packing
             */
            for (p = buf; *p; p++)
            {
                *p -= '0';
            }
            code -> vector[i] = *(bv_pack(buf, code -> width));
        }

        (void) free(buf);
        if (error)
        {
            (void) free((char *) code -> vector);
            code = NULL;
        }
    }

    return(code);
}


/*****************************************************************************
 *****************************************************************************
 ****                                                                     ****
 ****                        Private Routines                             ****
 ****                                                                     ****
 *****************************************************************************
 *****************************************************************************/

/*
 * Free list management routines.
 */
static atree_leaf *free_leaf_list = NULL;
static atree_node *free_node_list = NULL;

#define get_free_list(type, free_list) { \
  int i; \
  type *ptr = (type *) malloc(sizeof(type) * NEWMALLOC); \
  MEMCHECK(ptr); \
  for (i = 0; i < NEWMALLOC - 1; i++) \
    ptr[i].next = &ptr[i + 1]; \
  ptr[NEWMALLOC - 1].next = free_list; \
  free_list = ptr; \
}

private atree *
get_node(tag)
int tag;
{
    atree *node;

    if (tag == LEAF)
    {

        /*
         * add to free_leaf_list if necessary.
         */
        if (free_leaf_list == NULL)
        {
            get_free_list(atree_leaf, free_leaf_list);
        }

        node = (atree *) free_leaf_list;
        free_leaf_list = free_leaf_list -> next;
    }
    else {

        /*
         * add to free_node_list if necessary.
         */
        if (free_node_list == NULL)
        {
            get_free_list(atree_node, free_node_list);
        }

        node = (atree *) free_node_list;
        free_node_list = free_node_list -> next;
    }

    node -> c_tag = tag;
    return(node);
}
#undef get_free_list

private void
reclaim_node(tree)
atree *tree;
{
    if (tree -> c_tag == LEAF)
    {
        tree -> leaf.next = free_leaf_list;
        free_leaf_list = (atree_leaf *) tree;
    }
    else
    {
        tree -> node.next = free_node_list;
        free_node_list = (atree_node *) tree;
    }
}

private atree *
build_tree(connection, complemented, start, end)

int *connection;
bool_t *complemented;
int start;
int end;
{
    atree *node;
    int mid;


    /* Are we at a leaf? */

    if (start == end)
    {
        node = get_node(LEAF);
        node -> l_comp = complemented[start];
        node -> l_bit_no = connection[start];
    }
    else
    {
        /* This is a node. */

        node = get_node(AND);
        node -> c_tag = RANDOM(4);

        node -> n_cnt_left =
          (node -> c_tag == LEFT || node -> c_tag == OR) ? ABOVEMID : BELOWMID;
        node -> n_cnt_right =
          (node -> c_tag == RIGHT || node -> c_tag == OR) ? ABOVEMID : BELOWMID;
   
        /* clear the signals */

        node -> n_sig_right = UNEVALUATED;
        node -> n_sig_left = UNEVALUATED;

        /* Create descendants */

        mid = start + ((end - start)/2);
        node -> n_child[LEFT_CHILD]
           = build_tree(connection, complemented, start, mid);
        node -> n_child[RIGHT_CHILD]
           = build_tree(connection, complemented, mid+1, end);
    }
    return(node);
}

private void
print_tree(tree,indent,verbosity)

atree *tree;
int indent;
int verbosity;

{
    int i;

    if (tree -> c_tag != LEAF)
    {
        print_tree(tree -> n_child[RIGHT_CHILD], indent + 3, verbosity);
    }

    for (i = 0; i < indent; i++)
    {
        Printf(" ");
    }

    switch (tree -> c_tag)
    {
    case AND:
        Printf("AND\n");
        break;

    case LEFT:
        Printf("LEFT\n");
        break;

    case RIGHT:
        Printf("RIGHT\n");
        break;

    case OR:
        Printf("OR\n");
        break;

    case LEAF:
        Printf("%sLEAF : %d\n",
               tree -> l_comp ? "~" : "", tree -> l_bit_no);
        break;
    }

    if (tree -> c_tag != LEAF)
    {
       print_tree(tree -> n_child[LEFT_CHILD], indent + 3, verbosity);
    }
}

/*
 * This routine is actually responsible for the training of a tree for a 
 * single input vector and desired result. If the tree gets the correct result
 * before the adaptation step occurs, the routine returns TRUE, otherwise,
 * false.
 */

private bool_t 
train(tree,vec,desired)

atree *tree;
bit_vec *vec;
bool_t desired;

{
    bool_t correct;

    (void) atree_eval(tree,vec);

    adapt(tree,vec,desired);

    correct = atree_eval(tree,vec) == desired;

    if (!correct)
    {
        adapt(tree,vec,desired);
    }

    return(correct);
}

private void
adapt(tree,vec,desired)

atree *tree;
bit_vec *vec;
bool_t desired;

{
    register int lres;
    register int rres;
    register int funct;

/* 
 * INCR and DECR implement bounded counters.
 */
#define INCR(t) ((t) += ((t) < MAXSET))
#define DECR(t) ((t) -= ((t) > MINSET))

    funct = tree -> c_tag;

    if (funct != LEAF)
    {
        /* If the left child is unevaluated, evaluate it */
    
        if ((lres = tree -> n_sig_left) == UNEVALUATED)
        {
            lres = atree_eval(tree -> n_child[LEFT_CHILD], vec);
            tree -> n_sig_left = lres;
        }
    
        /* If the right child is unevaluated, evaluate it */
    
        if ((rres = tree -> n_sig_right) == UNEVALUATED)
        {
            rres = atree_eval(tree -> n_child[RIGHT_CHILD], vec);
            tree -> n_sig_right = rres;
        }
    
        /* Update the counters if needed */
    
        if (lres != rres)
        {
            if (lres)
            {
                (void) (desired ? INCR(tree -> n_cnt_left)
                                : DECR(tree -> n_cnt_left));
            }
            else
            {
                (void) (desired ? INCR(tree -> n_cnt_right)
                                : DECR(tree -> n_cnt_right));
            }
    
            /* has the node function changed? */
    
            tree -> c_tag = node_function(tree);
            funct = tree -> c_tag;
        }
    
        /* Assign responsibility to the left child */
    
        if (rres != desired || funct != (rres ? OR : AND))
        {
            adapt(tree -> n_child[LEFT_CHILD], vec, desired);
        }
    
        /* Assign responsibility to the right child */
    
        if (lres != desired || funct != (lres ? OR : AND))
        {
            adapt(tree -> n_child[RIGHT_CHILD], vec, desired);
        }
    }
#undef INCR
#undef DECR
}

private int
node_function(tree)

atree *tree;

{
    int result;

    if ((tree -> n_cnt_left) >= ABOVEMID)
    {
        result = (tree -> n_cnt_right >= ABOVEMID) ? OR : LEFT;
    }
    else
    {
        result = (tree -> n_cnt_right >= ABOVEMID) ? RIGHT : AND;
    }

    return(result);
}

/*
 * compress_tree:
 *   Alters the respresentation of an atree into a fast_tree.
 * A fast_tree is essentially a list of leaves; each leaf in the
 * list contains two pointers to subsequent leaves to evaluate,
 * one for each possible result of evaluating the current leaf.
 * A next pointer of NULL indicates that the evaluation is complete,
 * and the result of the tree is the result of the current leaf.
 *
 * parameters:
 *   'next_if_0' and 'next_if_1' each hold the location of the
 * next leaf to evaluate if the value of the current 'node' is
 * known.  Of course, for the root these are NULL.  'leaf_num' is the
 * current index into 'ftree'; it starts at the last leaf.
 *
 * notes:
 *   For non-leaf nodes, we traverse the children in reverse order
 * because we need to know the first leaf of a node's sibling (this
 * is the next leaf to evaluate if any children of 'node' produce a
 * tag value).  If we go backwards, we know that this is the last
 * leaf we visited.
 */

private void
compress_tree(node, next_if_0, next_if_1, ftree, leaf_num)
atree *node;
fast_tree *next_if_0, *next_if_1;
fast_tree ftree[];
int *leaf_num;
{

    assert(*leaf_num >= 0);
    switch (node -> c_tag)
    {
    case LEAF:
        ftree[*leaf_num].next[0] = next_if_0;
        ftree[*leaf_num].next[1] = next_if_1;
        ftree[*leaf_num].bit_no = node -> l_bit_no;
        ftree[*leaf_num].comp = node -> l_comp;
        (*leaf_num)--;
        break;
    case AND:
        compress_tree(node->n_child[1], next_if_0, next_if_1, ftree, leaf_num);
        compress_tree(node->n_child[0], next_if_0, &ftree[*leaf_num+1], ftree,
                      leaf_num);
        break;
    case OR:
        compress_tree(node->n_child[1], next_if_0, next_if_1, ftree, leaf_num);
        compress_tree(node->n_child[0], &ftree[*leaf_num+1], next_if_1, ftree,
                      leaf_num);
        break;
    case LEFT:                                                     
        compress_tree(node->n_child[0], next_if_0, next_if_1, ftree, leaf_num);
        break;
    case RIGHT:
        compress_tree(node->n_child[1], next_if_0, next_if_1, ftree, leaf_num);
        break;
    default:
        assert(0);
    }
}

private int
count_leaves(tree, fold)
atree *tree;
int fold;
{
    if (tree -> c_tag == LEAF)
    {
        return 1;
    }
    else if (fold && tree -> c_tag == LEFT)
    {
        return count_leaves(tree -> n_child[LEFT_CHILD], fold);
    }
    else if (fold && tree -> c_tag == RIGHT)
    {
        return count_leaves(tree -> n_child[RIGHT_CHILD], fold);
    }
    else
    {
        return count_leaves(tree -> n_child[LEFT_CHILD], fold)
             + count_leaves(tree -> n_child[RIGHT_CHILD], fold);
    }
}

/*
 * function: store_tree
 *
 * Stores _tree_ onto _stream_ using postfix notation.
 * Returns non-zero on failure.
 */
private int
store_tree(tree, stream)
atree *tree;
FILE *stream;
{
    int error;

    if (tree -> c_tag == LEAF)
    {
        error = fprintf(stream, "%s%d ",
                        tree -> l_comp ? "!" : "", tree -> l_bit_no) == EOF;
    }
    else
    {
        error = store_tree(tree -> n_child[LEFT_CHILD], stream)
             || store_tree(tree -> n_child[RIGHT_CHILD], stream);
        if (!error)
        {
            switch (tree -> c_tag)
            {
            case AND:
                error = fprintf(stream, "&") == EOF;
                break;
            case OR:
                error = fprintf(stream, "|") == EOF;
                break;
            case LEFT:
                error = fprintf(stream, "L") == EOF;
                break;
            case RIGHT:
                error = fprintf(stream, "R") == EOF;
                break;
            }
        }
    }
    return(error);
}

/*
 * function: get_token
 *
 * Reads the next token from _stream_ and returns information about it
 * in _t_.  Returns 0 for success, 1 for failure.
 */
private int
get_token(stream, t)
FILE *stream;
token_t *t;
{
    int c;

    /* skip white space */
    while (isspace(c = getc(stream)))
       ;

    t -> complemented = 0;
    switch (c)
    {
    case EOF:
    case 'L':
    case 'R':
    case '&':
    case '|':
    case ';':
        t -> token = c;
      break;
    case '!':
        t -> complemented = 1;
        /* skip white space */
        while (isspace(c = getc(stream)))
           ;
         
        if (!isdigit(c))
        {
            (void) fprintf(stderr,
                           "get_token: expecting digit after '!', offset %ld\n",
                           ftell(stream));
            return(1);
        }
        /* FALLTHRU */

    case '0': case '1': case '2': case '3': case '4':
    case '5': case '6': case '7': case '8': case '9':
        t -> token = LEAF_TOKEN;
        t -> bit_no = c - '0';
        while (isdigit(c = getc(stream)))
        {
            t -> bit_no = t -> bit_no * 10 + c - '0';
        }
        (void) ungetc(c, stream);
        break;

    default:
        (void) fprintf(stderr,
                       "get_token: unexpected symbol: '%c', offset %ld\n",
                       c, ftell(stream));
        return(1);
    }

    return(0);
}
