/*****************************************************************************
 ****                                                                     ****
 **** bv.c                                                                ****
 ****                                                                     ****
 **** atree release 2.0                                                   ****
 **** Adaptive Logic Network (ALN) simulation program.                    ****
 **** Copyright (C) A. Dwelly, R. Manderscheid, W.W. Armstrong, 1991.     ****
 ****                                                                     ****
 **** License:                                                            ****
 **** A royalty-free license is granted for the use of this software for  ****
 **** NON_COMMERCIAL PURPOSES ONLY. The software may be copied and/or     ****
 **** modified provided this notice appears in its entirety and unchanged ****
 **** in all derived source programs.  Persons modifying the code are     ****
 **** requested to state the date, the changes made and who made them     ****
 **** in the modification history.                                        ****
 ****                                                                     ****
 **** Patent License:                                                     ****
 **** The use of a digital circuit which transmits a signal indicating    ****
 **** heuristic responsibility is protected by U. S. Patent 3,934,231     ****
 **** and others assigned to Dendronic Decisions Limited of Edmonton,     ****
 **** W. W. Armstrong, President.  A royalty-free license is granted      ****
 **** by the company to use this patent for NON_COMMERCIAL PURPOSES ONLY  ****
 **** to adapt logic trees using this program and its modifications.      ****
 ****                                                                     ****
 **** Limited Warranty:                                                   ****
 **** This software is provided "as is" without warranty of any kind,     ****
 **** either expressed or implied, including, but not limited to, the     ****
 **** implied warrantees of merchantability and fitness for a particular  ****
 **** purpose.  The entire risk as to the quality and performance of the  ****
 **** program is with the user.  Neither the authors, nor the             ****
 **** University of Alberta, its officers, agents, servants or employees  ****
 **** shall be liable or responsible in any way for any damage to         ****
 **** property or direct personal or consequential injury of any nature   ****
 **** whatsoever that may be suffered or sustained by any licensee, user  ****
 **** or any other party as a consequence of the use or disposition of    ****
 **** this software.                                                      ****
 ****                                                                     ****
 **** Modification history:                                               ****
 ****                                                                     ****
 **** 90.05.09 Initial implementation, A.Dwelly                           ****
 **** 91.07.15 Release 2, Rolf Manderscheid                               ****
 ****                                                                     ****
 *****************************************************************************/

/*****************************************************************************
 ****                                                                     ****
 **** Include Files                                                       ****
 ****                                                                     ****
 *****************************************************************************/

#include <stdio.h>
#include <assert.h>
#include "bv.h"

extern char *malloc();

/* This keeps lint quiet */

#define Printf (void) printf 

/* Public and private procedures */

#define public
#define private static

/* Memory check */

#define MEMCHECK(p) \
   if ((p) == NULL){(void) fprintf(stderr, "Out of memory\n"); exit(1);}

#define BYTE	8 /* Byte size in bits.*/

#define FALSE	0
#define TRUE	1

/*
 * Types
 */

typedef int bool_t;      /*
                          * Only TRUE, FALSE, UNEVALUATED or ERROR
                          * are used in these variables
                          */

/*****************************************************************************
 ****                                                                     ****
 **** bit_vec *bv_create(length)                                          ****
 ****                                                                     ****
 **** int length;                                                         ****
 ****                                                                     ****
 **** Synopsis:                                                           ****
 ****                                                                     ****
 **** Produces a vector of _length_ bits, each one of which has been set  ****
 **** to 0                                                                ****
 *****************************************************************************/

public bit_vec *
bv_create(length)
     
int length;
     
{
    int i;
    bit_vec *out_vec;
  
    assert(length > 0);
  
    out_vec = (bit_vec *)malloc((unsigned)sizeof(bit_vec));
    MEMCHECK(out_vec);

    out_vec -> len = length;
    out_vec -> bv = malloc((unsigned) (length + BYTE - 1) / BYTE);
    MEMCHECK(out_vec -> bv);
  
    for (i = 0; i < (length + BYTE - 1) / BYTE; i++)
    {
        *((out_vec -> bv) + i) = (char) 0;
    }
  
    return(out_vec);
}

/*****************************************************************************
 ****                                                                     ****
 **** bit_vec *bv_pack(unpacked,length)                                   ****
 ****                                                                     ****
 **** char *unpacked;                                                     ****
 **** int length;                                                         ****
 ****                                                                     ****
 **** This routine takes an array _arr_ of zero and one characters        ****
 **** and returns a packed bit vector suitable for use with the other     ****
 **** routines in this library.                                           ****
 *****************************************************************************/

public bit_vec *
bv_pack(unpacked,length)

char *unpacked;
int length;

{

    bit_vec *out_vec;
    char *out_ptr;
    int i;
    int j;
    int bitptr;

    /* Create the structure */

    out_vec = (bit_vec *) malloc(sizeof(bit_vec));
    MEMCHECK(out_vec);

    out_vec -> len = length;
    out_vec -> bv = malloc((unsigned) (length + BYTE - 1) / BYTE);
    MEMCHECK(out_vec -> bv);
    
    /* Pack the vector */

    out_ptr = out_vec -> bv;

    bitptr = 0;
    for (i = 0; i < (length + BYTE - 1) / BYTE; i++)
    {
        *out_ptr = 0;

        for (j = 0; j < BYTE; j++)
        {
            if (bitptr < length)
            {
               *out_ptr |= (unpacked[bitptr] << j);
               bitptr++;
            }
            else
            {
                break;
            }
        }
        out_ptr++;
    }

    /* Return the vector */

    return(out_vec);
}

/*****************************************************************************
 ****                                                                     ****
 **** int bv_diff(v1,v2)                                                  ****
 ****                                                                     ****
 **** bit_vec *v1;                                                        ****
 **** bit_vec *v2;                                                        ****
 ****                                                                     ****
 **** This function returns the number of bits that are unequal in the    ****
 **** two vectors. They must be the same number of bits in each vector    ****
 *****************************************************************************/

public int
bv_diff(v1,v2)

bit_vec *v1;
bit_vec *v2;

{
    int diff = 0;
    int i;

    assert (v1 -> len == v2 -> len);
    for (i = 0; i < v1 -> len; i++)
    {
        if (bv_extract(i,v1) != bv_extract(i,v2))
        {
            diff++;
        }
    }

    return(diff);
}

/*****************************************************************************
 ****                                                                     ****
 **** bit_vec *bv_concat(n,vectors)                                       ****
 ****                                                                     ****
 **** int n;                                                              ****
 **** bit_vec **vectors;                                                  ****
 ****                                                                     ****
 **** Synopsis:                                                           ****
 ****                                                                     ****
 **** Returns the bit vector which is the string concatenation of each    ****
 **** bit vector in _vector_; _n_ is the number of elements in _vector_.  **** 
 *****************************************************************************/

public bit_vec *
bv_concat(n,vector)

int n;
bit_vec **vector;

{
    int size;
    bit_vec *out_vec;
    char *str;
    int i;
    int j;
    int count;

    /* Work out how big the new vector will be */

    size = 0;
    for (i = 0; i < n; i++)
    {
        size += vector[i] -> len;
    }

    /* Unpack the input vectors */

    str = malloc((unsigned) size);
    MEMCHECK(str);

    count = 0;
    for (i = 0; i < n; i++)
    {
        for (j = 0; j < vector[i] -> len; j++)
        {
            str[count] = bv_extract(j,vector[i]);
            count++;
        }
    }

    out_vec = bv_pack(str,size);

    free(str);

    return(out_vec);
}


/*****************************************************************************
 ****                                                                     ****
 **** bit_vec *bv_copy(vector)                                            ****
 ****                                                                     ****
 **** bit_vec *vector;                                                    ****
 ****                                                                     ****
 **** Synopsis:                                                           ****
 ****                                                                     ****
 **** Returns a copy of the bit vector _vector_.                          ****
 *****************************************************************************/

public bit_vec *
bv_copy(vector)

bit_vec *vector;

{
    bit_vec *out_vec;
    int i;

    /* Work out how big the new vector will be */

    out_vec = (bit_vec *) malloc(sizeof(bit_vec));
    MEMCHECK(out_vec);

    out_vec -> len = vector -> len;
    out_vec -> bv = malloc((unsigned) (vector -> len + BYTE - 1) / BYTE);
    MEMCHECK(out_vec -> bv);
    
    for (i = 0; i < (vector -> len + BYTE - 1) / BYTE; i++)
    {
        out_vec -> bv[i] = vector -> bv[i];
    }

    return(out_vec);
}

/*****************************************************************************
 ****                                                                     ****
 **** void bv_set(n,vec,bit)                                              ****
 ****                                                                     ****
 **** int n;                                                              ****
 **** bit_vec *vec;                                                       ****
 **** bool_t bit;                                                         ****
 ****                                                                     ****
 **** Synopsis:                                                           ****
 ****                                                                     ****
 **** Sets bit _n_ in _vec_ to have the value in _bit_.                   ****
 *****************************************************************************/

public void
bv_set(n,vec,bit)

int n;
bit_vec *vec;
bool_t bit;

{
    char mask;
    char *b;

    assert(n >= 0 && n < vec -> len);

    mask = 0x1;
    b = (vec -> bv) + ((int) (n / BYTE));

    if (bit)
    {
        *b |= (mask << (n % BYTE));
    }
    else
    {
        *b &= ~(mask << (n % BYTE));
    }
}

/*****************************************************************************
 ****                                                                     ****
 **** int bv_extract(n,vec)                                               ****
 ****                                                                     ****
 **** int n;                                                              ****
 **** bit_vec *vec;                                                       ****
 ****                                                                     ****
 **** Synopsis:                                                           ****
 ****                                                                     ****
 **** Returns the _n_th bit of _vec_.                                     ****
 *****************************************************************************/

public int
bv_extract(n,vec)

int n;
bit_vec *vec;

{
    register int mask = 0x1;

    assert(n < vec -> len);
    return(((*((vec -> bv) + ((int) (n / BYTE)))) & (mask << (n % BYTE))) != 0); 
}

/*****************************************************************************
 ****                                                                     ****
 **** int bv_print(stream, vector)                                        ****
 ****                                                                     ****
 **** FILE *vector;                                                       ****
 **** bit_vec *vector;                                                    ****
 ****                                                                     ****
 **** Synopsis:                                                           ****
 ****                                                                     ****
 **** Prints out _vector_ in binary, MSB is the rightmost.                ****
 **** Returns 0 for success, 1 on failure.                                ****
 *****************************************************************************/

public int
bv_print(stream, vector)

FILE *stream;
bit_vec *vector;

{
    char *ptr; /* Points to the current char */
    char mask;
    int bits; /* Counts the number of bits output */
    int  i;

    bits = 0; 
    for (ptr = vector -> bv; bits < vector -> len; ptr++) 

    {
        mask = 0x1;
        for (i = 0; i < BYTE; i++)
        {
            if (fprintf(stream, (*ptr & mask) ? "1" : "0") == EOF)
            {
                return(1);
            }
            bits++;
            if (bits == vector -> len)
            {
                break;
            }
            mask <<= 1;
        }
    }
    return(0);
}

/*****************************************************************************
 ****                                                                     ****
 **** void bv_free(vector)                                                ****
 ****                                                                     ****
 **** bit_vec *vector;                                                    ****
 ****                                                                     ****
 **** Synopsis:                                                           ****
 ****                                                                     ****
 **** Frees the memory used by _vector_                                   ****
 *****************************************************************************/

public void
bv_free(vector)

bit_vec *vector;

{
    free(vector -> bv);
    free((char *)vector);
}

/*****************************************************************************
 ****                                                                     ****
 **** int bv_equal(v1,v2)                                                 ****
 ****                                                                     ****
 **** bit_vec *v1;                                                        ****
 **** bit_vec *v2;                                                        ****
 ****                                                                     ****
 **** Synopsis:                                                           ****
 ****                                                                     ****
 **** Returns TRUE if each bit in v1 and v2 have the same value in the    ****
 **** same position. It returns FALSE otherwise.                          ****
 *****************************************************************************/

public int
bv_equal(v1,v2)

bit_vec *v1;
bit_vec *v2;

{
    bool_t eq;
    int i;

    eq = TRUE;

    if (v1 -> len != v2 -> len)
    {
        eq = FALSE;
    }
    else
    {
        for (i = 0; i < (v1 -> len + BYTE - 1) / BYTE; i++)
        {
            if (v1 -> bv[i] != v2 -> bv[i])
            {
                eq = FALSE;
                break;
            }
        }
    }

    return(eq);
}
