/*****************************************************************************
 ****                                                                     ****
 **** mosquito.c                                                          ****
 ****                                                                     ****
 **** Copyright (C) A. Dwelly and W.W. Armstrong, 1990.                   ****
 ****                                                                     ****
 **** All rights reserved.                                                ****
 ****                                                                     ****
 **** Attachment of copyright does not imply publication.                 ****
 **** This file contains information which is proprietary                 ****
 **** to the authors.                                                     ****
 ****                                                                     ****
 **** This is a test of the research version of the adaptive logic        ****
 **** network package atree. This particular program demonstrates the     ****
 **** algorithm's ability to extract a simple relationship from complex   ****
 **** data.                                                               ****
 ****                                                                     ****
 **** Each entry in the training set represents a human about which       ****
 **** 80 facts are known, represented by randomly setting bits. The       ****
 **** function _mosquito()_ is a single bit which is set if the patient   ****
 **** has been bitten by a mosquito (bit 1) is not taking quinine (bit 7) ****
 **** and does not have sickle cell anemia (bit 12). The programer is free****
 **** to increase the difficulty of the experiment by modifying the       ****
 **** initial constants.                                                  ****
 ****                                                                     ****
 **** As the program is set up, the algorithm is searching in a space     ****
 **** of 2^80 bits, that is, 1208925819614629174706176 seperate bit       ****
 **** patterns, from this, it sees a mere 500 (a fraction of 1% of the    ****
 **** possible patterns) but correctly deduces the relationship between   ****
 **** the known facts, and the health of the patients.                    ****
 ****                                                                     ****
 **** The adaptive logic network package based on work done by Prof. W.   ****
 **** W. Armstrong and others in the Department of Computing Science,     ****
 **** University of Alberta, and previous work at the Universite de       ****
 **** Montreal, and at AT&T Bell Laboratories, Holmdel, N. J.  The        ****
 **** software demonstrates that networks consisting of many layers of    ****
 **** linear threshold elements can indeed be effectively trained.        ****
 ****                                                                     ****
 **** License:                                                            ****
 **** A royalty-free license is granted for the use of this software for  ****
 **** NON_COMMERCIAL PURPOSES ONLY. The software may be copied and        ****
 **** modified provided this notice appears in its entirety and unchanged ****
 **** in all copies, whether changed or not.  Persons modifying the code  ****
 **** are requested to state the date, the changes made and who made them ****
 **** in the modification history.                                        ****
 ****                                                                     ****
 **** Warranty:                                                           ****
 **** No warranty of any kind is provided with this software.             ****
 **** This software is not supported, and is made available only under    ****
 **** the condition that the authors not be held responsible for damages  ****
 **** of any kind arising through use of the software, however caused.    ****
 ****                                                                     ****
 **** Patent:                                                             ****
 **** The use of a digital circuit which transmits a signal indicating    ****
 **** heuristic responsibility is protected by U. S. Patent 3,934,231     ****
 **** and others assigned to Dendronic Decisions Limited of Edmonton,     ****
 **** W. W. Armstrong, President.                                         ****
 ****                                                                     ****
 **** Modification history:                                               ****
 ****                                                                     ****
 **** 5.9.90 Initial implementation, A.Dwelly                             ****
 ****                                                                     ****
 *****************************************************************************/

#include <stdio.h>
#include "atree.h"

#define TRAIN_SIZE  500 
#define TEST_SIZE   500
#define WIDTH       80
#define LEAVES      256
#define EPOCHS      100
#define VERBOSITY   1

#define BITTEN 1
#define QUININE 7
#define ANEMIA 12

char mosquito(v)
char *v;
{
    return(v[BITTEN] && (!v[QUININE]) && (!v[ANEMIA]));
}

main()
{
    atree *tree;
    bit_vec training_set[TRAIN_SIZE];
    bit_vec result_set[TRAIN_SIZE];
    bit_vec *test;
    char vec[WIDTH];
    char unpacked_result[1];
    int correct = 0;
    int malaria = 0;
    int healthy = 0;
    int i;
    int j;

    /* Initialize */

    atree_init();

    /* Create the test data */

    printf("Creating training data\n");

    for (i = 0; i < TRAIN_SIZE; i++)
    {
        for (j = 0; j < WIDTH; j++)
        {
            vec[j] = RANDOM(2);
        }
        training_set[i] = *(bv_pack(vec, WIDTH));
        unpacked_result[0] = mosquito(vec);
        if (unpacked_result[0])
        {
            malaria++;
        }
        else
        {
            healthy++;
        }
        result_set[i] = *(bv_pack(unpacked_result, 1));
    }

    printf("There are %d patients with malaria, and %d healthy in the training set\n",malaria,healthy);

    /* Create a tree and train it */

    printf("Training tree\n");

    tree = atree_create(WIDTH, LEAVES);
    (void) atree_train(tree, training_set, result_set, 0,
                       TRAIN_SIZE, TRAIN_SIZE - 1, EPOCHS, VERBOSITY);

    /* Test the trained tree */

    printf("Testing the tree\n");

    for (i = 0; i < TEST_SIZE; i++)
    {
        for (j = 0; j < WIDTH; j++)
        {
            vec[j] = RANDOM(2);
        }
        test = bv_pack(vec, WIDTH);
        if (atree_eval(tree, test) == mosquito(vec))
        {
            correct++;
        }
        bv_free(test);
    }

    printf("%d correct out of %d in final test\n", correct, TEST_SIZE);

    exit(0);
}
