/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

/*--------------------------------------------------------------------------
  Author: Nigel Goddard
  Date: May 1 1987
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <ctype.h>
#ifdef BFLY
# include "bflysim.h"
#else
# include "uniproc.h"
#endif

extern char* NameToType();

static char  buffer[80];

/*------------------------load file format-------------------------

 save:
  search name table and write out count of sites, functions and types
  write out name table entries, remember count of sites, types and functions
        as writing site, function or type, insert count # in index field of nte
  write out units, ignoring name and using value in nte index field for
           site, functions and type names

 load:
  read in count of sites, functions and types and create string array
  read in name table entries, putting sites, functions and types in string
       array
  read in units, ignoring name and using index into string array for
         site, function and type names
  free string array
  for all name table entries: if unit name, copy pointer to unit array
      for all units covered

-------------------------------------------------------------------*/

static counterr(msg)
     char * msg;

{
  LOGfprintf(stderr,"error: didn't find count for %s\n",msg);
}

static termerr(msg)
     char * msg;

{
  LOGfprintf(stderr,"error: couldn't find $ terminator after %s\n",msg);
}

si_Cleanup()

{
  LOGfprintf(stderr,"fatal error, aborting now\n");
  abort();			/* should do something more intelligent */
}

/***** NetLoad *****/

/*---------------------------------------------------------------------------
  The function complimentary to  NetSave .  Reads in and
  checks the time stamp in the file  nfp  with  CheckStamp ,
  reconstructs the network from the file data, and then calls 
  RestoreState  to restore the  state  of the network.
----------------------------------------------------------------------------*/
NetLoad(nfp) 
    FILE *nfp;
{
    int c,ucount,strcnt,icount,t1,t2,i,no_strings,no_site,no_inputs;
    int version,no_units,alloc_units,scount,no_names;
    /*int no_sets,no_states;*/
    register Link * ip;
    register Site * sp;
    register Unit * up;
    NameDesc nte;
    int index, type, size, length;
    char **strings;
    char * name;
#ifdef TSIM
    int delay;
#endif

/*---------------------------check no existing network---------------*/

    if(NoUnits != 0)
      {
	LOGfprintf(stderr,"cannot load over existing network\n");
	return;
      }
    LOGfprintf(stdout,"loading units ");

/*---------------------------check time stamp------------------------*/

    CheckStamp(nfp,0);		/* check time stamp 0 means load file */

/*---------------------------check version number---------------------*/
    
    c = getc(nfp);
    while(isspace(c))
      c = getc(nfp);
    if(c != '#' || getc(nfp) != '#')
      {
	LOGfprintf(stderr,"network file format incorrect (no initial ##)\n");
	return;
      }
    fscanf(nfp,"%d",&version);	/* 3 is normal, 4 is with link delays */
#ifndef TSIM
    if(version != 3)
#else
    if (version != 4)
#endif
      {
	LOGfprintf(stderr,
		   "expecting different network file format; cannot load version %d\n",
		   version);
	return;
      }

/*---------------------------load in name table info-----------------*/

    if(fscanf(nfp,"%d", &no_strings) != 1)
      {
	counterr("string entries");
	return;
      }
    strings = (char **) si_malloc(no_strings * sizeof(char *));

    if(fscanf(nfp,"%d",&no_names) != 1)
      {
	counterr("name table entries");
	return;
      }
    for(i = 0,strcnt = 0;i < no_names;i++)
      {
	if(fscanf(nfp,"%s %d %d %d %d",
		  buffer,&type,&index,&size,&length) != 5)
	  {
	    LOGfprintf(stderr,"input error: name table entry no %d\n",i);
	    si_Cleanup();
	    return;
	  }
	if ((type == FUNC_SYM || type == DATA_SYM) &&
	    (name = (FindName(buffer,&nte)->name)) != NULL)
	  if (!(nte.type == STRING_SYM || nte.type == type))
	    {
	      LOGfprintf(stderr,"error: %s is a %s, cannot reload as a %s\n",
			 buffer,NameToType(buffer),
			 NameToType(AlterName(buffer,type,index,
					      size,length,&nte)));
	      si_Cleanup();
	      return;
	    }
	  else
	    ;			/* functions/variables already all set up ok */
	else
	  if (type == CODE_SYM && FindName(buffer,&nte) == NULL)
	    LOGfprintf(stderr,
		       "warning: codefile %s not loaded, probably fatal\n",
		       buffer);
	  else
	    if ((name =  EnterName(buffer,type,
				   index,size,length,&nte)) == NULL)
	      {
		LOGfprintf(stderr,
			   "error: name table entry for %s has changed.\n",
			   buffer);
		si_Cleanup();
		return;
	      }
	if (type == FUNC_SYM || type == TYPE_SYM || type == SITE_SYM)
	  {
	    if (strcnt >= no_strings)
	      {
		LOGfprintf(stderr,"error: more types, sites and functions than expected\n");
		si_Cleanup();
		return;
	      }
	    strings[strcnt] = (char *) si_malloc(strlen(name)+1);
	    strcpy(strings[strcnt++],name);
	  }
      }

    if(fscanf(nfp,"%s",buffer) != 1 || buffer[0] != '$')
      {
	termerr("name table entries");
	si_Cleanup();
	return;
      }

/*-------------------------load unit array------------------------------*/

    if(fscanf(nfp,"%d %d", &no_units, &alloc_units) != 2)
      {
        counterr("number of units");
	si_Cleanup();
	return;
      }

    AllocateUnits(alloc_units);
    for ( ucount = 0; ucount < no_units; ucount++)
      {
	if(ucount % 10 == 0)
	  {					  /* output . every 10 units */
	    putchar('.');
	    fflush(stdout);
	  }

/*----------------load unit type and function and site count---------------*/

	if(fscanf(nfp,"%d %d %d",&t1,&t2,&no_site) != 3)
	  {
	    LOGfprintf(stderr,
		       "error: type /function for unit %d\n", ucount);
	    si_Cleanup();
	    return;
	  }
	if(t1 >= 0 && t1 < no_strings && t2 >= 0 && t2 < no_strings)
#ifndef TSIM	  
	  up = UnitList + MakeUnit(strings[t1],NameToFunc(strings[t2]),
		   (FLINT) 0,(FLINT) 0,(FLINT) 0,(FLINT) 0,0,0);
#else
	  {
	    if (fscanf(nfp,"%d",&delay) != 1)
	      {
		LOGfprintf(stderr,
			   "error: delay buffer size for unit %d\n", ucount);
		si_Cleanup();
		return;
	      }
	    up = UnitList + MakeDelayUnit(strings[t1],NameToFunc(strings[t2]),
			  (FLINT) 0,(FLINT) 0,(FLINT) 0,(FLINT) 0,0,0,delay);
	  }
#endif
	else
	  {
            LOGfprintf(stderr,
		    "error: type (%d) / function (%d) string out of range for unit %d\n",
		    t1,t2,ucount);
	    si_Cleanup();
	    return;
	  }
	if (si_User_Unit_Load != NULL)
	  si_User_Unit_Load(nfp, up);

/*-------------------------load sites for unit-------------------------*/

	for ( scount = 0; scount < no_site; scount++)
	  {
	    if(fscanf(nfp,"%d %d %d",&t1,&t2, &no_inputs) != 3)
	      {
		LOGfprintf(stderr,"error: name / function for unit %d\n",
			   ucount);
		si_Cleanup();
		return;
	      }
	    if(t1 >= 0 && t1 < no_strings && t2 >= 0 && t2 < no_strings)
	      sp = AddSite(ucount,strings[t1],
			   NameToFunc(strings[t2]),(FLINT)0);
	    else
	      {
		LOGfprintf(stderr,
			   "error: name (%d) / function (%d) string out of range for site on unit %d\n",
			   t1,t2,ucount);
		si_Cleanup();
		return;
	      }
	    if (si_User_Site_Load != NULL)
	      si_User_Site_Load(nfp, up, sp);
	    
/*----------------------------load links for site----------------------*/

	    for ( icount = 0; icount < no_inputs; icount++ )
	      {
		if(fscanf(nfp,"%d %d",
			  & t1,
			  & t2) != 2)
		  {
		    LOGfprintf(stderr,"error: source / function name for link on site %s of unit %d\n", sp->name, ucount);
		    si_Cleanup();
		    return;
		  }
		NoUnits = no_units; /* HACK!! so no illegal unit error */
#ifndef TSIM
		ip = MakeLink(t2,ucount,sp->name,(FLINT) 0, (FLINT) 0,
			      NameToFunc(strings[t1]));
#else
		if (fscanf(nfp,"%d",&delay) != 1)
		  {
		    LOGfprintf(stderr,"error: delay not found for link from unit %d to site %s on unit %d\n", t2, sp->name, ucount);
		    si_Cleanup();
		    return;
		  }
		else
		  {
		    if (Outputs[t2] == NULL)
		      {
			Outputs[t2] = (Output *) si_calloc(1+delay,
							   sizeof(Output));
			*(Outputs[t2]) = delay;
		      }		/* make the buffer before the unit */
		    ip = MakeDelayLink(t2,ucount,sp->name,(FLINT) 0, (FLINT) 0,
				     NameToFunc(strings[t1]),delay);
		  }
#endif

		NoUnits = ucount+1; /* reset */
		if (si_User_Link_Load != NULL)
		  si_User_Link_Load(nfp, up, sp, ip);
	      }
	  }
	if(fscanf(nfp,"%s",buffer) != 1 || buffer[0] != '$')
	  {
	    LOGfprintf(stderr,"no '$' at end of unit %d",ucount);
	    si_Cleanup();
	    return;
	  }
      }

    if(fscanf(nfp,"%s",buffer) != 1 || buffer[0] != '$')
      {	
	LOGfprintf(stderr,"error: no $ at end of units\n");
	si_Cleanup();
	return;
      }
    printf(" units loaded.\n");
    for (i = 0; i < strcnt; i++)
      free(strings[i]);
    free((char *)strings);

/*--------------load in unit names and set and state arrays-------------*/

    if(fscanf(nfp,"%d %d", &NoSets, &StateCount) != 2)
      {
        counterr("number of sets and states");
	si_Cleanup();
	return;
      }
    if (!si_LoadUnitNames())
      {
	si_Cleanup();
	return;
      }
/*    if (no_sets != NoSets)
      {
	LOGfprintf(stderr,"error: wrong number of sets entered\n");
	goto clean;
      }
    if (no_states != StateCount)
      {
	LOGfprintf(stderr,"error: wrong number of states entered\n");
	goto clean;
      }*/
    if(fscanf(nfp,"%s",buffer) != 1 || buffer[0] != '$')
      {	
	LOGfprintf(stderr,"error: no $ after state and set count\n");
	si_Cleanup();
	return;
      }

/*-------------------------load network state data---------------------*/
    RestoreState(nfp);
    printf(" Done!\n");
    fflush(stdout);
    return;

  /*clean:
    si_Cleanup();*/

} /* NetLoad */

static int DI(name)
    char *name;
{
    NameDesc look;

    if(FindName(name,&look) == NULL)
      {
	fprintf(stderr,"NetSave(DI): cannot find symbol %s -- saved file will be no good\n",name);
	/*exit(1);*/
	return (-1);
      }
    return (look.length);	/* set when name table is dumped */
}

/***** NetSave  ******/
/*---------------------------------------------------------------------------
  Writes a time stamp into the file  savef  using  StampTime
  , followed by the  structure  of the network (that is enough
  information to be able to reconstruct all the units, sites and links,
  including unit names and types and all function pointers.  Finally
  calls  SaveState  to save the  state  of the network (i.e.
  the weights, outputs, potentials, etc).
----------------------------------------------------------------------------*/
/*-------------------------load file format----------------------------

 save:
  search name table and write out count of sites, functions and types
  write out name table entries, remember count of sites, types and functions
        as writing site, function or type, insert count # in index field of nte
  write out units, ignoring name and using value in nte index field for
           site, functions and type names

----------------------------------------------------------------------*/

NetSave(savef)
FILE	*savef;

{
    register int ucount;
    register Site *sp;
    register Link *ip;
    register Link *ip2;
    register Link *ip3;
    register Unit *up;

/*------------------issue Time Stamp-------------------------*/

    StampFile(savef,0);		/* timestamp file 0 means load file */

/*------------------issue version number----------------------*/

#ifdef TSIM
    fprintf(savef, "##4\n");
#else
    fprintf(savef, "##3\n");
#endif

/*--------------------print out number of sites, functions and types-----*/

    fprintf(savef, "%d\n",si_GetStringCount());

/*------------------dump name table---------------------------*/

    si_SaveNameTable(savef);
    fprintf(savef, "$\n");

/*------------------dump unit array---------------------------*/

    printf ("saving units");
    fprintf(savef, "%d %d\n", NoUnits, LastUnit);
    for(ucount = 0; ucount < NoUnits; ucount++)
      {
	if(ucount % 10 == 0)
	  {					  /* output . every 10 units */
	    putchar('.');
	    fflush(stdout);
	  }
	up = UnitList + ucount;
	fprintf(savef, "%d %d %hd", 
		DI(up->type),
		DI(FuncToName(up->unit_f)),
		up->no_site);
#ifdef TSIM
	fprintf(savef, " %d ", *(Outputs[ucount]));
#endif
	if (si_User_Unit_Save == NULL)
	  fprintf(savef, "\n");
	else
	  si_User_Unit_Save(savef, up);
    
/*------------------dump site info for a unit----------------*/
	
	for(sp = up->sites;sp != NULL;sp = sp->next)
	  {
	    fprintf(savef, "%d %d %hd",
		    DI(sp->name),
		    DI(FuncToName(sp->site_f)),
		    sp->no_inputs);
	    if (si_User_Site_Save == NULL)
	      fprintf(savef, "\n");
	    else
	      si_User_Site_Save(savef, up, sp);

/*------------------dump link info for a site----------------*/

	    ip = sp->inputs;			  /* current input */
	    if(ip == NULL) continue;              /* no links */
	    ip2 = ip->next;	                  /* next input */
	    ip3 = NULL;				  /* previous input */
	    Guard();
	    while (ip2 != NULL)			  /* reverse chain pointers */
	      {
		ip->next = ip3;			  /* current -> previous */
		ip3 = ip;			  /* new previous */
		ip = ip2;			  /* new current */
		ip2 = ip2->next;		  /* new next */
	      }
	    ip->next = ip3;                       /* ip is last link */

	  /* Now ip is a pointer to the last link in the chain, the
	     first link points to NULL.  Print links and reverse pointers */
	    ip2 = ip->next;			  /* next in chain */
	    ip3 = NULL;				  /* previous in chain */
	    while (ip2 != NULL)			  /* reverse chain pointers */
	      {					  /* and print each link */
		fprintf(savef, "%d %d",
			DI(FuncToName(ip->link_f)),
			ip->from_unit);
#ifdef TSIM
		fprintf(savef, " %d", ip->value - Outputs[ip->from_unit]);
#endif
		if (si_User_Link_Save == NULL)
		  fprintf(savef, "\n");
		else
		  si_User_Link_Save(savef, up, sp, ip);
		ip->next = ip3;
		ip3 = ip;
		ip = ip2;
		ip2 = ip2->next;
	      }
	    fprintf(savef, "%d %d",
		    DI(FuncToName(ip->link_f)),
		    ip->from_unit);
#ifdef TSIM
		fprintf(savef, " %d", ip->value - Outputs[ip->from_unit]);
#endif
	    if (si_User_Link_Save == NULL)
	      fprintf(savef, "\n");
	    else
	      si_User_Link_Save(savef, up, sp, ip);
	    ip->next = ip3;			  /* ip is first link */
	    if (ip != sp->inputs)
	      LOGfprintf(stderr,"Reversing pointers screwed up\n");
	    Release();
	  }
	fprintf(savef, "$\n");
      }
    
    fprintf(savef, "$\n");
    fprintf(savef,"%d %d\n$\n", NoSets,StateCount);

/*-----------------reset length field in name table entries-------*/

    si_ResetLengthField();

/*-----------------dump network state values----------------------*/

    SaveState(savef);		/* save network state */
}

