/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

/*--------------------------------------------------------------------------
  Author: Nigel Goddard
  Date: May 1 1987
----------------------------------------------------------------------------*/

#ifdef BSIM
#  include "bflysim.h"
#  define GUARD  
#  define RELEASE
#else
#  include "uniproc.h"
#  define GUARD Guard();
#  define RELEASE Release();
#endif

char * calloc();
char * malloc();

static CopyUnits();
static CopyOutputs();
static MapinOutputs();
static MarkFreeLinks();
static ChainUsedLinks();
static UnchainUsedLinks();
static RelocateFreeLinks();

#define CHUNK_SIZE 4096				  /* size of memory chunks */
#define DEFAULT_UNIT_BLOCK_SIZE 1000
#define SPACE_ALERT_SIZE 4096	/* saved in case memory full */

#define LAST_SITE ((CHUNK_SIZE / (sizeof(Site))) - 2) /* allocate in chunks */
static Site *free_sites;

#define LAST_LINK ((CHUNK_SIZE / (sizeof(Link))) - 2)	/* allocate in chunks */
static Link *free_links;

static Site * First_Site;		/* first in list of site arrays */
static Link * First_Link;		/* first in list of link arrays */

static UnitBlockSize = DEFAULT_UNIT_BLOCK_SIZE;	/* size of unit chunks */
static char * SpaceAlert = NULL;

/*---------------------------------------------------------------------------
  Cleans out all data structures, destroying entire network
----------------------------------------------------------------------------*/

TrashNetwork()

  {
    si_CleanNameTable(FALSE);
    si_clean_make();
    si_init_make();
    si_InitGlobals();
  }

/*---------------------------------------------------------------------------
  Allocates initial arrays for links and sites.  Space for sites and links
  is malloced in chunks of size CHUNK_SIZE bytes.  The last link or site
  in each chunk is used to point to the next chunk, not as a link or site.
  Then when trashing, the chunks form a linked list.
  The links or sites in a chunk are chained
  together (not including the last one) to form a free list.  Links or sites
  are allocated from the free list until it is empty, then a new chunk is
  malloced.  Deleted links or sites are returned to the front of the free list.
----------------------------------------------------------------------------*/
si_init_make()

  {
    register Site * new_site;
    register Link * new_link;
    register int i;

    free_sites = (Site *) si_malloc((LAST_SITE+1) * sizeof(Site));
    free_links = (Link *) si_malloc((LAST_LINK+1) * sizeof(Link));
    
    if (free_sites == (Site *) NULL || free_links == (Link *) NULL)
      {
        fprintf(stderr,"si_init_make: out of space\n");
        fflush(stderr);
        abort();
      }
    for (new_link = free_links, i = 0;
	 i < LAST_LINK-1;
	 new_link++,i++)
      new_link->next = new_link+1;	     /* chain the new links */
    new_link->next = NULL;		     /* null terminate chain*/
    for (new_site = free_sites, i = 0;
	 i < LAST_SITE-1;
	 new_site++,i++)
      new_site->next = new_site+1;           /* chain the new sites */
    new_site->next = NULL;		     /* null terminate chain*/
    First_Site = free_sites;			  /* first site block */
    First_Link = free_links;			  /* first link block */
    free_sites[LAST_SITE].next = NULL; /* last link/site in block points */
    free_links[LAST_LINK].next = NULL; /* to next link/site block and */
    free_links[LAST_LINK].value = NULL;/* points to previous link block */
    UnitBlockSize = DEFAULT_UNIT_BLOCK_SIZE;
    if ((SpaceAlert = malloc((unsigned) SPACE_ALERT_SIZE)) == NULL)
      LOGfprintf(stderr,"Something strange here, not enough space\n");
  }

/*---------------------------------------------------------------------------
  Destroys all units, links and sites, and output array.
----------------------------------------------------------------------------*/
    
si_clean_make()

  {
    register Site * sp;
    register Link * lp;

#ifdef TSIM
    register Unit * up;
    register int count;
    for (count = 0, up = UnitList; count < NoUnits; up++,count++)
	free((char *)Outputs[count]);	/* free up delay buffers */
#endif
    if (UnitList != NULL)
      cfree(UnitList);			/* unit vector */
    if (Outputs != NULL)
      free((char *)Outputs);			/* output vector */
    while (First_Site != NULL)
      {					/* sites */
        sp = First_Site;
        First_Site = (First_Site + LAST_SITE)->next;
	free((char *)sp);
      }
    while (First_Link != NULL)
      {					/* links */
        lp = First_Link;
        First_Link = (First_Link + LAST_LINK)->next;
	free((char *)lp);
      }
    if (SpaceAlert != NULL)
      free(SpaceAlert);
  }

/*---------------------------------------------------------------------------
  Mallocs space for the unit array and the output vector.  May be used to
  increase size of arrays.  Sets UnitBlockSize.
----------------------------------------------------------------------------*/

AllocateUnits(count)
    int count;
{
  Unit * NewList;
  Outvec * NewOut;
  
BFLYCATCH
  GUARD
  NewList = (Unit *) si_calloc((unsigned) count+LastUnit,sizeof(Unit));
  if (LastUnit > 0)		/* already some units */
    {
      CopyUnits(UnitList, NewList, NoUnits);
      free((char *)UnitList);
    }
  UnitList = NewList;
#ifndef BFLY
  NewOut = (Outvec *) si_calloc((unsigned) count+LastUnit,sizeof(Outvec));
  if (LastUnit > 0)		/* already some units */
    {
      CopyOutputs(Outputs, NewOut, NoUnits);
      free((char *)Outputs);
    }
  Outputs = NewOut;
#endif

#ifndef TSIM
  MapinOutputs();		/* reset all link value pointers */
#endif

  LastUnit += count;
  UnitBlockSize = count;
  RELEASE
BFLYTHROW("AllocateUnits",MySimNumber)
}

si_AutoAllocateUnits()

{
  LOGfprintf(stderr,"Auto-Allocating %d units...\n",UnitBlockSize);
  AllocateUnits(UnitBlockSize);
}

static CopyUnits(old, new, count)
     Unit * old, * new;
     int count;

{
  register Unit * up;
  register Unit * tp;
  register int cnt;
  register int tot;

  for (up = new, tp = old, cnt = 0, tot = count;
	   cnt < tot;
	   up++, tp++, cnt++)
	*up = *tp;		/* copy the unit */
}

static CopyOutputs(old, new, count)
     Outvec * old;
     Outvec * new;
     int count;

{
  register Outvec * up;
  register Outvec * tp;
  register int cnt;
  register int tot;

  for (up = new, tp = old, cnt = 0, tot = count;
	   cnt < tot;
	   up++, tp++, cnt++)
	(*up) = (*tp);		/* copy the output */

}

#ifndef TSIM
static MapinOutputs()

{
  register Unit * up;
  register Site * sp;
  register Link * lp;
  register int cnt;

  for (up = UnitList, cnt = 0;
       cnt < NoUnits;
       up++,cnt++)
    for (sp = up->sites; sp != NULL; sp = sp->next)
      for (lp = sp->inputs; lp != NULL; lp = lp->next)
	lp->value = &(Outputs[lp->from_unit]);
}
#endif

func_type NullFunc(){} /* used so a null function pointer
			  will never be called */

/*---------------------------------------------------------------------------
  Makes a unit and returns the index.  Consecutive indices on consecutive
  calls.   type is a pointer to a character string, and is simply used for
  display purposes.   func is a pointer to the function used to
  simulate the unit's action.   potential is the activation level
  for the unit.   data is a four byte value for the unit  data
  field described above.   output is the initial output of the
  unit.   state is a short integer representing the initial state
  value.   init-pot and  init-state are the values to set the
  unit potential and state when the network is  reset.  If Debug is
  non-zero, calls the debug version to check parameters valid.
----------------------------------------------------------------------------*/

int MakeUnit(type,func,ipot,pot,data,out,istate,state)
     char *type;
     func_ptr func;
     int istate,state;
     FLINT ipot, pot, out, data;

{
    register Unit *up;
    NameDesc nte;
    Index * typename;

BFLYCATCH
#ifndef BFLY
    if (Debug)			/* no debug on butterfly yet */
      return (si_debug_makeunit(type,func,ipot,pot,data,out,istate,state));
#endif
    GUARD
    if (LastUnit <= 0)
      si_AutoAllocateUnits();
    up = &UnitList[NoUnits++];
    typename = AlterName(type,TYPE_SYM,0,0,0,&nte);
#ifdef BFLY
    up->type = StoreString(type);
#else
    up->type = typename;
#endif
    if(func == NULL)
      up->unit_f = (func_ptr)NullFunc;
    else
      up->unit_f = func;
    up->init_potential = (pot_type) ipot;
    up->no_site = (short) 0;
    up->potential = (pot_type) pot;
    up->data = (data_type) data;
    up->output = (Output) out;
    up->init_state = (short) istate;
    up->state = (short) state;
    up->sites = NULL;
    up->sets = (unsigned int) 0;
    up->flags = (unsigned int) 0;
#ifdef TSIM
    if (Outputs[NoUnits-1] == NULL) /* "load" may have already allocated */
      *(Outputs[NoUnits-1] = (Output *) si_calloc(2 , sizeof (Output))) = 1;
				/* ordinairy link takes one time step */
    *(Outputs[NoUnits-1]+1) = (Output) out;
#else
    Outputs[NoUnits-1] = (Output) out;
#endif
#ifndef BFLY
    up->name = NULL;
#endif
    RELEASE
    return NoUnits - 1;
BFLYTHROW("MakeUnit",MySimNumber)
}

/*---------------------------------------------------------------------------
  Adds a site to a unit.
   index is the index of the unit to which the site is to be
  attached.   name is a pointer to a character string which will be
  the name of the site.   function is a pointer to the function to
  be called to simulate the action of the site.   data is the four
  byte value to be placed in the site  data field.  If Debug is non-zero
  calls the debug version to check parameters values are valid.  Returns
  a pointer to the site constructed.
----------------------------------------------------------------------------*/

Site * si_debug_addsite();

Site * AddSite(unit,name,func,data)
    int unit;
    FLINT data;
    char *name;
    func_ptr func;

{
    Site *new_site,*sp;
    NameDesc nte;
    Index * sitename;
    int i;

BFLYCATCH
#ifndef BFLY
    if (Debug)			/* no debug on butterfly yet */
      return (si_debug_addsite(unit,name,func,data));
#endif
    GUARD
   if (free_sites->next == NULL)	     /* no more sites on free chain */
     {
       (free_sites+1)->next = (Site *) si_malloc((LAST_SITE+1) * sizeof(Site));
       if (free_sites == (Site *) NULL)
	 {
	   fprintf(stderr,"MakeSite: out of space\n");
	   fflush(stderr);
	   abort();
	 }
       (((free_sites+1)->next)+LAST_SITE)->next = NULL;
					     /* terminate site block chain */
       free_sites->next = (free_sites+1)->next;
       for (new_site = free_sites->next, i = 0;
	    i < LAST_SITE-1;
	    new_site++,i++)
	 new_site->next = new_site+1;           /* chain the new sites */
       new_site->next = NULL;		     /* null terminate free chain*/
     }
   new_site = free_sites;
   free_sites = free_sites->next;
   sitename = AlterName(name,SITE_SYM,0,0,0,&nte);
#ifdef BFLY
    new_site->name = StoreString(name);
#else
    new_site->name = sitename;
#endif
    if(func == NULL)
      new_site->site_f = (func_ptr)NullFunc;
    else
      new_site->site_f = func;
    new_site->value = 0;
    new_site->no_inputs = (short) 0;
    new_site->data = (data_type)data;
    new_site->next = NULL;
    new_site->inputs = NULL;
#ifdef BFLY
    unit = OFFSET_UNIT(unit);	/* get local unit index */
#endif
    if (UnitList[unit].sites == NULL)
      UnitList[unit].sites = new_site;
    else
      {
	for (sp = UnitList[unit].sites; sp->next != NULL; sp = sp->next);
	sp->next = new_site;
      }
    ++(UnitList[unit].no_site);
    RELEASE
    return new_site;
BFLYTHROW("AddSite",MySimNumber)
}

/*---------------------------------------------------------------------------
  Makes a link between two units.
   source-unit is the index of the unit where the link originates.
   destination-unit is the index of the unit to which the link is
  going.   site-name is a pointer to a character string which is
  the name of the site on the destination unit at which the link is to
  arrive.   weight is the weight to put on the link, and should be
  within range of a short integer.  By convention
  weights are scaled down by a factor of 1000, thus a specified weight
  of 500 will be treated as a weight of 0.5.  This is to allow weights
  in the range 0 to 1 without having to use floating point arithmetic.
  Weights may be negative.  MakeLink returns a pointer to the link
  structure created.
----------------------------------------------------------------------------*/

Link * si_debug_makelink();

Link * MakeLink(from,to,site,weight,data,func)
     int from,to;
     FLINT weight, data;
     char *site;
     func_ptr func;

{
   Link *new_link, *tlink, *flink;
   Site *sp;
   char *sitename;
   NameDesc nte;
   int i;

BFLYCATCH
#ifndef BFLY
    if (Debug)			/* no debug on butterfly yet */
      return (si_debug_makelink(from,to,site,weight,data,func));
#endif
   GUARD
#ifdef BFLY
   sitename = FindString(site);
   for (sp = UnitList[OFFSET_UNIT(to)].sites;
#else
   sitename = (FindName(site,&nte))->name;	/* pointer to stored name */
   for (sp = UnitList[to].sites;
#endif
       sp != NULL && sp->name != sitename;
       sp = sp->next);
   if (free_links->next == NULL)	     /* no more links on free chain */
     {
       (free_links+1)->next = (Link *) si_malloc((LAST_LINK+1) * sizeof(Link));
       if (free_links == (Link *) NULL)
	 {
	   fprintf(stderr,"MakeLink: out of space\n");
	   fflush(stderr);
	   abort();
	 }
       (((free_links+1)->next)+LAST_LINK)->next = NULL;	/* terminate chain */
       (((free_links+1)->next)+LAST_LINK)->value = (Output *) (free_links + 1);
       free_links->next = (free_links+1)->next;
       for (new_link = free_links->next, i = 0;
	    i < LAST_LINK-1;
	    new_link++,i++)
	 new_link->next = new_link+1;	     /* chain the new links */
       new_link->next = NULL;		     /* null terminate free chain*/
     }
   new_link = free_links;
   free_links = free_links->next;
   new_link->from_unit = from;
   new_link->weight = weight;
   new_link->data = data;
#ifdef BFLY
   new_link->value = &(OutputArrays[SIM_UNIT(from)][OFFSET_UNIT(from)]);
#else
#ifndef TSIM
     new_link->value = &(Outputs[from]);
#else
   new_link->value = Outputs[from] + 1;
#endif
#endif
   if(func == NULL)
     new_link->link_f = (func_ptr)NullFunc;
   else
     new_link->link_f = func;
   if (sp->inputs == NULL)			  /* first link */
     {
       sp->inputs = new_link;
       new_link->next = NULL;		  /* initialize */
     }
   else
     {
       for (tlink = flink = sp->inputs;	  /* find place in link chain*/
	    tlink->next != NULL &&		  /* or end of chain */
	    from < tlink->from_unit;	  /* ordered by from_unit */
	    tlink = tlink->next)		  /* highest first */
	 flink = tlink;			  /* save current pointer */
       if (from < tlink->from_unit)	  /* so tlink->next == NULL */
	 {					  /* must be at end of list */
	   new_link->next = NULL;		  /* no links further */
	   tlink->next = new_link;		  /* new link now last */
	 }
       else
	 {					  /* before end of list */
	   if (tlink == sp->inputs)	  /* at beginning of list? */
	     sp->inputs = new_link;
	   else
	     flink->next = new_link;	  /* in middle of list */
	   new_link->next = tlink;	          /* reconnect rest of list */
	 }
     }
   sp->no_inputs++;
   NoLinks++;
   RELEASE
	return new_link;
BFLYTHROW("MakeLink",MySimNumber)
}
   
#ifdef TSIM

int MakeDelayUnit(type,func,ipot,pot,data,out,istate,state,delay)
     char *type;
     func_ptr func;
     int istate,state, delay;
     FLINT ipot, pot, out, data;

{
  return (ChangeDelayBuffer( MakeUnit( type,func,ipot,pot,data,out,
					istate,state ),
			      delay )
	  );
}

char * si_realloc();

int ChangeDelayBuffer(from,delay)
  int from, delay;

{
  register Unit * up;
  register Site * sp;
  register Link * lp;
  register int count;
  Output * buf, * tbuf;
  int pdelay;

  if (delay == *Outputs[from])
    return from;
  if (delay > 255)
    {
      LOGfprintf(stderr,"maximum delay of %d for links from unit %d reset to 255\n",delay,from);
      delay = 255;
    }
  if (delay < 1)
    {
      LOGfprintf(stderr,"maximum delay of %d for links from unit %d reset to 1\n",delay,from);
      delay = 1;
    }
  
  buf = (Output *) si_realloc((tbuf = Outputs[from]),
			      (unsigned) ((delay+1)*sizeof(Output)) );
  if (*tbuf < delay)		/* is buffer getting bigger? */
    for (count = delay; count > *tbuf; count--)
      *(buf+count) = (Output) 0; /* zero new piece of buffer */

  if (buf != tbuf)
    {				/* buffer was relocated, copy and remap */
      Outputs[from] = buf;
      for (count = (*tbuf < delay) ? *tbuf : delay; count > 0; count--)
	*(buf + count) = *(tbuf + count); /* copy retained buffer */
      for (count = 0, up = UnitList; count < NoUnits; count++,up++)
	for (sp = up->sites; sp != NULL; sp = sp->next)
	  for (lp = sp->inputs; lp != NULL; lp = lp->next)
	    if (lp->from_unit == from)
	      lp->value += buf - tbuf; /* remap to new buffer */
      free((char *)tbuf);
    }
  *Outputs[from] = delay;	/* first element holds max delay */
  return from;
}

Link * MakeDelayLink(from,to,site,weight,data,func,delay)
     int from,to,delay;
     FLINT weight, data;
     char *site;
     func_ptr func;

{
  Link * tlp;

  if (delay < 1 || delay > 255)
    {
      LOGfprintf(stderr,"delay of %d from unit %d to site %s on unit %d reset to 255\n",delay,from,site,to);
      delay = 255;
    }
  tlp = MakeLink(from,to,site,weight,data,func);
  if (*(Outputs[from]) < delay) 
    ChangeDelayBuffer(from,delay);
  tlp->value = Outputs[from]+delay;
}

static Link * FindLink(from,to,site)
     int from, to;
     char * site;

{
  register Site * sp;
  register Link * lp;

  if (!(LegalUnit(from)) || !(LegalUnit(to)))
    return(NULL);			/* invalid unit index */
  for (sp = UnitList[to].sites;
       sp != NULL && strcmp(sp->name,site); 
       sp = sp->next);
  if (sp == NULL)
    return(NULL);			/* no such site name */
  for (lp = sp->inputs; lp != NULL && lp->from_unit != from; lp = lp->next);
  return lp;
}

Link * SetDelay(from,to,site,delay)
     int from, to, delay;
     char * site;

{
  Link * tlp;
  register Unit * up;
  register Site * sp;
  register Link * lp;
  register int count;
  Output * buf, * tbuf;

  if (delay < 1 || delay > 255)
    {
      LOGfprintf(stderr,"delay of %d from unit %d to site %s on unit %d reset to 255\n",delay,from,site,to);
      delay = 255;
    }
  if ((tlp = FindLink(from,to,site)) == NULL)
    return(NULL);			/* no such link */ 
  ChangeDelayBuffer(from,delay); /* make sure buffer big enough */
  tlp->value = Outputs[from] + delay;
  return (tlp);
}

int GetDelay(from,to,site)
     int from, to;
     char * site;

{
  register Link * lp;

  if ((lp = FindLink(from,to,site)) == NULL)
    return(-1);			/* no such link */
  return(lp->value - Outputs[lp->from_unit]);
}
#endif
      
/*---------------------------------------------------------------------------
  This function deletes on one or more sites.  The first four
  parameters specify the units to which the sites are attached.
  <sitename> is the name of the site, or optionally ALL, meaning
  any site on a destination unit.  Deleted sites are returned to
  the free list.  All links arriving at the site(s) are deleted.

  The units are in the range  ulow  to
  uhigh .  If  uset  is FALSE, all the units in these two
  ranges are considered.  If  uset  is TRUE, then only those units
  in the range that are in the set whose number is  usind are
  considered.
----------------------------------------------------------------------------*/

Site * DeleteSite(index,sitename)
   int index;					  /* delete one link */
   char * sitename;
   

{
  Link * lp, * tlp;
  Site ** spp;
  Site * sp;
  Unit * up;
  func_ptr func;
  
  func = NameToFunc("User_Link_Delete");	  /* ptr to user delete func */
  for (spp = &((up = (UnitList+index))->sites);
       (*spp) != NULL && strcmp((*spp)->name,sitename);
       spp = &((*spp)->next));   
  if ((*spp) != NULL)
    {
      for (lp = (*spp)->inputs; lp != NULL;)
	{					  /* delete incoming links */
	  if (func != NULL)
	    func(up,(*spp),lp);
	  tlp = lp->next;
	  lp->next = free_links;
	  free_links = lp;
	  lp = tlp;
	  NoLinks--;
	}
      sp = (*spp)->next;			  /* ptr to rest of chain */
      (*spp)->next = free_sites;		  /* add to free list */
      free_sites = (*spp);			  /* reset head of free list */
      (*spp) = sp;				  /* reconnect site chain */
      return free_sites;			  /* return deleted site */
    }
  else
    return NULL;
}

DeleteSites(ulow,uhigh,uset,usetind,sitename)
     int ulow,uhigh,uset,usetind;
     char * sitename;				  /* delete many links */

{
  int ut;
  register Link * lp;
  register Link * tlp;
  Site * sp, *tsp, ** spp;
  Unit * up;
  register func_ptr linkfunc;
  func_ptr sitefunc;
 
  linkfunc = NameToFunc("User_Link_Delete");	  /* ptr to user delete func */
  sitefunc = NameToFunc("User_Site_Delete");	  /* ptr to site delete func */
  if (strcmp(sitename,"all") && strcmp(sitename,"ALL"))
    for (ut = ulow,up = UnitList+ulow; ut <= uhigh; ut++,up++)
      {
	if (!uset || (up->sets >> usetind) & 1)
	  {
	    for (spp = &(up->sites);
		 (*spp) != NULL && strcmp((*spp)->name,sitename);
		 spp = &((*spp)->next)); 
	    if ((*spp) != NULL)
	      {
		for (lp = (*spp)->inputs; lp != NULL;)
		  {				  /* delete incoming links */
		    if (linkfunc != NULL)
		      linkfunc(up,(*spp),lp);
		    tlp = lp->next;
		    lp->next = free_links;
		    free_links = lp;
		    lp = tlp;
		    NoLinks--;
		  }
		if (sitefunc != NULL)
		  sitefunc(up,(*spp));
		sp = (*spp)->next;		  /* ptr to rest of chain */
		(*spp)->next = free_sites;	  /* add to free list */
		free_sites = (*spp);		  /* reset head of free list */
		(*spp) = sp;			  /* reconnect site chain */
		up->no_site--;
	      }
	  }
      }
  else
    for (ut = ulow,up = UnitList+ulow; ut <= uhigh; ut++,up++)
      if (!uset || (up->sets >> usetind) & 1)
	{
	  for (sp = up->sites; sp != NULL;)
	    {
	      for (lp = sp->inputs; lp != NULL;)
		{				  /* delete incoming links */
		  if (linkfunc != NULL)
		    linkfunc(up,sp,lp);
		  tlp = lp->next;
		  lp->next = free_links;
		  free_links = lp;
		  lp = tlp;
		  NoLinks--;
		}
	      if (sitefunc != NULL)
		sitefunc(up,sp);
	      tsp = sp->next;
	      sp->next = free_sites;
	      free_sites = sp;
	      sp = tsp;
	    }
	  up->sites = NULL;
	  up->no_site = 0;
	}
}

/*---------------------------------------------------------------------------
  These functions delete on one or more links.  The first four
  parameters specify the units from which the links originate.  The
  second four parameters specify the destination units.   sitename
  is the name of the site at which the links arrive, or optionally
  ALL, meaning any site on a destination unit.  Deleted links are
  returned to the free list

  The source and destination units are in the range  ul(ow)  to
  uh(igh) .  If  uset  is FALSE, all the units in these two
  ranges are considered.  If  us(et)  is TRUE, then only those units
  in the range that are in the set whose number is  usind  (for
  source units) or  usetind  (for destination units) are
  considered.
----------------------------------------------------------------------------*/

Link * DeleteLink(from,to,sitename)
   int from, to;				  /* delete one link */
   char * sitename;
   

{
  register Link ** lpp;
  register Link * tlp;
  Site * sp;
  
  for (sp = UnitList[to].sites;
       sp != NULL && strcmp(sp->name,sitename);
       sp = sp->next);   
  if (sp != NULL)
    for (lpp = &(sp->inputs); (*lpp) != NULL; )
      if ((*lpp)->from_unit == from)
	{
	  tlp = (*lpp)->next;		     /* ptr to rest of chain */
	  (*lpp)->next = free_links;
	  free_links = (*lpp);
	  (*lpp) = tlp;
	  NoLinks--;
	  sp->no_inputs--;
	  return free_links;
	}
      else
	lpp = &((*lpp)->next);
  return NULL;
}

DeleteLinks(ul,uh,us,usind,ulow,uhigh,uset,usetind,sitename)
     int ul,uh,us,usind,ulow,uhigh,uset,usetind;
     char * sitename;				  /* delete many links */

{
  int ut;
  register Link ** lpp;
  register Link * tlp;
  Site * sp;
  Unit * up;
  register func_ptr func;
  
  func = NameToFunc("User_Link_Delete");	  /* ptr to user delete func */
  if (strcmp(sitename,"all") && strcmp(sitename,"ALL"))
    for (ut = ulow,up = UnitList+ulow; ut <= uhigh; ut++,up++)
      {
	if (!uset || (up->sets >> usetind) & 1)
	  {
	    for (sp = up->sites;
		 sp != NULL && strcmp(sp->name,sitename);
		 sp = sp->next);
	    if (sp != NULL)
	      for (lpp = &(sp->inputs); (*lpp) != NULL; )
		if ((*lpp)->from_unit >= ul && (*lpp)->from_unit <= uh &&
		    (!us || (UnitList[(*lpp)->from_unit].sets >> usind) & 1))
		  {
		    if (func != NULL)             /* if user structures */
		      func(up,sp,*lpp);		  /* delete user structures */
		    tlp = (*lpp)->next;		  /* ptr to rest of chain */
		    (*lpp)->next = free_links;
		    free_links = (*lpp);
		    (*lpp) = tlp;
		    sp->no_inputs--;
		    NoLinks--;
		  }
		else
		  lpp = &((*lpp)->next);
	  }
      }
  else
    for (ut = ulow,up = UnitList+ulow; ut <= uhigh; ut++,up++)
      if (!uset || (up->sets >> usetind) & 1)
	for (sp = up->sites; sp != NULL; sp = sp->next)
	  for (lpp = &(sp->inputs); (*lpp) != NULL; )
	    if ((*lpp)->from_unit >= ul && (*lpp)->from_unit <= uh &&
		(!us || (UnitList[(*lpp)->from_unit].sets >> usind) & 1))
	      {
		if (func != NULL)		  /* if user structures */
		  func(up,sp,*lpp);		  /* delete user structures */
		tlp = (*lpp)->next;		  /* ptr to rest of chain */
		(*lpp)->next = free_links;
		free_links = (*lpp);
		(*lpp) = tlp;
		sp->no_inputs--;
		NoLinks--;
	      }
	    else
	      lpp = &((*lpp)->next);
}

/*------------------------------------------------------------------------
  Scavenging links works as follows.  First all links on the free list are
  marked by setting their <value> field to NULL.  Then the links attached to
  sites, which are connected in linked lists, are linked in the reverse
  direction, so now forming doubly linked lists.  The <value> field of each
  is used to point to the previous link.  The first link (the one pointed
  to by the <inputs> field of the site) has its <value> field set to point
  the site, and this link is further marked by negating the value in its
  <from_unit> field.  For the propagation delay version, this is slightly
  modified:  in addition to doubly linking the lists using the <value> field,
  the high bit of the <from_unit> field is set for the first link in each
  list (that pointed to by the site), and the next highest 8 bits (23 - 30)
  are used to recored the delay on the link.  This means that the maximum
  delay is 2**8 or 256 steps, and that there can be no more than 2**23 or
  approximately 8 million units in any simulation using the propagation
  delay facility.

  To scavenge the links, we use two pointers <freelp> and <usedlp>.  The
  first one always points to an unused link.  It starts at the beginning of
  the first link block (pointed to by First_Link) and moves through each
  block in turn.  The link blocks form a doubly linked list, using the <next>
  and <value> fields of the last link in each block, which is never used as
  a real link.  <usedlp> starts at the last link in the last link block and
  works backwards towards the first link in the first link block.  It always
  points to a link which is in use (value pointer not NULL).  <usedlp> is
  copied into <freelp>, the pointers in the doubly linked list of links which
  held <usedlp> updated, and <freelp> and <usedlp> moved to the next free
  and used link respectively.  When they pass each other the used links have
  all been moved to the blocks at the beginning of the block chain.  The
  empty blocks are then freed, and the <value> and <from_unit> fields of
  the used links reset to their original values.  Finally the remaining free
  links are chained and form the free list.

  Or something like that.
----------------------------------------------------------------------------*/

ScavengeLinks()

{
  if (MarkFreeLinks() > LAST_LINK+1)
    {
      ChainUsedLinks();
      RelocateFreeLinks();
      UnchainUsedLinks();
    }
}

static MarkFreeLinks()

{
  Link * lp;
  int i;
  
  for (lp = free_links, i = 0; lp != NULL; lp = lp->next,i++)
    lp->value = NULL;
  LOGfprintf(stderr,"scavenging %d links on free list...\n",i);
  return i;
}

static ChainUsedLinks()

{
  Unit * up;
  register Site * sp;
  register Link * lp;
  int i;

  for (up = UnitList, i = 0; i < NoUnits; up++,i++)
    for (sp = up->sites; sp != NULL; sp = sp->next)
      if (sp->inputs != NULL)
	{
#ifdef TSIM
	register unsigned tmp;
	  tmp = (sp->inputs->from_unit & 0x007fffff) |
	        (((sp->inputs->value - Outputs[sp->inputs->from_unit]) &
		  0xff) << 23) | 0x80000000;
	  sp->inputs->from_unit = tmp;
	  sp->inputs->value = (Output *) sp;
	  for (lp = sp->inputs; lp->next != NULL; lp = lp->next)
	    {
	      lp->next->from_unit = (lp->next->from_unit & 0x007fffff) |
	        (((lp->next->value - Outputs[lp->next->from_unit]) &
		  0xff) << 23);
	      lp->next->value = (Output *) lp;
	    }
#else
	  sp->inputs->from_unit = 0 - sp->inputs->from_unit;
	  sp->inputs->value = (Output *) sp;
	  for (lp = sp->inputs; lp->next != NULL; lp = lp->next)
	    lp->next->value = (Output *) lp;
#endif
	}
}

static UnchainUsedLinks()

{
  Unit * up;
  register Site * sp;
  register Link * lp;
  int i;
  
  for (up = UnitList, i = 0; i < NoUnits; up++,i++)
    for (sp = up->sites; sp != NULL; sp = sp->next)
      if (sp->inputs != NULL)
	{
#ifdef TSIM
	register unsigned tmp;
/*	  tmp = sp->inputs->from_unit ;
	  sp->inputs->from_unit = tmp & 0x007fffff;
	  sp->inputs->value = Outputs[sp->inputs->from_unit]+
			       ((tmp >> 23) & 0xff);
*/
	  for (lp = sp->inputs; lp != NULL; lp = lp->next)
	    {
	      tmp = lp->from_unit ;
	      lp->from_unit = tmp & 0x007fffff;
	      lp->value = Outputs[lp->from_unit] +
		          ((tmp >> 23) & 0xff);
	    }
#else
	  sp->inputs->from_unit = 0 - sp->inputs->from_unit;
	  for (lp = sp->inputs; lp != NULL; lp = lp->next)
	    lp->value = &(Outputs[lp->from_unit]);
#endif
	}
}

static RelocateFreeLinks()

{
  Link * usedlp, * freelp;
  Link * usedbp, * freebp;
  int usedcnt, freecnt;

  usedcnt = freecnt = 0;
  freebp = freelp = First_Link;
  for (usedbp = freebp+LAST_LINK;
       usedbp->next != NULL;
       usedbp = usedbp->next+LAST_LINK);     /* find last link in last block */
  usedlp = usedbp-1;			     /* last link used as link */
  do
    {
      while (freelp->value != NULL)
	{			/* look for a free link */
	  while (freecnt < LAST_LINK && freelp->value != NULL)
	    {			/* look to end of block */
	      freecnt++;
	      freelp++;
	    }
	  if (freecnt == LAST_LINK)
	    {			/* reached end of block */
	      freecnt = 0;		     /* so start on next block */
	      freebp = freelp = (freebp+LAST_LINK)->next;
	    }
	}
      while (usedlp->value == NULL)
	{			/* look for a used link */
	  while (usedcnt < LAST_LINK && usedlp->value == NULL)
	    {			/* look to end of block */
	      usedcnt++;
	      usedlp--;
	    }
	  if (usedcnt == LAST_LINK)
	    {			/* reached end of block */
	      usedcnt = 0;
	      usedbp = (Link *) usedbp->value; /* previous block */
	      usedlp = usedbp-1; /* last link in block */
	    }
	}
      if ((usedbp->next != freebp && usedbp != freebp+LAST_LINK) ||
	  (freebp+LAST_LINK == usedbp && freelp < usedlp))
	{
	  *freelp = *usedlp;	/* copy used into free */
	  if (freelp->from_unit & 0x80000000)/*used was first link in site */
	    ((Site *) (freelp->value))->inputs = freelp;
	  else			/* reset forward link chain */
	    ((Link *) (freelp->value))->next = freelp;
	  if (freelp->next != NULL) /* reset backward link chain */
	    freelp->next->value = (Output *) freelp;
	  usedlp->value = NULL; /* make into unused */
	}
    }
  while ((usedbp->next != freebp && usedbp != freebp+LAST_LINK) ||
	 (freebp+LAST_LINK == usedbp && freelp < usedlp));

  for (usedbp = (freebp+LAST_LINK)->next;
       usedbp != NULL;		/* free up unused blocks */
       usedlp = usedbp, usedbp = (usedbp+LAST_LINK)->next, free((char*)usedlp));
  for (free_links = freelp; freecnt < LAST_LINK-1; freecnt++, freelp++)
    freelp->next = freelp+1;	/* restore free chain */
  freelp->next = NULL;		/* terminate free chain */
  (freelp+1)->next = NULL;	/* terminate link block chain */
}

#ifdef TSIM
/*---------------------------------------------------------------------------
  Buffer scavenging works like this.  First we go through all links and in
  each buffer mark the maximum delay needed in the first buffer location.
  Then for each buffer that is longer than needed, we reallocate the buffer.
  Finally we reset the first buffer location to the correct value, i.e. the
  value in the unit <output> field.
----------------------------------------------------------------------------*/

ScavengeBuffers()

{
  MarkMaxNeeded();
  ReallocateBuffs();
}

static int MarkMaxNeeded()

{
  Unit * up;
  Site * sp;
  register Link * lp;
  int ucnt;
  Output ** opv;
  Output * op;

  for (ucnt = 0, opv = Outputs; ucnt < NoUnits; ucnt++,opv++)
    *((*opv)+1) = 1;		/* in case no links, minimum is size 1 */
  for (ucnt = 0, up = UnitList; ucnt < NoUnits; ucnt++,up++)
    for (sp = up->sites; sp != NULL; sp = sp->next)
      for (lp = sp->inputs; lp != NULL; lp = lp->next)
	{
	  op = Outputs[lp->from_unit];
	  if ((lp->value - op) > *(op+1))
	    *(op+1) = lp->value - op; /* save maximum delay in first pos */
	}
}

static int ReallocateBuffs()

{
  Unit * up, * tup;
  Site * tsp;
  int ucnt;
  register Output ** opv;
  register Output * op;
  register int count;
  register Link * rlp;

  for (ucnt = 0, opv = Outputs, op = *opv, up = UnitList;
       ucnt < NoUnits;
       ucnt++,opv++,op = *opv,up++)
    {				/* opv points to pointer to buffer */
      if (*op < *(op+1))
	LOGfprintf(stderr,
		   "simulator: incorrect buffer size for link delay!\n");
      if (*op > *(op+1))

	{
	  *opv = (Output *) si_realloc( op, /* reallocate the buffer */
				      (unsigned) ((1+ *(op+1))*sizeof(Output)) );
	  if (*opv != op)
	    {			/* buffer was relocated, copy and remap */
	      for (count = *(op+1); count > 0; count--)
		*((*opv) + count) = *(op + count); /* copy retained buffer */
	      for (count = 0, tup = UnitList; count < NoUnits; count++,tup++)
		for (tsp = tup->sites; tsp != NULL; tsp = tsp->next)
		  for (rlp = tsp->inputs; rlp != NULL; rlp = rlp->next)
		    if (rlp->from_unit == ucnt)
		      rlp->value += (*opv) - op; /* remap to new buffer */
	      free((char *) op);
	      op = *opv;
	    }
	  *op = *(op+1);	/* buffer was not relocated, reset size */
	}
      *(op+1) = up->output;	/* reset to correct value */
    }
}
#endif

static space_alert(cnt)
     unsigned int cnt;

{
  free(SpaceAlert);	/* need space to print and read */
  SpaceAlert = NULL;
  LOGfprintf(stderr,"Out of space, scavenging has been done, I need %d bytes.  There is a little\nspace for you to work in.  Type quit when you have created enough space (this\nmessage will reappear until enough is free),or control_D to exit to UNIX.\n",cnt);
  debug_command_reader("quit to continue, control_D to exit");
  if ((SpaceAlert = malloc((unsigned) SPACE_ALERT_SIZE)) == NULL)
    {
      LOGfprintf(stderr,"Ooops, now even less space...\n");
      if ((SpaceAlert = malloc((unsigned) SPACE_ALERT_SIZE/4)) == NULL)
        {
	  LOGfprintf(stderr,"Too bad, dumping core...\n");
	  abort();
	}
    }
}

char * si_malloc(siz)
     unsigned int siz;

{
  char * t;
  
  if ((t = malloc(siz)) == NULL)
    {
      ScavengeLinks();
      while ((t = malloc(siz)) == NULL)
	space_alert(siz);
    }
  return t;
}

char * si_calloc(cnt,siz)
     unsigned int cnt,siz;

{
  char * t;
  
  if ((t = calloc(cnt,siz)) == NULL)
    {
      ScavengeLinks();
      while ((t = calloc(cnt,siz)) == NULL)
	space_alert(cnt*siz);
    }
  return t;
}

char * realloc();

char * si_realloc(ptr,siz)
     char * ptr;
     unsigned int siz;

{
  char * t;
  
  if ((t = realloc(ptr,siz)) == NULL)
    {
      ScavengeLinks();
      while ((t = realloc(ptr,siz)) == NULL)
	space_alert(siz);
    }
  return t;
}

