/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

/*--------------------------------------------------------------------------
  Author: Nigel Goddard
  Date: May 1 1987
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <ctype.h>
#include <sys/time.h>
#include "uniproc.h"

char * getenv();
char * strcat();

static   char currentpath[1024];		  /* current path, incl file */
/*---------------------------------------------------------------------------
  Returns the full file name, including path.  searchpath is the environment
  variable $path, name is the file name.
----------------------------------------------------------------------------*/

static char * search (searchpath,name)

     char *searchpath;
     char *name;

{       
  char *searchptr;				  /* ptr to searchpath   */
  char *currentptr = currentpath;		  /*  */
  int loc;					  /* for building currentpath*/
  struct stat buf;				  /* for calls to stat */
  
  /* build currentpath     */
  /* look for file, and return */
  
  for (searchptr = searchpath ; *searchptr != NULL ; ++searchptr)
    {   
      loc = 0;                        /* init currentpath and build*/
      /* and build from next path */
      while ((*searchptr != ':') && (*searchptr != NULL))
	currentpath[loc++] = *searchptr++;
      currentpath[loc] = NULL;          /* end of currentpath */
      if (currentpath[0] != NULL)       /* non-null path, so  */
	currentptr = strcat(currentpath,"/"); /* add slash  */
      currentptr = strcat(currentpath,name); /* add name      */
      if ( ! stat(currentptr,&buf))     /* check for file     */
	return(currentptr); /* print name */
      if (*searchptr == NULL) --searchptr; /* end of searchpath  */
      }
  return (name);
}

/*---------------------------------------------------------------------------
  Returns the name of the program that is currently running.
----------------------------------------------------------------------------*/

static char * GetImageName()

{
   int i;
   char * searchpath, *fname;
   char * pathname = "PATH\0\0";

   i = 0;
   while (isspace(si_Program[i])) i++;
   if (si_Program[i] =='/')	/* full path name */
     fname = si_Program;
   else				/* in current dir or on path */
     if ((searchpath = getenv(pathname)) == NULL) /* path defined? */
       fname = si_Program;		/* if not, in current dir */
     else
       fname = search(searchpath,si_Program); /* search path for file */
   return fname;
 }

/*---------------------------------------------------------------------------
  Writes a time, process and image name stamp to the file  nfp .
  If  type  is TRUE (i.e. a  checkpoint  file) the process ID number
  is written first, otherwise (i.e. a  save  file) the process ID
  number is not written.  Then the function writes out the name of the
  program that is running (i.e. the simulator executable) and the time
  it was made in human-readable form.  Next it writes out the current
  time in human-readable form.  Finally it writes out the current system
  time in seconds.  Returns FALSE on failure, TRUE on success.  For example:

        Processid = 882
        Image = sim written Wed Apr 29 15:03:58 1987
        Current Time = Thu Apr 30 23:32:58 1987
	546838378
----------------------------------------------------------------------------*/

StampFile(nfp,type)
     FILE *nfp;
     int type;

{
  struct stat filestat;
  char * imagename;
  struct timeval tp;
  struct timezone tzp;

  imagename = GetImageName();
  if (stat(imagename,&filestat) != 0) /* get image file statistics */
    {
      perror("simulator - can't get program file statistics");
      return FALSE;
    }
  if (type == 1)		/* checkpoint file */  
    fprintf(nfp,"Processid = %ld\n", getpid());

  fprintf(nfp,"Image = %s written %s", /* print image file name and date */
	  imagename,ctime(&filestat.st_mtime));
  if (gettimeofday(&tp,&tzp) == -1)
    {
      perror("simulator - can't get time of day");
      return FALSE;
    }
				/* print current time in ASCII and seconds */
  fprintf(nfp,"Current Time = %s%ld\n",ctime(&tp.tv_sec),tp.tv_sec);
  return TRUE;
}

/*---------------------------------------------------------------------------
  Reads in and checks a stamp made by  StampFile .  If  type
  is TRUE, it expects a process ID number, if FALSE it expects no
  process ID number.  The function will return FALSE if the check fails
  because the stamp format is incorrect or missing.  It will issue
  warnings if the file is over a week old, if the stamped image name is
  different to the current program file name (i.e. the simulator has
  been recreated), or  type  is TRUE, meaning a  checkpoint 
  file and the stamped process ID number is different to the current
  process ID number.  Returns TRUE on sucess.
----------------------------------------------------------------------------*/

CheckStamp(nfp,type)
     FILE *nfp;
     int type;

{
  struct stat filestat;
  char * imagename, *imagetime, *errstr;
  struct timeval tp;
  struct timezone tzp;
  long pid, its;
  char str[80], str2[80], timestamp[80], checktime[80], fileimagename[1024];

  imagename = GetImageName();	/* get image file statistics */
  if (stat(imagename,&filestat) != 0)
    {
      perror("simulator - can't get program file statistics");
      return FALSE;
    }
  imagetime = ctime(&filestat.st_mtime); /* get image time in seconds */
  
  if (type == 1)		/* checkpoint file */
    {
      errstr = "restore";
      if (fscanf(nfp,"%s %1s %ld", str,str2,&pid) != 3)
	{
	  fprintf(stderr,"%s error: can't read process id stamp\n",errstr);
	  return FALSE;
	}
      if (strcmp(str,"Processid") || strcmp(str2,"=")) /* wrong format */
	if (strcmp(str,"Image")) /* not right for load file either */
	  {
	    fprintf(stderr,"%s error: file is not a checkpoint file\n",errstr);
	    return FALSE;
	  }
	else			/* must be load file */
	  {
	    fprintf(stderr,
		    "%s error: file is a load file, cannot restore\n",errstr);
	    return FALSE;
	  }
    }
  else				/* load file */
    errstr = "load";

  if (fscanf(nfp, "%s %1s %s %s %[^\n]",str2,str,
	     fileimagename,str,timestamp) != 5)	/* read image timestamp */
    {
      fprintf(stderr,"%s error: can't read image time stamp\n",errstr);
      return FALSE;
    }
  if (strcmp(str2,"Image"))	/* correct image time stamp format? */
    if (type == 1)		/* checkpoint file */
      {
	fprintf(stderr,"%s error:  file is not a checkpoint file\n",errstr);
	return FALSE;
      }
    else			/* load file */
      if (strcmp(str2,"Processid")) /* not correct for checkpoint file */
	{
	  fprintf(stderr,"%s error:  file is not a load file\n",errstr);
	  return FALSE;
	}
      else                        /* must be checkpoint file  */
	{
	  fprintf(stderr,
		  "%s error: file is a checkpoint file, cannot load\n",errstr);
	  return FALSE;
	}
    
    
  timestamp[24] = '\0';		/* end of read in date string */
  imagetime[24] = '\0';		/* end of this image date string */
  if (strcmp(imagetime,timestamp)) /* different image */
    fprintf(stderr,"%s warning: the file was saved from a different simulator image\nto the one you are currently running:\nOld was %s written at %s\nNew is  %s written at %s\n",
	    errstr, fileimagename, timestamp, imagename, imagetime);
  else
    if (type == 1)		/* checkpoint file */
      if (pid != getpid())	/* check process the same */
	fprintf(stderr,
		"warning: restoring state from previous session\n");
  if (fscanf(nfp,"%s %s %1s %[^\n] %ld",str,str,str,
	     checktime,&its) != 5)
    {
      fprintf(stderr,"%s error: can't read file time stamp\n",errstr);
      return FALSE;
    }
  if (gettimeofday(&tp,&tzp) == -1)
    {
      perror("simulator - can't get time of day");
      return FALSE;
    }
  if (its - tp.tv_sec > 604800)	/* longer than one week */
    fprintf(stderr,
	    "%s warning: file is over one week old\n",errstr);
  return TRUE;
}


	  

