/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

/********************************************************************
 * X11 Graphics Interface Widget
 *
 * This file contains routines to initialize the display widget
 * and to setup and change the popup menus for custom mode.
 *
 * Nigel Goddard
 *   first implementation    :   April 1989
 *   custom popup menus added:   May   1989
 *********************************************************************/

#include <X11/IntrinsicP.h>
#include <X11/Xlib.h>
#include <X11/StringDefs.h>
#include <X11/List.h>
#include <X11/Shell.h>
#include "DisplayP.h"
#include "macros.h"
#include "externs.h"
#include "sim.h"

#define EDIT_ITEM 0
#define EXECUTE_ITEM 1
#define DELETE_ITEM 2

#define SIM_CMD 0
#define ARGV_FUNC 1

typedef struct sim_pars
{
  char * cmd;
} sim_cmd, * sim_cmd_ptr;

typedef struct argv_func_pars
{
  char * func;
  char ** argv;
  int argc;
} argv_func, * argv_func_ptr;

typedef struct m_item
{
  int type;
  union
    {
      sim_cmd_ptr scmd;
      argv_func_ptr avfunc;
    } item;
} menu_item, * menu_item_ptr;


static XtResource resources[] = {
#define offset(field) XtOffset(DisplayWidget, display.field)
    /* {name, class, type, size, offset, default_type, default_addr}, */
    {XtNredrawCallback, XtCCallback, XtRCallback, sizeof(XtCallbackList),
       offset(redraw_callback), XtRCallback, NULL },
    {XtNexposeCallback, XtCCallback, XtRCallback, sizeof(XtCallbackList),
       offset(expose_callback), XtRCallback, NULL },
    {XtNactionCallback, XtCCallback, XtRCallback, sizeof(XtCallbackList),
       offset(action_callback), XtRCallback, NULL },
#undef offset
};

static void DisplayExpose();
static void Redisplay();
static void DisplayAction();
static void donothing(){};
static void Initialize();
static void MenuSelect();
static void MenuGrab();
static void ItemDelete();
static void ItemExecute();
static void PostExecute();
static void ItemEdit();

static void do_x_menu_item();
static void do_item_edit();

static XtActionsRec actions[] =
{
  /* {name, procedure}, */
    {"input",	DisplayAction},
    {"null",    donothing},
    {"MenuSelect", MenuSelect},
    {"MenuGrab", MenuGrab},
    {"ItemEdit", ItemEdit},
    {"ItemExecute", ItemExecute},
    {"PostExecute", PostExecute},
    {"ItemDelete", ItemDelete},
};

static char main_translations[] =
"  <Key>:		input()	\n\
   <BtnDown>:           input() \n\
   <BtnUp>:             input() \n\
   <BtnMotion>:         input() \n\
   <Btn1Down>:          null() \n\
   <Btn2Down>:          null() \n\
   <Btn3Down>:          null() \n\
";

static char custom_translations[] =
"  <Btn1Down>:          MenuSelect(0) MenuPopup(custom_menu) MenuGrab(0) \n\
   <Btn2Down>:          MenuSelect(1) MenuPopup(custom_menu) MenuGrab(1) \n\
   <Btn3Down>:          MenuSelect(2) MenuPopup(custom_menu) MenuGrab(2) \n\
   <Key>:		null()	\n\
   <BtnDown>:           null() \n\
   <BtnUp>:             null() \n\
   <BtnMotion>:         null() \n\
";

static char menu_translations[] =
"  None<Leave>:         MenuPopdown() \n\
";

static char item_translations[] =
"  <Btn1Down>:         ItemExecute() Set() Notify() PostExecute() Unset() \n\
   <Btn2Down>:         ItemEdit() Set() Notify() Unset() \n\
   <Btn3Down>:         ItemDelete() Set() Notify() Unset() \n\
";

static XtTranslations main_trans;
static XtTranslations custom_trans;
static XtTranslations menu_trans;
static XtTranslations item_trans;

static Widget menu_list[3] = { NULL, NULL, NULL };
static Widget current_menu = NULL; /* holds pointer to current list widget (menu) */
static Widget custom_popup = NULL;
static Widget editpopup, editdialog;
static Widget current_display;	/* holds pointer to display in which we are now */

static int NotifyAction = EXECUTE_ITEM;
static int buttonx, buttony;	/* globals where button event occurred */
static int edit_menu;
static int edit_item;

#define ITEMS_PER_MENU 11
static menu_item_ptr menu_item_content[3][ITEMS_PER_MENU];
static char unused[] = "  unused  ";
static char * menu_items[3][ITEMS_PER_MENU] =
{ 
  { "  MENU A  ", unused, unused, unused, unused, unused,
      unused, unused, unused, unused, unused },
  { "  MENU B  ", unused, unused, unused, unused, unused,
      unused, unused, unused, unused, unused },
  { "  MENU C  ", unused, unused, unused, unused, unused,
      unused, unused, unused, unused, unused },
};

DisplayClassRec displayClassRec = {
  { /* core fields */
    /* superclass		*/	(WidgetClass) &widgetClassRec,
    /* class_name		*/	"Display",
    /* widget_size		*/	sizeof(DisplayRec),
    /* class_initialize		*/	NULL,
    /* class_part_initialize	*/	NULL,
    /* class_inited		*/	FALSE,
    /* initialize		*/	NULL,
    /* initialize_hook		*/	Initialize,
    /* realize			*/	XtInheritRealize,
    /* actions			*/	actions,
    /* num_actions		*/	XtNumber(actions),
    /* resources		*/	resources,
    /* num_resources		*/	XtNumber(resources),
    /* xrm_class		*/	NULLQUARK,
    /* compress_motion		*/	TRUE,
    /* compress_exposure	*/	TRUE,
    /* compress_enterleave	*/	TRUE,
    /* visible_interest		*/	FALSE,
    /* destroy			*/	NULL,
    /* resize			*/	Redisplay,
    /* expose			*/	DisplayExpose,
    /* set_values		*/	NULL,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	XtInheritSetValuesAlmost,
    /* get_values_hook		*/	NULL,
    /* accept_focus		*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private		*/	NULL,
    /* tm_table			*/	main_translations,
    /* query_geometry		*/	XtInheritQueryGeometry,
    /* display_accelerator	*/	XtInheritDisplayAccelerator,
    /* extension		*/	NULL
  },
 {
    /* display fields */
    /* empty			*/	0
  }
};

DisplayWidgetClass displayWidgetClass = (DisplayWidgetClass)&displayClassRec;

/*------------------------private routines------------------------*/

static void Initialize(w, args, num_args)
     Widget w;
     ArgList args;		/* NULL */
     Cardinal * num_args;	/* NULL */

{
  static Arg a[] =
    {
      { XtNlist, (XtArgVal) NULL },
      { XtNverticalList, (XtArgVal) True },
      { XtNdefaultColumns, (XtArgVal) 1 },
      { XtNnumberStrings, (XtArgVal) ITEMS_PER_MENU },
      { XtNfont, (XtArgVal) &gi_default_font },
    };
  int i;

  if (custom_popup != NULL)
    return; 

  XtAddActions(actions, XtNumber(actions));
  main_trans = XtParseTranslationTable(main_translations);
  custom_trans = XtParseTranslationTable(custom_translations);
  menu_trans = XtParseTranslationTable(menu_translations);
  item_trans = XtParseTranslationTable(item_translations);

  custom_popup = XtCreatePopupShell("custom_menu",overrideShellWidgetClass,
				    gi_tool,NULL,0);
  XtOverrideTranslations(custom_popup, menu_trans);
  for (i = 0; i < 3; i++)
    {
      a[0].value = (XtArgVal) menu_items[i];
      menu_list[i] = XtCreateWidget("menu_list", listWidgetClass,
				    custom_popup, a, XtNumber(a));
      XtOverrideTranslations(menu_list[i], item_trans);
      XtAddCallback(menu_list[i], XtNcallback, do_x_menu_item, i);
    }
}

static void DisplayAction(w, event, params, num_params)
     Widget w;
     XEvent* event;
     String* params;
     Cardinal* num_params;

{
  XtCallCallbacks(w, XtNactionCallback, (caddr_t)event);
}

static void DisplayExpose(w, event, region)
     Widget w;
     XEvent *event;
     Region region;

{
  XtCallCallbacks(w, XtNexposeCallback, (caddr_t)event);
}

static void Redisplay(w, event, region)
     Widget w;
     XEvent *event;
     Region region;

{
  XtCallCallbacks(w, XtNredrawCallback, (caddr_t)event);
}

static void MenuSelect(w, event, params, num_params)
     Widget w;
     XEvent* event;
     String* params;
     Cardinal* num_params;

{
  XWindowAttributes atts;
  Widget temp;
  Position x,y;
  int menu;

				/* get display window location */
  XGetWindowAttributes(XtDisplay(gi_tool), XtWindow(XtParent(XtParent(XtParent(w)))), &atts);
  x = (buttonx = event->xbutton.x) + atts.x;	/* add in button location */
  y = (buttony = event->xbutton.y) + atts.y;

  if (*num_params != 1)
    {
      fprintf(stderr, "error in move menu: 1");
      return;
    }
  sscanf(params[0],"%d", &menu);
  if (menu < 0 || menu > 2)
    {
      fprintf(stderr, "error in move menu: 2");
      return;
    }
  XtMoveWidget(custom_popup, x-20, y-20); /* move into menu */
  if (current_menu != NULL)
    XtUnmanageChild(current_menu);
  gi_put_message("  execute  |  edit  |  delete  ");
  current_display = w;		/* save id of display for items to use */
  XtManageChild(current_menu = menu_list[menu]);
}

static void MenuGrab(w, event, params, num_params)
     Widget w;
     XEvent* event;
     String* params;
     Cardinal* num_params;

{
  int menu;

  if (*num_params != 1)
    {
      fprintf(stderr, "error in menu grab: 1");
      return;
    }
  sscanf(params[0],"%d", &menu);
  if (menu < 0 || menu > 2)
    {
      fprintf(stderr, "error in move menu: 2");
      return;
    }
  XtAddGrab(menu_list[menu], False, False);
}

static void ItemEdit(w, event, params, num_params)
     Widget w;
     XEvent* event;
     String* params;
     Cardinal* num_params;

{
  NotifyAction = EDIT_ITEM;
}

static void ItemExecute(w, event, params, num_params)
     Widget w;
     XEvent* event;
     String* params;
     Cardinal* num_params;

{
				/* select globals for current display window */
  XtCallCallbacks(current_display, XtNactionCallback, (caddr_t)event);
  NotifyAction = EXECUTE_ITEM;
}

static void PostExecute(w, event, params, num_params)
     Widget w;
     XEvent* event;
     String* params;
     Cardinal* num_params;

{
  if (gi_reshow_flag)
    gi_reshow_all(); 
}

static void ItemDelete(w, event, params, num_params)
     Widget w;
     XEvent* event;
     String* params;
     Cardinal* num_params;

{
  NotifyAction = DELETE_ITEM;
}

static void do_x_menu_item(w, menu, item)
     Widget w;
     int menu;			/* menu number */
     XtListReturnStruct * item;	/* item */
     
{
  menu_item_ptr mitemptr;
  char err[128];
  int i;

  if (menu < 0 || menu > 2)
    {
      gi_put_error("Invalid menu parameter in AddItemToXMenu");
      return ;
    }
  if (item->index == 0)
    return;			/* clicked on menu name, should put up help info */
  if (item->index < 0 || item->index >= ITEMS_PER_MENU)
    {
      gi_put_error("Invalid menu parameter in AddItemToXMenu");
      return ;
    }

  if (NotifyAction != EDIT_ITEM && menu_items[menu][item->index] == unused)
    return;

  mitemptr = menu_item_content[menu][item->index];
  if (NotifyAction != EDIT_ITEM && mitemptr == NULL)
    {
      (void) sprintf(err, "Screw up! can't find item %s in menu %s",
	      item->string, menu_items[menu][0]);
      gi_put_error(err);
      return;
    }

  switch (NotifyAction)
    {
    case EDIT_ITEM:
      do_item_edit(w, menu, item);
      break;
    case EXECUTE_ITEM:
      switch(mitemptr->type)
	{
	case SIM_CMD:
	  gi_process_button(mitemptr->item.scmd->cmd, buttonx, buttony);
	  break;
	case ARGV_FUNC:
	  NameToFunc(mitemptr->item.avfunc->func)
	    (mitemptr->item.avfunc->argc, mitemptr->item.avfunc->argv);
	  break;
	}
      break;
    case DELETE_ITEM:
      switch(mitemptr->type)
	{
	case SIM_CMD:
	  free(mitemptr->item.scmd->cmd);
	  break;
	case ARGV_FUNC:
	  for (i = 0; i < mitemptr->item.avfunc->argc; i++)
	    free((mitemptr->item.avfunc->argv)+i);
	  free(mitemptr->item.avfunc->func);
	  free(mitemptr->item.avfunc);
	  break;
	}
      free(mitemptr);
      menu_item_content[menu][item->index] = NULL;
      free(menu_items[menu][item->index]);
      menu_items[menu][item->index] = unused;
      break;
    default:
      fprintf(stderr, "do_x_menu (invalid) called for menu %s\n", menu_items[menu][0]);
    }      
}

static void change_menu_item(w, mitemptr, call_data)
     Widget w;
     menu_item_ptr * mitemptr;
     caddr_t call_data;

{
  int i;
  char * str;
  char name[32], * cmd;
  char err[128];
  menu_item_ptr mptr;
  char colon[128];

  str = XtDialogGetValueString(editdialog);
  if (sscanf(str, "%s %s ", name, colon) != 2 || strcmp(colon,":"))
    {
      gi_put_error("Wrong syntax! Use \"name : command\"");
      goto gout;
    }
  if (!strcmp(name, "unused") && *mitemptr == NULL)
    goto gout;
  for (i = 0; i < ITEMS_PER_MENU; i++)
    if (i != edit_item && !strcmp(menu_items[edit_menu][i], name))
      {
	(void) sprintf(err, "Menu %s already contains item %s!!",
		menu_items[edit_menu][0], name);
	gi_put_error(err);
	goto gout;
      }
  if (*mitemptr == NULL)
    {
      *mitemptr = (mptr) = (menu_item_ptr) malloc (sizeof(menu_item));
      mptr->type = SIM_CMD;
      mptr->item.scmd = (sim_cmd_ptr) malloc (sizeof(sim_cmd));
    }
  else
    switch((mptr = *mitemptr)->type)
      {
	  case SIM_CMD:
	    free(mptr->item.scmd->cmd);
	    break;
	  case ARGV_FUNC:
	    for (i = 0; i < mptr->item.avfunc->argc; i++)
	      free((mptr->item.avfunc->argv)+i);
	    free(mptr->item.avfunc->func);
	    free(mptr->item.avfunc);
	    if (strcmp(name, "unused"))
	      {
		mptr->item.scmd = (sim_cmd_ptr) malloc (sizeof(sim_cmd));
		mptr->type = SIM_CMD;
	      }
	    break;
	  }
  if (strcmp(name, "unused"))
    {
      cmd = str + strlen(name) + 3;
      mptr->item.scmd->cmd = (char *) malloc (strlen(cmd)+1);
      strcpy(mptr->item.scmd->cmd, cmd);
      if (strcmp(name, menu_items[edit_menu][edit_item]))
	{
	  if (menu_items[edit_menu][edit_item] == unused)
	    menu_items[edit_menu][edit_item] = (char *) malloc (sizeof(char) * 11);
	  if (strlen(name) > 10)
	    {
	      gi_put_error("Only accept first ten characters of item name");
	      name[10] = '\0';
	    }
	  (void) sprintf(menu_items[edit_menu][edit_item], "%s", name);
	}
    }
  else
    menu_items[edit_menu][edit_item] = unused;
  XtListChange(current_menu, menu_items[edit_menu], ITEMS_PER_MENU, 0, False);

 gout:
  XtDestroyWidget(editpopup);
  XWarpPointer(XtDisplay(gi_tool), None, XtWindow(custom_popup), 0, 0, 0, 0, 20, 20);
}

static void cancel_item_change(w, mitemptr, call_data)
     Widget w;
     menu_item_ptr mitemptr;
     caddr_t call_data;

{
  XtDestroyWidget(editpopup);
  XWarpPointer(XtDisplay(gi_tool), None, XtWindow(custom_popup), 0, 0, 0, 0, 20, 20);
}

static void do_item_edit(w, menu, item)
     Widget w;
     int menu;			/* menu number */
     XtListReturnStruct * item;	/* item */
     
{
  menu_item_ptr mitemptr;
  char cmd[256];
  char buf[256], * t;
  int i;

  static Arg p[] = 
    {
      { XtNx, (XtArgVal) 0 },
      { XtNy, (XtArgVal) 0 },
      { XtNwidth, (XtArgVal) 0 },
      { XtNheight, (XtArgVal) 70 },
    };
  static Arg d[] = 
    {
      { XtNvalue, (XtArgVal) NULL },
      { XtNlabel, (XtArgVal) "Specify Item and Command, separated by a colon, e.g. list : l u all" },
      { XtNwidth, (XtArgVal) 0 },
      { XtNborderWidth, (XtArgVal) 0 },
    };
  XWindowAttributes atts;

  XGetWindowAttributes(XtDisplay(gi_tool), XtWindow(XtParent(w)), &atts);
  p[0].value = atts.x;		/* display dialog at same location */
  p[1].value = atts.y;

  mitemptr = menu_item_content[menu][item->index];
  edit_menu = menu;
  edit_item = item->index;
  if (mitemptr == NULL)
    (void) sprintf(buf, "unused : ");
  else
    switch(mitemptr->type)
      {
      case SIM_CMD:
	(void) sprintf(buf, "%s : %s", menu_items[menu][item->index],
		mitemptr->item.scmd->cmd);
	break;
      case ARGV_FUNC:		/* convert to SIM_CMD */
	(void) sprintf(buf, "%s : call %s", menu_items[menu][item->index],
		mitemptr->item.avfunc->func);
	for (i = 0, t = buf + strlen(buf);
	     i < mitemptr->item.avfunc->argc;
	     t += strlen(mitemptr->item.avfunc->argv[i]) + 1, i++)
	  (void) sprintf(t, " %s", mitemptr->item.avfunc->argv[i]);
	break;
      }
  
  d[0].value = (XtArgVal) buf;
  if (strlen(buf) < 70)
    d[2].value = p[2].value = (XtArgVal) 40 + ((int) (gi_font_width * 70));
  else
    d[2].value = p[2].value = (XtArgVal) ((int) (strlen(buf) * gi_font_width)) + 40;

  editpopup = XtCreatePopupShell("menu_dialog",transientShellWidgetClass,
				     w, p, XtNumber(p));
  editdialog = XtCreateManagedWidget("dialog_box", dialogWidgetClass,
					     editpopup,d, XtNumber(d));
  AddButton(editdialog, "CHANGE", change_menu_item,
	    &(menu_item_content[menu][item->index]), NULL);
  AddButton(editdialog, "CANCEL", cancel_item_change, NULL, NULL);
  XtPopup(editpopup, XtGrabExclusive);
}

static void log_item(menu, mitemptr, name)
     int menu;
     menu_item_ptr mitemptr;
     char * name;

{
  char buf[256], * t;
  int i;

  switch(mitemptr->type)
    {
    case SIM_CMD:
      (void) sprintf(buf, "gi b %d \"%s\" %s", menu+1, mitemptr->item.scmd->cmd, name);
      break;
    case ARGV_FUNC:
      (void) sprintf(buf, "gi b %d \"%s ", menu+1, mitemptr->item.avfunc->func);
      t = buf + strlen(buf);
      for (i = 0, t = buf + strlen(buf);
	   i < mitemptr->item.avfunc->argc;
	   t += strlen((mitemptr->item.avfunc->argv) + i) + 1, i++)
		(void) sprintf(t, "%s ", (mitemptr->item.avfunc->argv) + i);
      (void) sprintf(t, "\" %s", name);
      break;
    }
  gi_show_prev(buf);
  gi_log(buf);
}
  
/*----------------------------public routines---------------------*/

Bool AddItemToXMenu(menu,name, type, cmd, argv, argc)
     int menu;			/* menu index */
     char * name;		/* menu item name*/
     int type;			/* type of item */
     char * cmd;		/* sim cmd or func name */
     char ** argv;		/* for func */
     int argc;			/* for func */

{
  char err[128];
  menu_item_ptr mitemptr;
  int i;

  if (menu < 0 || menu > 2)
    {
      gi_put_error("Invalid menu parameter in AddItemToXMenu");
      return False;
    }
  if (type != SIM_CMD && type != ARGV_FUNC)
    {
      gi_put_error("Invalid type parameter in AddItemToXMenu");
      return False;
    }
  if (cmd == NULL)
    {
      gi_put_error("Invalid command parameter in AddItemToXMenu");
      return False;
    }

  for (i = 0; i < ITEMS_PER_MENU; i++)
    if (!strcmp(name, menu_items[menu][i]))
      {
	(void) sprintf(err, "%s already has item %s", menu_items[menu][0], name);
	gi_put_error(err);
	return False;
      }
  for (i = 0; i < ITEMS_PER_MENU; i++)
    if (!strcmp(menu_items[menu][i], unused))
      break;
  if (i >= ITEMS_PER_MENU)	/* room for another item? */
      {
	(void) sprintf(err, "%s is full", menu_items[menu][0]);
	gi_put_error(err);
	return False;
      }
  if (menu_items[menu][i] != unused)
      {
	(void) sprintf(err, "%s is in deep doo-doo (thanks George)", menu_items[menu][0]);
	gi_put_error(err);
	return False;
      }
  if (menu_item_content[menu][i] != NULL)
      {
	(void) sprintf(err, "%s is in deeper doo-doo (thanks George)", menu_items[menu][0]);
	gi_put_error(err);
	return False;
      }
  
  menu_items[menu][i] = (char *) malloc(sizeof(char) * 11);
  (void) sprintf(menu_items[menu][i], "%s", name);		/* save it */

  mitemptr = (menu_item_ptr) malloc (sizeof(menu_item)); /* make new menu item */
  mitemptr->type = type;
  switch(type)
    {
    case SIM_CMD:
      mitemptr->item.scmd = (sim_cmd_ptr) malloc (sizeof(sim_cmd));
      mitemptr->item.scmd->cmd = (char *) malloc (sizeof(char) * (strlen(cmd)+1));
      strcpy(mitemptr->item.scmd->cmd, cmd);
      break;
    case ARGV_FUNC:
      mitemptr->item.avfunc = (argv_func_ptr) malloc (sizeof(argv_func));
      mitemptr->item.avfunc->func = (char *) malloc (sizeof(char) * (strlen(cmd)+1));
      strcpy(mitemptr->item.avfunc->func, cmd);
      mitemptr->item.avfunc->argv = argv;
      mitemptr->item.avfunc->argc = argc;
      break;
    default:
      (void) sprintf(err, "No such menu item type %d", type);
      gi_put_error(err);
      return False;
    }
  menu_item_content[menu][i] = mitemptr;
  return True;
}

Bool DeleteItemFromXMenu(menu, name)
     int menu;			/* menu index */
     char * name;		/* menu item */
     
{
  char err[128];
  menu_item_ptr mitemptr, pitemptr;
  int i;
  
  if (menu < 0 || menu > 2)
    {
      gi_put_error("Invalid menu parameter in DeleteItemFromXMenu");
      return False;
    }

  for (i = 0; i < ITEMS_PER_MENU; i++)
    if (!strcmp(name, menu_items[menu][i]))
      break;
  if (i >= ITEMS_PER_MENU)
    {
      (void) sprintf(err, "Menu %s has no item %s", menu_items[menu][0], name);
      gi_put_error(err);
      return False;
    }

  mitemptr = menu_item_content[menu][i];
  switch(mitemptr->type)
    {
    case SIM_CMD:
      free(mitemptr->item.scmd->cmd);
      free(mitemptr->item.scmd);
      break;
    case ARGV_FUNC:
      for (i = 0; i < mitemptr->item.avfunc->argc; i++)
	free((mitemptr->item.avfunc->argv)+i);
      free(mitemptr->item.avfunc->func);
      free(mitemptr->item.avfunc);
      break;
    }
  free(mitemptr);
  menu_item_content[menu][i] = NULL;
  free(menu_items[menu][i]);
  menu_items[menu][i] = unused;
  return True;
}

void SetDisplayModeToCustom(w,flag)
     Widget w;
     int flag;

{
  if (flag == TRUE)
    XtOverrideTranslations(w, custom_trans);
  else
    XtOverrideTranslations(w, main_trans);
}

void LogCustomXMenus()

{
  menu_item_ptr mitemptr;
  int i, j;
  
  for (i = 0; i < 3; i++)
    for (j = 1; j < ITEMS_PER_MENU; j++)
      if (menu_items[i][j] != unused)
	log_item(i, menu_item_content[i][j], menu_items[i][j]);
}
