/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

/***********************************************************************
 * Graphics Interface
 * ------------------
 * This file contains all the routines that manipulate and access
 * the command panel, in particular, it processes all string commands
 * from both the command panea, from command files read in, and
 * from calls within the user code (at gi_command)
 *
 * Kenton Lynne
 *                02/02/88
 * Nigel Goddard
 *   hacked for X: April, May 1989
 **********************************************************************/
#include "macros.h"
#include "externs.h"
#include "panel.h"
#include "control_panel.h"
#include "mode_panel.h"
#include "info_panel.h"
#include <X11/AsciiText.h>
#include <X11/Text.h>
#include <X11/Box.h>
#include <X11/Xmu.h>
#include <X11/Label.h>

/* these defines are for the gi_parse State machine */
#define START_STATE   0
#define REG_STATE     1
#define QUO_STATE     2
#define CHK_QUO_STATE 3

#define DR_NUM_VERTICES 2
#define TX_STRING_ARG 2
#define TX_LOCX_ARG   3
#define TX_LOCY_ARG   4
#define TX_FONT_ARG   5

/* number of nested read files allowed */
#define MAXFILES 8

/* panel items that are globally accessed */
extern Widget msg_panel;
Widget gi_cmd_prev_item;
char gi_cmd_prev_item_buf[MAX_CMD_LENGTH];

static Widget cmd_item;
static char cmd_item_buf[MAX_CMD_LENGTH];
static char filenamebuf[128];
char * gi_prev_fname;

/* routines that are forward referenced */
static void cmd_proc();

static Arg cmd_item_args[] = {
		{ XtNfromHoriz, (XtArgVal)NULL },
		{ XtNwidth, (XtArgVal)400 },
		{ XtNfromVert, (XtArgVal)NULL },
		{ XtNhorizDistance, (XtArgVal) 0 },
		{ XtNvertDistance, (XtArgVal) 5 },
		{ XtNstring, (XtArgVal) cmd_item_buf},
		{ XtNlength, (XtArgVal) MAX_CMD_LENGTH - 3},
		{ XtNborderWidth, (XtArgVal) 0 },
		{ XtNeditType, (XtArgVal) XttextEdit},
		{ XtNtextOptions, (XtArgVal)(editable | scrollOnOverflow | wordBreak) },
		{ XtNheight, (XtArgVal)40 },
		{ XtNfont, (XtArgVal) &gi_default_font },
};
static Arg cmd_prev_item_args[] = {
		{ XtNfromHoriz, (XtArgVal)NULL },
		{ XtNwidth, (XtArgVal)400 },
		{ XtNfile, (XtArgVal) filenamebuf },
		{ XtNhorizDistance, (XtArgVal)10 },
		{ XtNeditType, (XtArgVal)XttextAppend},
		{ XtNtextOptions, (XtArgVal) (scrollVertical | wordBreak)},
		{ XtNlength, (XtArgVal) MAX_CMD_LENGTH},
		{ XtNheight, (XtArgVal)60 },
		{ XtNfont, (XtArgVal) &gi_default_font },
};

static char override_trans_table[] =
"<Key>Return:        process_input(cmd_item) \n\
 <Key>Linefeed:      process_input(cmd_item) \n\
 Ctrl<Key>U:         beginning-of-line() kill-to-end-of-line() \n\
";

static XtActionsRec actionsList[] =
{
  {"process_input", cmd_proc },
};

static XtTranslations translations;

extern char *tempnam();

/*****************************************************************/
gi_make_cmd_panel(gi_tool)
     Widget gi_tool;
{
/* creates the command panel 
 */
  Widget prev,cmd;
  Widget cmd_panel; 

  FILE * fp;

  static Arg no_bdr [] = 
    {
      { XtNborderWidth, (XtArgVal) 0 },
      { XtNdefaultDistance, (XtArgVal) 2 },
    };
  static Arg prompt [] =
    {
      { XtNfromVert, (XtArgVal) NULL },
      { XtNvertDistance, (XtArgVal) 5 },
      { XtNjustify, (XtArgVal) XtJustifyRight },
      { XtNstring, (XtArgVal) "->" },
      { XtNborderWidth, (XtArgVal) 0 },
      { XtNhorizDistance, (XtArgVal) 5 },
      { XtNfromHoriz, (XtArgVal) NULL },
      { XtNfont, (XtArgVal) &gi_default_font },
    };

  /* initial definition of the command panel */
  /* need to do something about panel width! */

  if ((cmd_panel = XtCreateManagedWidget("CMD_PANEL", formWidgetClass, gi_tool,
					 no_bdr, XtNumber(no_bdr))) == NULL)
    {
      fputs("Can't create command panel\n", stderr);
      exit(1);
    }

  if ((fp = fopen((gi_prev_fname = tempnam("/tmp","gi.log")),"w")) == NULL)
    {
      fputs("Can't create log file\n", stderr);
      exit(1);
    }
  strcpy(filenamebuf, gi_prev_fname); /* for making widget */
  fclose(fp);			/* just make sure it exists */
  
  /* set up the individual panel items */

  cmd_prev_item_args[1].value = cmd_item_args[1].value =
    (XtArgVal) ((int) (gi_font_width * 65));
  gi_cmd_prev_item = XtCreateManagedWidget("cmd_prev_item", asciiDiskWidgetClass,
					   cmd_panel,cmd_prev_item_args,
					   XtNumber(cmd_prev_item_args));

  prompt[0].value = (XtArgVal) gi_cmd_prev_item;
  cmd_item_args[2].value = (XtArgVal) gi_cmd_prev_item;
  cmd_item_args[0].value = (XtArgVal) XtCreateManagedWidget("->", labelWidgetClass,
							    cmd_panel, prompt,
							    XtNumber(prompt));

  strcpy(cmd_item_buf, "");
  cmd_item=XtCreateManagedWidget("cmd_item",asciiStringWidgetClass,cmd_panel,
				cmd_item_args, XtNumber(cmd_item_args));
  XtAddActions(actionsList, XtNumber(actionsList));
  translations = XtParseTranslationTable(override_trans_table);
  XtOverrideTranslations(cmd_item, translations);
}

  
/*****************************************************************/
static void cmd_proc(item, client_data, call_data)
     Widget item;
     caddr_t client_data, call_data;

{
/* called when the user presses return or linefeed in
 * the command panel, this routine parses the command and
 * then routes it appropriately
 */
  char cmd[MAX_CMD_LENGTH+1];
  int cmdlen, clock;
  XtTextBlock text;

     /* save the command in a 'safe' area */
  cmdlen = XtTextGetInsertionPoint(item) - XtTextTopPosition(item);
  strncpy(cmd, cmd_item_buf+XtTextTopPosition(item), cmdlen);
  cmd[cmdlen] = '\0';

  /* execute the command */
  gi_command(cmd);
  
  /* if logging enabled, log this command */
/*  if (gi_log_on) gi_log(cmd);*/
  
  /* if display needs to be updated, call gi_reshow to do it */ 
  if (gi_reshow_flag) 
    gi_reshow_all();

  text.firstPos = 0;
  text.length = 0;
  text.ptr = "";
  text.format = FMT8BIT;
  
  XtTextReplace(item, 0, XtTextGetInsertionPoint(item), &text);
  XtTextSetInsertionPoint(item, 0);
}


/*****************************************************************/
gi_parse_cmd(string, args, arg_string)
  char *string, **args, *arg_string;
{
/* given a string, an array of arg pointers (ie. like argv)
 * and a working buffer, it parses the string into arguments moving the
 * arguments to the working buffer and setting the arg pointers pointing
 * to the begining of each arg in the working buffer
 * which has been translated to a string
 * and returns the number of arguments found
 * Requires: args to be at least MAX_ARGS long
 *           arg_string to be at least MAX_CMD_LENGTH long
 *           string must not be null
 * Note: this treats quoted strings specially
 *       - strips off surrounding double quotes and
 *       - treats 2 double quotes as a single double quote
 * Note: will parse only MAX_ARGS number of arguments and
 *       ignore the rest of the line
 */

  int arg_index, state, i;
  char last_char;
  char *ptr;

  /* initialize all the argument pointers to NULL */
  for (i=0; i<MAX_ARGS; i++) args[i] = NULL;

  arg_index = 0;
  state = START_STATE;
  ptr = arg_string;
  
  /* parse the command line into individual arguments */
  do
  {
    /* check for end of string */
    if ((last_char = *string++)!='\0')

    /* for each character, pass it through the following FA */
    switch (state)
    {
       case START_STATE:
            /* if char is non_blank, set the next arg pointer to it */
              if (last_char!=BLANK) 
              {
                args[arg_index++] = ptr;
                /* go to appropriate next state 
                   depending on whether this
                   arg is quoted 
                */
                if (last_char==DQUOTE)
                  state = QUO_STATE;
                else
                {
                  *ptr++ = last_char;
                  state = REG_STATE;
                }
              }
              break;

       case REG_STATE:
              /* append non-blank chars to current argument */
              if (last_char!=BLANK)
                *ptr++ = last_char;
              else
              {
                *ptr++ = '\0';
                state = START_STATE;
              }
              break;

       case QUO_STATE:
              /* append non-quote characters to current argument */
              if (last_char!=DQUOTE)
              {
                *ptr++ = last_char;
              }
              else
                /*if quote shows up, check if end of string is indicated */
                state = CHK_QUO_STATE;
              break;
               
       case CHK_QUO_STATE:
              /* prev char was a quote, if this char is also a quote
                 write a single quote to the current argument 
                 otherwise end current argument and look for next
              */
              if (last_char==DQUOTE)
              {
                *ptr++ = last_char;
                state = QUO_STATE;
              }
              else
              {
                *ptr++ = '\0';
                state = START_STATE;
              }
              break;
    }
  }
  while (last_char!='\0' && ptr < &arg_string[MAX_CMD_LENGTH]);

  /* put an end-of-string char at end of last arg */
  *ptr = '\0';

  /* return number of arguments parsed */
  return(arg_index);
}

/*************************************************************************/
static do_set_buttons(numargs,cmd_args)
  int numargs;
  char *cmd_args[]; 
{
/* processes a command to set up the menus
 * for use in custom mode
 * cmd format: gi b # "command string" "item name"
 *             where # is the menu number (1 - 3)
 *             and "item name" is optional
 */

  int index;
  char name[11];
  char * cmd;

  /* check number of arguments first */
  if (numargs < 4 || numargs > 5)
  {
    gi_put_error("Wrong number of args in set button command");
    return(ERROR);
  }

  /* make sure third arg is a button number */
  if (gi_check_num(cmd_args[2])!=OK)
  {
    gi_put_error("Button argument must be numeric");
    return(ERROR);
  }

  index = atoi(cmd_args[2]);

  /* check the index argument */
  if (index < 1 || index > 3)
  {
    gi_put_error("Invalid button argument");
    return(ERROR);
  }
 
  if (numargs == 4)
    cmd = cmd_args[3];
  else
    cmd = cmd_args[4];
  if (strlen(cmd) <= 10)
    (void) sprintf(name, "%s", cmd);
  else
    {
      strncpy(name, cmd, 10);
      name[10] = '\0';
    }

  AddItemToXMenu(index-1, name, 0, cmd_args[3], NULL, 0);

  return(OK);
}

/*************************************************************************/
extern char * gi_cur_ipanel;	/* not really char * */

static do_unitinfo(num_args,cmd_args)
  int num_args;
  char *cmd_args[];
{
/* processes an "info" command
 * (display individual unit information info display)
 * cmd format: gi i x y [sx sy]
 */

  int x, y, sx, sy;

  /* check number of arguments */
  if (num_args > 6 || num_args < 4)
  {
    gi_put_error("Invalid number of args in info command");
    return(ERROR);
  }


  /* check and convert x and y coordinate arguments */
  if (gi_conv_coord(cmd_args[2],&x)==ERROR
  || gi_conv_coord(cmd_args[3],&y)==ERROR)
  {
    gi_put_error("Invalid coordinates in info command");
    return(ERROR);
  }
  if (num_args < 6)
    sx = sy = -1;
  else
    {
      sscanf(cmd_args[4], "%d", &sx);
      sscanf(cmd_args[5], "%d", &sy);
    }
  
  /* find unit and put info in the appropriate column */
  gi_do_info(x, y, FALSE, sx, sy, gi_cur_ipanel);

  return(OK);
}
    
/*************************************************************************/
static do_reshow(numargs,cmd_args)
  int numargs;
  char *cmd_args[];
{
/* processes the reshow command
 * assuming that arguments are parsed into the cmd_args array
 */
  int new_x, new_y;

  if (numargs!=2 && numargs!=4)
  {
    gi_put_error("Wrong number of args in reshow command");
    return(ERROR);
  }

  if (numargs==4)
  {
    /* translate and check coordinates */
    if (gi_conv_coord(cmd_args[2],&new_x)==ERROR 
    || gi_conv_coord(cmd_args[3],&new_y)==ERROR)
    {
      gi_put_error("Invalid Origin coordinates");
      return(ERROR);
    }

    /* if origin has changed, make requested origin changes */
    if (new_x!=gi_origin_x || new_y!=gi_origin_y)
    {
       gi_change_origin(new_x-gi_origin_x,new_y-gi_origin_y); 
       gi_update_grobj_chain(TRUE, TRUE);
    }
  }

  /* force gi_reshow to clear screeen and redisplay everything */
  gi_reshow_flag |= RESHOW_ALL+CLEAR_NEEDED;
  gi_reshow();
  
  /* if we've gotten this far, everything must be ok */
  return(OK);
}

/*************************************************************************/
static do_logcmd(numargs,cmd_args)
  int numargs;
  char *cmd_args[];
{
/* processes the log command
 * assuming that arguments are parsed into the cmd_args array
 */
  int new_x, new_y;

  /* if logging is currently on, flush the buffer */
  if (gi_log_on && gi_log_fp) fflush(gi_log_fp);

  if (numargs!=3 && numargs!=4)
  {
    gi_put_error("Wrong number of args in log command");
    return(ERROR);
  }

  /* check that the 2nd argument is either "on" or "off" */
  if (!strcmp(cmd_args[2],LOG_OFF_STG))
    gi_log_on = FALSE;
  else if (!strcmp(cmd_args[2],LOG_ON_STG))
    gi_log_on = TRUE;
  else
  {
    gi_put_error("Log command arg is not `on' or `off'");
    return(ERROR);
  }

  /* check if optional file name is specified */
  if (numargs==4)
  {
    /* move filename to log filename area */
/* nhg no log file currently
    SetValue(gi_logfile_item, XtNstring, cmd_args[3]);
*/
  }

  /* Set the mode panel switch to correspond with log switch */
/*  SetValue(gi_log_item, XtNset, gi_log_on); nhg ??*/
  
  /* if we've gotten this far, everything must be ok */
  return(OK);
}

/*************************************************************************/
static do_go(num_args,cmd_args)
  int num_args;
  char *cmd_args[];
{
/* processes a "go" command
 * Assumes that command args are in cmd_arg array
 * command format: gi g [num_steps [num_update steps]]
 */
  int steps=1, update=1;

  /* check for correct number of arguments */
  if (num_args > 4)
  {
    gi_put_error("Wrong number of arguments in go command");
    return(ERROR);
  }

  /* check and translate number of steps argument */
  if (num_args >=3)
  {
    if (gi_check_num(cmd_args[2])==ERROR)
    {
      gi_put_error("Bad number of steps argument");
      return(ERROR);
    }
    steps = atoi(cmd_args[2]);
  }

  /* check and translate update steps argument */
  if (num_args==4)
  {
    if (gi_check_num(cmd_args[3])==ERROR)
    {
      gi_put_error("Bad update steps argument");
      return(ERROR);
    }
    update = atoi(cmd_args[3]);
  }

  /* run the simulator the requested number of steps */
  gi_do_steps(steps, update);

  /* if we've gotten this far, everything must be ok */
  return(OK);
}

/*************************************************************************/
static do_text(num_args,cmd_args)
  int num_args;
  char *cmd_args[];
{
/* processes a gi "text" command
 * Assumes that command args are in cmd_arg array
 * command format: gi text string xloc yloc [font]
 */
   XFontStruct * fptr;
   struct txobj *ptr;
   char *sptr;
   int len, x_coord, y_coord;

   /* error check number of arguments */
   if (num_args < 5)
   {
      gi_put_error("Not enough args for text command");
      return(ERROR);
   }

   /* convert and error check x and y coordinates */
   if (gi_conv_coord(cmd_args[TX_LOCX_ARG],&x_coord)==ERROR
   || gi_conv_coord(cmd_args[TX_LOCY_ARG],&y_coord)==ERROR)
   {
     gi_put_error("Invalid coordinates for text command");    
     return(ERROR);
   }

   /* error check font */
   if (cmd_args[TX_FONT_ARG] 
   && strlen(cmd_args[TX_FONT_ARG]) > MAX_FONT_LENGTH)
   {
     gi_put_error("Font name too long");
     return(ERROR);
   }

   /* check for valid font */
   if ((fptr=gi_get_font(cmd_args[TX_FONT_ARG]))==NULL)
   {
     return(ERROR);
   }
   
   /* save the text argument in string space */
   sptr = cmd_args[TX_STRING_ARG]; 
   for (len=0; *sptr!='\0'; sptr++, len++) gi_save_char(*sptr,len);

   /* build and display a text object for this string */
   if ((ptr=gi_remember_txobj(x_coord,y_coord,len,fptr))!=NULL)
   {
     /* if object is within display space window, display it */
     if (gi_overlap_txobj(ptr) && !(gi_reshow_flag & RESHOW_ALL))
       gi_display_txobj(ptr,GXxor);
   }

   /* we've gotten this far: everything must have gone ok */
   return(OK);
}

/*************************************************************************/
static do_draw(num_args,cmd_args)
  int num_args;
  char *cmd_args[];
{
/* processes a draw command assuming that the
 * command arguments are in cmd_args array
 * command format: gi draw #vertices | b x1 y1 x2 y2 ... xn yn
 *                 where N <= 10
 *                 (note: if 2nd argument is "b" then bounding box)
 */

   int ret_code = OK;
   struct drobj *ptr;
   int i, num_vertices;
   int cord[2*MAX_VERTICES];
   short flag;

   /* check number of arguments */
   if (num_args < 7 
   || num_args > (MAX_VERTICES*2+3) 
   || (num_args % 2)==0)
   {
      gi_put_error("Bad number of args for draw command");
      return(ERROR);
   }

   /* check if this is to be a bounding box and get #vertices arg */
   if (!strcmp(cmd_args[DR_NUM_VERTICES],BOUND_BOX_STG))
   {
     num_vertices = 5;
     flag = BOUND_BOX;
   }
   else
   {
     num_vertices = atoi(cmd_args[DR_NUM_VERTICES]);
     flag = 0;
   }

   /* check that number of vertices argument is correct */
   if (num_vertices!=(num_args-3)/2)
   {
     gi_put_error("Wrong number of vertices specified");
     return(ERROR);
   }

   /* convert and check that coordinates are in range */
   for (i=0; i<num_vertices*2; i++)
   {
     if (gi_conv_coord(cmd_args[i+3],&cord[i])!=OK) 
     {
       gi_put_error("Invalid coordinate in draw command");
       return(ERROR);
     }
   }

   /* create and chain the drobj and store first 2 vertices */
   if ((ptr=gi_remember_drobj(cord[0],cord[1],cord[2],cord[3],flag))==NULL)
   {
     return(ERROR);
   }

   /* now save the rest of the vertices (if any) */
   for (i=4; i<num_vertices*2; i+=2)
     ret_code = gi_save_vertex(ptr,cord[i],cord[i+1]);
   
   /* display the object if on the screen and not RESHOW_ALL */
   if (gi_overlap_drobj(ptr) && !(gi_reshow_flag & RESHOW_ALL))
     gi_display_drobj(ptr,GXcopy);

   /* return last return code from gi_save_vertex */
   return(ret_code);
}

/*************************************************************************/
static do_delete(num_args,cmd_args)
  int num_args;
  char *cmd_args[];
{
/* processes a delete command assuming that the
 * command arguments are in cmd_args array
 * command format: gi delete at_x at_y
 */

   int i;
   int x, y;     
   struct txobj *tptr;
   struct drobj *dptr;
   struct grobj *gptr;

   /* check number of arguments */
   if (num_args != 4 )
   {
      gi_put_error("Bad number of args for delete command");
      return(ERROR);
   }

   /* check and convert coordinates */
   if (gi_conv_coord(cmd_args[2],&x)==ERROR 
   || gi_conv_coord(cmd_args[3],&y)==ERROR)
   {
     gi_put_error("Invalid coordinates in delete command");
     return(ERROR);
   }

   /* try to locate text object */
   if ((tptr=gi_find_txobj(x,y))!=NULL)
   {
     if (gi_overlap_txobj(tptr))
       gi_display_txobj(tptr,GXxor);
     gi_erase_txobj(tptr);
   }

   /* if text not found, try to find drawn object */
   else if ((dptr=gi_find_drobj(x,y))!=NULL)
   {
     if (gi_overlap_drobj(dptr))
       gi_display_drobj(dptr,GXclear);
     gi_erase_drobj(dptr);
   }
     
   /* no object found at all: put out error message */
   else
   {
     gi_put_error("No object found at delete coordinates");
     return(ERROR);
   }
     
   /* made it this far; must have gone ok */
   return(OK);
}

/*************************************************************************/
static do_move(num_args,cmd_args)
  int num_args;
  char *cmd_args[];
{
/* processes a move command assuming that the
 * command arguments are in cmd_args array
 * command format: gi move fromx fromy tox toy
 */

   int i;
   int from_x, from_y, offset_x, offset_y;
   struct txobj *tptr;
   struct drobj *dptr;
   struct grobj *gptr;
   int cord[4];

   /* check number of arguments */
   if (num_args != 6 )
   {
      gi_put_error("Bad number of args for move command");
      return(ERROR);
   }

   /* convert and check coordinates */
   for (i=2; i<num_args; i++)
   {
     if (gi_conv_coord(cmd_args[i],&cord[i-2])!=OK) 
     {
       gi_put_error("Invalid coordinates for move command");
       return(ERROR);
     }
   }

   offset_x = cord[2] - cord[0];
   offset_y = cord[3] - cord[1];
   
   if ((tptr=gi_find_txobj(cord[0],cord[1]))!=NULL)
   {
     gi_move_txobj(tptr, offset_x, offset_y, TRUE);
   }
     
   else if ((dptr=gi_find_drobj(cord[0],cord[1]))!=NULL)
   {
     /* check if this is a bounding box */
     if (dptr->flag & BOUND_BOX)
     {
       /* find, mark and move all contained objects */
       gi_mark_bound(dptr, FALSE);
       gi_move_bound(offset_x, offset_y, FALSE);
     }
     else
       gi_move_drobj(dptr, offset_x, offset_y, GXclear, GXcopy);
   }
     
   else if ((gptr=gi_find_grobj(cord[0],cord[1]))!=NULL)
   {
     gi_move_grobj(gptr, offset_x, offset_y, TRUE);
   }
   else
   {
     gi_put_error("No object found at move coordinates");
     return(ERROR);
   }
  
   /* made it this far; must have gone ok */
   return(OK);
}

static do_display_cmd(num_args,cmd_args)
     int num_args;
     char *cmd_args[];
     
     /* make a new display or select existing display */
     
{
  char buf[128];
  
  switch(cmd_args[1][1])
    {				/* make a new display */
    case 'c':
      if (num_args > 7)
	goto nerror;
      else
	{
	  char * name = NULL;
	  int xorig = -1, yorig = -1, width = -1, height = -1;
	  
	  if (num_args > 6)
	    if (sscanf(cmd_args[6], "%d", &height) != 1)
	      goto derror;
	  if (num_args > 5)
	    if (sscanf(cmd_args[5], "%d", &width) != 1)
	      goto derror;
	  if (num_args > 4)
	    if (sscanf(cmd_args[4], "%d", &yorig) != 1)
	      goto derror;
	  if (num_args > 3)
	    if (sscanf(cmd_args[3], "%d", &xorig) != 1)
	      goto derror;
	  if (num_args > 1)
	    name = cmd_args[2];	  
	  
	  gi_make_display(name, xorig, yorig, width, height);
	  break;
	}
    case 'm':
      {				/* select a mode for a display */
	int mode_switch = MODE_MAIN;
	
	if (num_args < 3 || num_args > 4)
	  goto nerror;
	if (num_args == 4)
	  if (sscanf(cmd_args[3], "%d", &mode_switch) != 1)
	    goto derror;
	if (gi_select_display_mode(cmd_args[2], mode_switch) == ERROR)
	  {
	    (void) sprintf(buf, "Select: no such display named %s", cmd_args[1]);
	    gi_put_error(buf);
	    return(ERROR);
	  }
	break;
      }
    case 's':
      {				/* select a display */
	int mode_switch = False;
	
	if (num_args < 3 || num_args > 4)
	  goto nerror;
	if (num_args == 4)
	  if (sscanf(cmd_args[3], "%d", &mode_switch) != 1)
	    goto derror;
	if (gi_select_display_panel(cmd_args[2], mode_switch) == ERROR)
	  {
	    (void) sprintf(buf, "Select: no such display named %s", cmd_args[1]);
	    gi_put_error(buf);
	    return(ERROR);
	  }
	break;
      }
    case 'q':
      if (num_args != 3)
	goto nerror;
      if (gi_close_display_name(cmd_args[2]) == ERROR)
	{
	  char buf[128];
	  (void) sprintf(buf, "Close: no such display named %s", cmd_args[1]);
	  gi_put_error(buf);
	  return(ERROR);
	}
      break;
    case 'n':
      if (num_args != 4)
	goto nerror;
      if (gi_rename_display_name(cmd_args[2],cmd_args[3]) == ERROR)
	{
	  char buf[128];
	  (void) sprintf(buf, "Rename: no such display named %s", cmd_args[1]);
	  gi_put_error(buf);
	  return(ERROR);
	}
      break;
    case 'e':
      {
	int i;
	
	if (num_args != 3)
	  goto nerror;
	if (sscanf(cmd_args[2], "%d", &i) != 1)
	  goto derror;
	gi_echo_command(i);
      }
      break;
    case 'f':
      {
	int i;
	
	if (num_args != 4)
	  goto nerror;
	if (sscanf(cmd_args[3], "%d", &i) != 1)
	  goto derror;
	if (gi_freeze_display_name(cmd_args[2],i) == ERROR)
	  {
	    char buf[128];
	    (void) sprintf(buf, "Freeze: no such display named %s", cmd_args[1]);
	    gi_put_error(buf);
	    return(ERROR);
	  }
      }
      break;
    case 'r':
      if (num_args != 3)
	goto nerror;
      if (strcmp(cmd_args[2], "all"))
	if (gi_reshow_display_name(cmd_args[2]) == ERROR)
	  {
	    char buf[128];
	    (void) sprintf(buf, "Reshow: no such display named %s", cmd_args[1]);
	    gi_put_error(buf);
	    return(ERROR);
	  }
	else
	  break;
      else	  
	gi_reshow_all_displays();
      break;  
    default:
      goto derror;
    }

  return(OK);

 derror:
  gi_put_error("Error in display command");
  return(ERROR);
  
 nerror:
  gi_put_error("Wrong number of arguments in display command");
  return(ERROR);
}

				/*  */
/*************************************************************************/
static route_gicmd(numargs,args)
  int numargs;
  char *args[];
{
/* based on the 2nd argument in args array, this routes
 * control to the correct procedure
 * for execution
 */

  int ret_code;

  /* check that there is a command */
  if (args[1]==NULL)
  {
    gi_put_error("Unspecified gi command");
    ret_code = ERROR;
  }
  else
  {
    switch(args[1][0])
    {
      case SHOW_CMD_CHAR:    ret_code = gi_do_show(numargs,args); 
                             break;
      case ERASE_CMD_CHAR:   ret_code = gi_do_erase(numargs,args);  
                             break;
      case CHANGE_CMD_CHAR:  ret_code = gi_do_change(numargs,args);  
                             break;
      case RESHOW_CMD_CHAR:  ret_code = do_reshow(numargs,args);  
                             break;
      case GO_CMD_CHAR:      ret_code = do_go(numargs,args);  
                             break;
      case TEXT_CMD_CHAR:    ret_code = do_text(numargs,args);  
                             break;
      case DRAW_CMD_CHAR:    ret_code = do_draw(numargs,args);  
                             break;
      case MOVE_CMD_CHAR:    ret_code = do_move(numargs,args);  
                             break;
      case DEL_CMD_CHAR:     ret_code = do_delete(numargs,args);  
                             break;
      case BUT_CMD_CHAR:     ret_code = do_set_buttons(numargs,args);  
                             break;
      case INFO_CMD_CHAR:    ret_code = do_unitinfo(numargs,args);  
                             break;
      case LOG_CMD_CHAR:     ret_code = do_logcmd(numargs,args);  
                             break;
      case DISPLAY_CMD_CHAR: ret_code = do_display_cmd(numargs,args);
	                     break;
      default:
        gi_put_error("Unrecognized gi command");
        ret_code = ERROR;
    }
  }
  return(ret_code);
}

/*************************************************************************/
static char *read_line(fp)
  FILE *fp;
{
/* reads and returns a pointer to the next line (excluding linefeed)
 * of the requested (opened) file or NULL if EOF
 * note: this routine also "swallows" and ignores blank lines as
 *       well as lines beginning with # (a comment)
 */

  static char line[MAX_CMD_LENGTH+1];
  int i=0, done=FALSE, init=TRUE;
  int c;

  /* read until the first non-blank, non-eol, non-comment is found */
  while (init)
  {
    c = getc(fp);

    /* ignore EOLs and leading blanks */
    if (c!=BLANK && c!='\n' && c!=COMMENT_CHAR)
      init = FALSE;
    
    /* eat up and ignore comment line */
    else if (c==COMMENT_CHAR)
    {
      while (c!='\n')
      {
	c = getc(fp);
        if (c==EOF)
        {
	  init = FALSE;
       	  break;
	}
      }
    }
  }

  /* at this point c contains either first real character
     of the command line or EOF */
	  
  /* read characters until end-of-line or end-of-file */
  while (!done)
  {
    /* if not EOF or EOL or too long, add next char to line buffer */
    if (c!=EOF && c!='\n' && i < MAX_CMD_LENGTH) 
    {
      line[i++] = c;
      c = getc(fp);
    }
    else 
    {
      /* terminate string with '\0' and done */      
      line[i] = '\0';
      done = TRUE;
      if (c!='\n' && c!=EOF) 
      {
        gi_put_error("Truncating excessively long command line");

        /* ignore rest of line */
        while (c != EOF && c != '\n') c = getc(fp);
      }
    }
  }
  /* if a positive line length return buffer else NULL */
  return(i ? line : NULL);
}

/*************************************************************************/
gi_do_restart_all(num_args,cmd_args)
  int num_args;
  char *cmd_args[];

{
  gi_close_all_display_panels();
  /* if 1st argument isn't "gi", send to simulator as well */
  if ( cmd_args != NULL && (num_args!=2 || strcmp(cmd_args[1],GI_CMD)))
    gi_sim_read_cmd(cmd_args[0]);
}

gi_do_restart(num_args,cmd_args)
  int num_args;
  char *cmd_args[];
{
/* Completely reinitializes the current display panel
 * by erasing all graphic, text and drawn object, 
 * moves the marker to its initial position, hides it,
 * blanks out the info screen and
 * reshows the screen at coordinates 0, 0
 */

  info_panel_t ip, tip;
  int ret_code = OK;
  struct txobj *tptr;
  struct drobj *dptr;
  struct grobj *gptr;

  /* set the reshow_flag to clear and redisplay everything */
  gi_reshow_flag = CLEAR_NEEDED + RESHOW_ALL;

   /* delete all text objects */
  for (tptr=gi_text_head.next; tptr!=&gi_text_head; tptr=tptr->next)
  {
    gi_erase_txobj(tptr);
  }

   /* delete all drawn objects */
  for (dptr=gi_draw_head.next; dptr!=&gi_draw_head; dptr=dptr->next)
  {
    gi_erase_drobj(dptr);
  }

  /* delete all unit objects */
  for (gptr=gi_marker.next; gptr!=&gi_marker; gptr=gptr->next)
  {
    gi_erase_grobj(gptr);
  }
  for (gptr=gi_off_display.next; gptr!=&gi_off_display; gptr=gptr->next)
  {
    gi_erase_grobj(gptr);
  }

  /* reinitialize marker */
  gi_marker.flag |= NOSHOW;
  gi_move_marker(5, 5);

  /* destroy all info panel widgets */
  if (gi_info_panel_head != NULL)
    gi_clear_info_list(gi_info_panel_head);
  
  /* now call reshow to clear screen and redisplay */
  gi_command("gi r 0 0");

  return(ret_code);
}
 
/*************************************************************************/
gi_do_quit(num_args,cmd_args)
  int num_args;
  char *cmd_args[];
{
/* return to caller
 */

  if (strcmp(cmd_args[0], QUIT_CMD_LONG))
    {
      gi_put_error("use `quit' or `return' to exit to caller");
      return(ERROR);
    }
  gi_back_to_caller();
}
 
/*************************************************************************/
static process_read(num_args,cmd_args)
  int num_args;
  char *cmd_args[];
{
/* process read command (possibly recursively) by reading
 * and executing all commands in the file (including possibly
 * other read commands) thus it uses a stack of file
 * descriptors to represent the possible stack of
 * files we could be reading from
 * assumes that the current read command has been
 * parsed into cmd_args
 */

  static FILE *readfile[MAXFILES];
  static int cur_file = -1;
  char read_error[MAX_CMD_LENGTH];
  char *next_cmd;
  int ret_code = OK;
  int log_status;

  /* check number of arguments */
  if (num_args!=2)
  {
    gi_put_error("Wrong number of args in read command");
    return(ERROR);
  }

  /* bump up index into stack of file descriptors */
  if (++cur_file >= MAXFILES)
  {
    gi_put_error("Too many nested reads : ignoring read command");
    cur_file--;
    return(ERROR);
  }

  /* attempt to open this requested file */
  if ((readfile[cur_file]=fopen(cmd_args[1],"r"))==NULL)
  {
    if (strlen(cmd_args[1]) <= 60)
      (void) sprintf(read_error,"can't open file %s",cmd_args[2]);
    else
      strcpy(read_error,"bad input file name (too long) on read");
    gi_put_error(read_error);
    cur_file--;
    return(ERROR);
  } 

  /* read and execute commands from this file until EOF */
  log_status = gi_log_on;
  gi_log_on = FALSE;
  while ((next_cmd=read_line(readfile[cur_file]))!=NULL)
  {
    PROCESS_EVENTS	/* process mouse clicks */
    if (gi_command(next_cmd)==ERROR)
      ret_code = ERROR;
  }
  gi_log_on = log_status;

  /* close the file and pop the stack of file pointers */
  fclose(readfile[cur_file--]);

  return(ret_code);
} 
  

/*************************************************************************/
gi_command(cmd)
  char *cmd;
{
/* given a character string containing a command
 * parses the command into cmd_args and calls the
 * appropriate routine to execute that command
 * note: this routine may be called recursively via
 *       execution of the "read" command 
 *       or through user executable code
 * returns OK (0) if no error encountered, else returns ERROR (-1)
 */
  int num_args, ret_code;
  char arg_buf[MAX_CMD_LENGTH+1];
  char *args[MAX_ARGS];

  /* check for a null command */
  if ((num_args=gi_parse_cmd(cmd,args,arg_buf))==0)
  {
    ret_code = ERROR;
  }

  else if (!strcmp(args[0],GI_CMD))
  {
    /* this is a gi command */
    ret_code = route_gicmd(num_args,args);
  }

  else if (!strcmp(args[0],READ_CMD_SHORT)
  || !strcmp(args[0],READ_CMD_LONG))
  {
    /* read command : intercept and process ourselves */
    ret_code = process_read(num_args,args);
  }

  else if (!strcmp(args[0],RESTART_CMD))
  {
    /* restart command: process ourselves */
    ret_code = gi_do_restart_all(num_args,args);
  }

  else if ((!strcmp(args[0],QUIT_CMD_SHORT)) || (!strcmp(args[0],QUIT_CMD_LONG)))
  {
    /* quit command: process ourselves */
    ret_code = gi_do_quit(num_args,args);
  }

  else /* this is not a gi command : send it to simulator */
  {
    char sim_cmd[MAX_CMD_LENGTH+1];

    strcpy (sim_cmd, cmd);	/* simulator strips null terminator off string */
    ret_code = gi_sim_read_cmd(sim_cmd);
    /* assume that everything and anything could change
       based on a command sent to the simulator
    */
    gi_reshow_flag |= ALL_VALS_NEEDED + ALL_LINKS_NEEDED;
    gi_off_display_ok = FALSE;
  }

  /* display this command on the line above the command line */
  gi_show_prev(cmd);
  gi_log(cmd);

  return(ret_code);
}

gi_show_prev(cmd)
     char * cmd;

{
  XtTextBlock text;
  int inpos;
  char nlcmd[MAX_CMD_LENGTH+1];

  if (gi_echo_on)
    {
      if (cmd[strlen(cmd)-1] != '\n') /* add newline if it isn't there */
	{
	(void) sprintf(nlcmd, "%s\n", cmd);
	cmd = nlcmd;
	}
      text.firstPos = 0;
      text.length = strlen(cmd);
      text.ptr = cmd;
      text.format = FMT8BIT;
      
      inpos =  XtTextGetInsertionPoint(gi_cmd_prev_item);
      XtTextReplace(gi_cmd_prev_item, inpos, inpos, &text);
    }
}
