;;; Functions related to adaptive filtering along directions of dominant orientation.
;;; HISTORY:  Sept. 22, 1989  Freeman most recently modified.  Added energy-filter function
;;;           Nov. 22, 1989  Freeman changed the rotation angle convention of  orientation-map.
;;;                          Older programs will have to be modified.  Added get-fourier-coeffs
;;;                          function.
;;;           Dec. 10, 1989  Freeman added  normalized  option to plot-orientation.  And tested
;;;                          that dc-oriented-energy gives the correct average of square-magnitude
;;;                          over all angles, for G2 H2 qsb's, anyway.
;;;           March  26, 1990 Freeman changed "get-fourier-coeffs" to "get-energy-fourier-coeffs".
;;;           August 8, 1990  Freeman added  get-energy-from-fourier-coeffs
;;;                           and get-all-energy-coeffs.  These make an image sequence
;;;                           of the fourier coefficients in the expansion for oriented
;;;                           energy.  I added a method on plot-orientation  to take
;;;                           such an image sequence of coefficients and make a floret
;;;                           plot out of that.
;;;          Nov. 15, 1991    Freeman added order 3 (non-separable) to the method orientation map.
;l;;                          So now you can find orientation from the steerable pyramid outputs.
;;;                           Made slight modification to the function calling of order 2,
;;;                           which has no effect on the result for that case.
;;;         March 27, 1992    Freeman added ":power" optional flag to plot-orientation functions.
;;;                           If it is nil, nothing changes.  Otherwise, raise oriented energy to
;;;                           that power before you plot it.

(in-package 'obvius)
(export '(orientation-map plot-orientation  plot-orientation-dots dc-oriented-energy 
	  get-energy-fourier-coeffs
	  get-all-energy-coeffs get-energy-from-fourier-coeffs))


;;; the oriented energy as a function of angle is a fourier series in angle.
;;; make an image pair of the cos and sin coefficients for the angular frequency   freq  .
;;;  eg, (get-energy-fourier-coeffs qsb 2)  gives you an image pair which are 
;;;  the cos 2theta and  sin 2theta
;;;  coefficients in the expansion.
(defmethod get-energy-fourier-coeffs ((qsb quadrature-steerable-basis) freq &key ->)
    (let ((order (order qsb)))
      (cond ((= order 2)
	     (cond ((= freq 0)
		    (dc-oriented-energy qsb :-> ->))
		   ((= freq 2)
		    (orientation-map qsb :-> ->))
		   ((= freq 4)
		    (get-energy-fourier-coeffs-g2h2-4 qsb :-> ->))
		   ((= freq 6)
		    (get-energy-fourier-coeffs-g2h2-6 qsb :-> ->)))))))

(defun get-energy-fourier-coeffs-g2h2-4 (qsb &key ->)
  (with-result-image-pair ((result ->) (dimensions qsb))
    (with-local-images ((a (square (nth 0 (image-list (even-steerable-basis qsb)))))
			(b (square (nth 2 (image-list (even-steerable-basis qsb)))))
			(c (add a b))
			(cospart (make-image (dimensions qsb)))
			(sinpart (make-image (dimensions qsb)))
			(total (mul c 0.125)))  
      (square (nth 0 (image-list (odd-steerable-basis qsb))) :-> a)
      (square (nth 3 (image-list (odd-steerable-basis qsb))) :-> b)
      (add a b :-> c)
      (mul c 0.1875 :-> a)
      (add a total :-> total)
      (mul (nth 0 (image-list (odd-steerable-basis qsb)))
	   (nth 2 (image-list (odd-steerable-basis qsb))) :-> a)
      (mul (nth 1 (image-list (odd-steerable-basis qsb)))
	   (nth 3 (image-list (odd-steerable-basis qsb))) :-> b)
      (add a b :-> c)
      (mul c -0.375 :-> a)
      (add a total :-> total)
      (square (nth 1 (image-list (odd-steerable-basis qsb))) :-> a)
      (square (nth 2 (image-list (odd-steerable-basis qsb))) :-> b)
      (add a b :-> c)
      (mul c -0.5625 :-> a)
      (add a total :-> total)
      (mul (nth 0 (image-list (even-steerable-basis qsb)))
	   (nth 2 (image-list (even-steerable-basis qsb))) :-> a)
      (mul a -0.25 :-> a)
      (add a total :-> total)
      (square (nth 1 (image-list (even-steerable-basis qsb))) :-> a)
      (mul a -0.5 :-> a)
      (add a total :-> cospart)

      (mul (nth 0 (image-list (even-steerable-basis qsb)))
	   (nth 1 (image-list (even-steerable-basis qsb))) :-> a)
      (mul (nth 1 (image-list (even-steerable-basis qsb)))
	   (nth 2 (image-list (even-steerable-basis qsb))) :-> b)
      (sub b a :-> c)
      (mul c 0.5 :-> total)
      (mul (nth 0 (image-list (odd-steerable-basis qsb)))
	   (nth 1 (image-list (odd-steerable-basis qsb))) :-> a)
      (mul (nth 2 (image-list (odd-steerable-basis qsb)))
	   (nth 3 (image-list (odd-steerable-basis qsb))) :-> b)
      (sub b a :-> c)
      (mul c 0.75 :-> a)
      (add a total :-> sinpart)
      (copy sinpart :-> (y-component result))
      (copy cospart :-> (x-component result)))))



(defun get-energy-fourier-coeffs-g2h2-6 (qsb &key ->)
  (with-result-image-pair ((result ->) (dimensions qsb))
    (with-local-images ((a (square (nth 1 (image-list (odd-steerable-basis qsb)))))
			(b (square (nth 2 (image-list (odd-steerable-basis qsb)))))
			(c (sub b a))
			(cospart (make-image (dimensions qsb)))
			(sinpart (make-image (dimensions qsb)))
			(total (mul c 0.28125)))  
      (square (nth 0 (image-list (odd-steerable-basis qsb))) :-> a)
      (square (nth 3 (image-list (odd-steerable-basis qsb))) :-> b)
      (sub a b :-> c)
      (mul c 0.03125 :-> a)
      (add a total :-> total)
      (mul (nth 0 (image-list (odd-steerable-basis qsb)))
	   (nth 2 (image-list (odd-steerable-basis qsb))) :-> a)
      (mul (nth 1 (image-list (odd-steerable-basis qsb)))
	   (nth 3 (image-list (odd-steerable-basis qsb))) :-> b)
      (sub b a :-> c)
      (mul c 0.1875 :-> a)
      (add a total :-> cospart)

      (mul (nth 0 (image-list (odd-steerable-basis qsb)))
	   (nth 1 (image-list (odd-steerable-basis qsb))) :-> a)
      (mul (nth 2 (image-list (odd-steerable-basis qsb)))
	   (nth 3 (image-list (odd-steerable-basis qsb))) :-> b)
      (add a  b :-> c)
      (mul c -0.1875 :-> total)
      (mul (nth 0 (image-list (odd-steerable-basis qsb)))
	   (nth 3 (image-list (odd-steerable-basis qsb))) :-> a)
      (mul a 0.0625 :-> a)
      (add a total :-> total)
      (mul (nth 1 (image-list (odd-steerable-basis qsb)))
	   (nth 2 (image-list (odd-steerable-basis qsb))) :-> b)
      (mul b 0.5625 :-> a)
      (add a total :-> sinpart)
      (copy sinpart :-> (y-component result))
      (copy cospart :-> (x-component result)))))


;;; load up an image sequence with all the fourier coefficients for a particular
;;; quadrature filter pair.   Loads into an image sequence.  Use this prior
;;; to using  get-energy-from-fourier-coeffs.
;;; The ordering of the fourier coefficients in the output image sequence:
;;; 0      1           2             3          4            5            6
;;; dc  sin 2theta  cos 2theta   sin 4theta  cos 4theta   sin 6theta   cos 6theta
;;; (if this is a qsb, then you need to get coefficients up to a higher order
;;; than you do if this is just a steerable basis.)
(defmethod get-all-energy-coeffs ((qsb quadrature-steerable-basis) &key ->)
  (with-result-image-sequence ((result ->) (append (list (+ 3 (* 2 (order qsb)))) (dimensions qsb)))
    (get-energy-fourier-coeffs qsb 0 :-> (frame 0 result))
    (with-local-images ((im-pair (make-image-pair (dimensions qsb))))
      (dotimes (i (1+ (order qsb)))
	(get-energy-fourier-coeffs qsb (* 2 (1+ i)) :-> im-pair)
	(copy (y-component im-pair) :-> (frame (1- (* 2 (1+ i))) result))
	(copy (x-component im-pair) :-> (frame (* 2 (1+ i)) result))))))


;;; outputs energy, given the energy as a function of angle fourier series,
;;; the angle cannot be an image (for now), but it must be a scalar.
(defun get-energy-from-fourier-coeffs (sequence theta &key -> )
  (with-result-image ((result -> ) (dimensions sequence))
    (with-local-images ((cos-part (make-image (dimensions sequence)))
			(sin-part (make-image (dimensions sequence))))
      (copy (frame 0 sequence) :-> result)
      (let ((j 1))
	(dotimes (i (/ (1- (sequence-length sequence)) 2))  ;;; i goes 0, 1, 2
	  (setq j (* 2 (1+ i)))                             ;;; j goes 2, 4, 6
	  (mul (frame j sequence) (cos (* j theta)) :-> cos-part)
	  (mul (frame (1- j) sequence) (sin (* j theta)) :-> sin-part)
	  (add cos-part result :-> result)
	  (add sin-part result :-> result))))))



;;; Finds the dominant orientation, given a quadrature steerable basis.
;;; Defined for quadrature steerable bases of orders 2 and 3 (separable 
;;; and non-separable, respectively). 
(defmethod orientation-map ((qsb quadrature-steerable-basis) &key ->)
  (with-result-image-pair ((result ->) (dimensions qsb))
    (let ((order (order qsb)))
      (cond ((and (= order 2) (separable-steerable-basis-p (even-steerable-basis qsb)))
	     (get-ori-g2h2-separable qsb :-> result))
	    ((and (= order 3) (not (separable-steerable-basis-p (even-steerable-basis qsb))))
	     (get-ori-g3h3-nonseparable qsb :-> result))))))

	

;;; get the lowest order terms in the fourier expansion of orientation energy
;;; as a function of angle for a non-separable g3h3 basis set (eg, for steerable
;;; pyramid filters).
(defmethod get-ori-g3h3-nonseparable ((qsb quadrature-steerable-basis) &key ->)
  (with-result-image-pair ((result ->) (dimensions qsb))
    (with-local-images ((a (square (nth 0 (image-list (odd-steerable-basis qsb)))))
			(b (square (nth 2 (image-list (odd-steerable-basis qsb)))))
			(cospart (make-image (dimensions qsb)))
			(sinpart (make-image (dimensions qsb))))
;;; + 0.375*g0^2 - 0.375*g2^2 
      (sub a b :-> a)
      (mul a 0.375 :-> (x-component result))

;;; + 0.17678*g0*g1 - 0.17678*g1*g2 - 0.17678*g0*g3 - 0.17678*g2*g3 
      (sub (nth 0 (image-list (odd-steerable-basis qsb)))
	   (nth 2 (image-list (odd-steerable-basis qsb))) :-> a)
      (add (nth 0 (image-list (odd-steerable-basis qsb)))
	   (nth 2 (image-list (odd-steerable-basis qsb))) :-> b)
      (mul a (nth 1 (image-list (odd-steerable-basis qsb))) :-> a)
      (mul b (nth 3 (image-list (odd-steerable-basis qsb))) :-> b)
      (sub a b :-> a)
      (mul a 0.17678 :-> a)
      (add a (x-component result) :-> (x-component result))

;;; + 0.25*g1*g3 
      (mul (nth 1 (image-list (odd-steerable-basis qsb)))
	   (nth 3 (image-list (odd-steerable-basis qsb))) :-> a)
      (mul a 0.25 :-> a)
      (add a (x-component result) :-> (x-component result))

;;; + 0.32*h0^2 
      (square (nth 0 (image-list (even-steerable-basis qsb))) :-> a)
      (mul a 0.32 :-> a)
      (add a (x-component result) :-> (x-component result))

;;; - 0.16*h1*h4 
      (mul (nth 1 (image-list (even-steerable-basis qsb)))
	   (nth 4 (image-list (even-steerable-basis qsb))) :-> a)
      (mul a -0.16 :-> a)
      (add a (x-component result) :-> (x-component result))

;;; + 0.098885*h1^2 + 0.098885*h4^2
      (square (nth 1 (image-list (even-steerable-basis qsb))) :-> a)
      (square (nth 4 (image-list (even-steerable-basis qsb))) :-> b)
      (add a b :-> a)
      (mul a 0.098885 :-> a)
      (add a (x-component result) :-> (x-component result))

;;; -0.0494422*h0*h2 - 0.049442*h1*h2 - 0.049442*h3*h4 - 0.049442*h0*h3 
      (add (nth 0 (image-list (even-steerable-basis qsb)))
	   (nth 1 (image-list (even-steerable-basis qsb))) :-> a)
      (add (nth 0 (image-list (even-steerable-basis qsb)))
	   (nth 4 (image-list (even-steerable-basis qsb))) :-> b)
      (mul a (nth 2 (image-list (even-steerable-basis qsb))) :-> a)
      (mul b (nth 3 (image-list (even-steerable-basis qsb))) :-> b)
      (add a b :-> a)
      (mul a -0.0494422 :-> a)
      (add a (x-component result) :-> (x-component result))

;;; - 0.159999*h2*h3 
      (mul (nth 2 (image-list (even-steerable-basis qsb)))
	   (nth 3 (image-list (even-steerable-basis qsb))) :-> a)
      (mul a -0.159999 :-> a)
      (add a (x-component result) :-> (x-component result))

;;; -0.258885*h2^2 - 0.258885*h3^2 
      (square (nth 2 (image-list (even-steerable-basis qsb))) :-> a)
      (square (nth 3 (image-list (even-steerable-basis qsb))) :-> b)
      (add a b :-> a)
      (mul a -0.258885 :-> a)
      (add a (x-component result) :-> (x-component result))

;;; + 0.129442*h1*h3 + 0.129442*h0*h1 + 0.129442*h2*h4 + 0.129442*h0*h4 
      (add (nth 0 (image-list (even-steerable-basis qsb)))
	   (nth 3 (image-list (even-steerable-basis qsb))) :-> a)
      (add (nth 0 (image-list (even-steerable-basis qsb)))
	   (nth 2 (image-list (even-steerable-basis qsb))) :-> b)
      (mul a (nth 1 (image-list (even-steerable-basis qsb))) :-> a)
      (mul b (nth 4 (image-list (even-steerable-basis qsb))) :-> b)
      (add a b :-> a)
      (mul a 0.129442 :-> a)
      (add a (x-component result) :-> (x-component result))

;;; + 0.375*g1^2 - 0.375*g3^2 
      (square (nth 1 (image-list (odd-steerable-basis qsb))) :-> a)
      (square (nth 3 (image-list (odd-steerable-basis qsb))) :-> b)
      (sub a b :-> a)
      (mul a 0.375 :-> (y-component result))

;;; - 0.25*g0*g2 
      (mul (nth 0 (image-list (odd-steerable-basis qsb)))
	   (nth 2 (image-list (odd-steerable-basis qsb))) :-> a)
      (mul a -0.25 :-> a)
      (add a (y-component result) :-> (y-component result))

;;; + 0.17678*g0*g1 + 0.17678*g1*g2 + 0.17678*g0*g3 - 0.17678*g2*g3 
      (add (nth 0 (image-list (odd-steerable-basis qsb)))
	   (nth 2 (image-list (odd-steerable-basis qsb))) :-> a)
      (sub (nth 0 (image-list (odd-steerable-basis qsb)))
	   (nth 2 (image-list (odd-steerable-basis qsb))) :-> b)
      (mul a (nth 1 (image-list (odd-steerable-basis qsb))) :-> a)
      (mul b (nth 3 (image-list (odd-steerable-basis qsb))) :-> b)
      (add a b :-> a)
      (mul a 0.17678 :-> a)
      (add a (y-component result) :-> (y-component result))

;;; + 0.304338*h1^2 - 0.304338*h4^2
      (square (nth 1 (image-list (even-steerable-basis qsb))) :-> a)
      (square (nth 4 (image-list (even-steerable-basis qsb))) :-> b)
      (sub a b :-> a)
      (mul a 0.304338 :-> a)
      (add a (y-component result) :-> (y-component result))

;;; - 0.152169*h0*h2 + 0.152169*h1*h2 - 0.152169*h3*h4 + 0.152169*h0*h3 
      (sub (nth 1 (image-list (even-steerable-basis qsb)))
	   (nth 0 (image-list (even-steerable-basis qsb))) :-> a)
      (sub (nth 0 (image-list (even-steerable-basis qsb)))
	   (nth 4 (image-list (even-steerable-basis qsb))) :-> b)
      (mul a (nth 2 (image-list (even-steerable-basis qsb))) :-> a)
      (mul b (nth 3 (image-list (even-steerable-basis qsb))) :-> b)
      (add a b :-> a)
      (mul a 0.152169 :-> a)
      (add a (y-component result) :-> (y-component result))

;;; - 0.188091*h3^2 + 0.188091*h2^2 
      (square (nth 2 (image-list (even-steerable-basis qsb))) :-> a)
      (square (nth 3 (image-list (even-steerable-basis qsb))) :-> b)
      (sub a b :-> a)
      (mul a 0.188091 :-> a)
      (add a (y-component result) :-> (y-component result))

;;; + 0.094045*h0*h1 - 0.094045*h1*h3 - 0.094045*h0*h4 + 0.094045*h2*h4 
      (sub (nth 0 (image-list (even-steerable-basis qsb)))
	   (nth 3 (image-list (even-steerable-basis qsb))) :-> a)
      (sub (nth 2 (image-list (even-steerable-basis qsb)))
	   (nth 0 (image-list (even-steerable-basis qsb))) :-> b)
      (mul a (nth 1 (image-list (even-steerable-basis qsb))) :-> a)
      (mul b (nth 4 (image-list (even-steerable-basis qsb))) :-> b)
      (add a b :-> a)
      (mul a 0.094045 :-> a)
      (add a (y-component result) :-> (y-component result)))))




;;; get the lowest order terms in the fourier expansion of orientation energy
;;; as a function of angle for a separable g2h2 basis set.
(defmethod get-ori-g2h2-separable ((qsb quadrature-steerable-basis) &key ->)
  (with-result-image-pair ((result ->) (dimensions qsb))
    (with-local-images ((a (square (nth 0 (image-list (even-steerable-basis qsb)))))
			(b (square (nth 2 (image-list (even-steerable-basis qsb)))))
			(c (sub a b))
			(cospart (make-image (dimensions qsb)))
			(sinpart (make-image (dimensions qsb)))
			(total (mul c .5)))  
      (square (nth 0 (image-list (odd-steerable-basis qsb))) :-> a)
      (square (nth 3 (image-list (odd-steerable-basis qsb))) :-> b)
      (sub a b :-> c)
      (mul c 0.46875 :-> a)
      (add a total :-> total)
      (mul (nth 0 (image-list (odd-steerable-basis qsb)))
	   (nth 2 (image-list (odd-steerable-basis qsb))) :-> a)
      (mul (nth 1 (image-list (odd-steerable-basis qsb)))
	   (nth 3 (image-list (odd-steerable-basis qsb))) :-> b)
      (sub a b :-> c)
      (mul c 0.1875 :-> a)
      (add a total :-> total)
      (square (nth 1 (image-list (odd-steerable-basis qsb))) :-> a)
      (square (nth 2 (image-list (odd-steerable-basis qsb))) :-> b)
      (sub a b :-> c)
      (mul c 0.28125 :-> a)
      (add a total :-> cospart)

      (mul (nth 0 (image-list (even-steerable-basis qsb)))
	   (nth 1 (image-list (even-steerable-basis qsb))) :-> a)
      (mul (nth 1 (image-list (even-steerable-basis qsb)))
	   (nth 2 (image-list (even-steerable-basis qsb))) :-> b)
      (add a b :-> c)
      (mul c 1.0 :-> total)
      (mul (nth 0 (image-list (odd-steerable-basis qsb)))
	   (nth 1 (image-list (odd-steerable-basis qsb))) :-> a)
      (mul (nth 2 (image-list (odd-steerable-basis qsb)))
	   (nth 3 (image-list (odd-steerable-basis qsb))) :-> b)
      (add a b :-> c)
      (mul c 0.9375 :-> a)
      (add a total :-> total)
      (mul (nth 1 (image-list (odd-steerable-basis qsb)))
	   (nth 2 (image-list (odd-steerable-basis qsb))) :-> a)
      (mul a 1.6875 :-> c)
      (add c total :-> total)
      (mul (nth 0 (image-list (odd-steerable-basis qsb)))
	   (nth 3 (image-list (odd-steerable-basis qsb))) :-> b)
      (mul b 0.1875 :-> a)
      (add a total :-> sinpart)
      (mul sinpart -1.0  :-> (y-component result))
      (copy cospart :-> (x-component result)))))


#|
;;; remember:  angle is measured relative to vertical for this stuff.
(setq al (gauss-out (load-image "/images/einstein")))
(setq qsb (make-quadrature-steerable-basis al))

(orientation-map qsb :-> 'einpair)
(complex-phase einpair :-> 'fangle)
(magnitude einpair :-> 'mag)
(div fangle 2.0 :-> 'angle)
(setq *auto-scale-vector-fields* nil)
(setq *vector-field-arrow-heads* nil)
(setq *vector-field-skip* 5)
(setq *vector-field-magnify* 8)
(setq *vector-field-lengthen* .3)
(square-root mag :-> 'sqrt-mag)
(clip sqrt-mag 0 68 :-> 'cl-sqrt-mag)
(make-image-pair 
 (list (mul (cos. angle) cl-sqrt-mag)
       (mul (sin. angle) cl-sqrt-mag))
 :display-type 'vector-field :-> 'pic)

|#



;;;  Make a field of polar plots at various positions in the image.
;;; Each single polar plot shows the oriented energy as a 
;;; function of angle, at a given position in the image.  The plots
;;; are made at many positions.
;;; Skip, magnify and lengthen play the same role as they do
;;; in the vector-field display of image pairs.
;;; If normalized = t, then normalize the length of the plots by the 
;;; dc-oriented energy.
;;;         March 27, 1992    Freeman added ":power" optional flag to plot-orientation functions.
;;;                           If it is nil, nothing changes.  Otherwise, raise oriented energy to
;;;                           that power before you plot it.
(defmethod plot-orientation ((qsb quadrature-steerable-basis)
			     &key
			     (skip 8)
			     (magnify 32)
			     (lengthen 2.7)
			     (n-angles 32)
			     (normalized nil)
			     (power nil)
			     ->)
  ;;; to set the dimensions of the image to be drawn into.
  ;;; image is the image of which the orientational energy will be found.
  (let ((out-dims (list (round (* magnify (ceiling (y-dim qsb) skip)))
			(round (* magnify (ceiling (x-dim qsb) skip))))))
    (with-result-image ((result ->) out-dims 'plot-orientation)
      (with-local-images ((previous-energy (make-image (dimensions qsb)))
			  (current-energy (make-image (dimensions qsb)))
			  (norm (make-image (dimensions qsb))))
	;; initialize previous energy
	(square-magnitude qsb 0.0 :-> previous-energy)
	;;; normalize the value, if necessary.
	(cond (normalized
	       (dc-oriented-energy qsb :-> norm)
	       (div previous-energy norm :-> previous-energy))
	      (power 
	       (pow previous-energy power :-> previous-energy)))
	;; loop over all angles, and draw lines from the current to the past
	;; oriented energies.
	(loop for i from 1 to n-angles 
	      for current-angle = (* 2 (print i) (/ pi n-angles))
	      for previous-angle = (* 2 (- i 1) (/ pi n-angles))
	      do
	      ;; get the appropriate oriented energy measure to the appropriate order.
	      (square-magnitude qsb current-angle :-> current-energy)
	      (if normalized (div current-energy norm :-> current-energy))
	      (if power (pow current-energy power :-> current-energy))
	      (draw-orient-vector previous-energy 
				  previous-angle
				  current-energy
				  current-angle
				  skip magnify lengthen out-dims :-> result)
	      (copy current-energy :-> previous-energy))))))

;;;  Same as above, except this method works on sequences which represent the fourier coefficients,
;;; instead of on quadrature steerable bases.
;;;         March 27, 1992    Freeman added ":power" optional flag to plot-orientation functions.
;;;                           If it is nil, nothing changes.  Otherwise, raise oriented energy to
;;;                           that power before you plot it.
(defmethod plot-orientation ((sequence image-sequence)
			     &key
			     (skip 8)
			     (magnify 32)
			     (lengthen 2.7)
			     (n-angles 32)
			     (normalized nil)
			     (power nil)
			     ->)
  ;;; to set the dimensions of the image to be drawn into.
  ;;; image is the image of which the orientational energy will be found.
  (let ((out-dims (list (round (* magnify (ceiling (y-dim sequence) skip)))
			(round (* magnify (ceiling (x-dim sequence) skip))))))
    (with-result-image ((result ->) out-dims 'plot-orientation)
      (with-local-images ((previous-energy (make-image (dimensions sequence)))
			  (current-energy (make-image (dimensions sequence)))
			  (norm (make-image (dimensions sequence))))
	;; initialize previous energy
	(get-energy-from-fourier-coeffs sequence 0.0 :-> previous-energy)
	;;; normalize the value, if necessary.
	(cond (normalized
	       (frame 0 sequence :-> norm)
	       (div previous-energy norm :-> previous-energy))
	      (power 
	       (pow previous-energy power :-> previous-energy)))
	;; loop over all angles, and draw lines from the current to the past
	;; oriented energies.
	(loop for i from 1 to n-angles 
	      for current-angle = (* 2 (print i) (/ pi n-angles))
	      for previous-angle = (* 2 (- i 1) (/ pi n-angles))
	      do
	      ;; get the appropriate oriented energy measure to the appropriate order.
	      (get-energy-from-fourier-coeffs sequence current-angle :-> current-energy)
	      (if normalized (div current-energy norm :-> current-energy))
	      (if power (pow current-energy power :-> current-energy))
	      (draw-orient-vector previous-energy 
				  previous-angle
				  current-energy
				  current-angle
				  skip magnify lengthen out-dims :-> result)
	      (copy current-energy :-> previous-energy))))))

;;; a dots-version of the image-sequence version of plot-orientation
;;;  Same as above, except this method works on sequences which represent the fourier coefficients,
;;; instead of on quadrature steerable bases.
;;;         March 27, 1992    Freeman added ":power" optional flag to plot-orientation functions.
;;;                           If it is nil, nothing changes.  Otherwise, raise oriented energy to
;;;                           that power before you plot it.
(defmethod plot-orientation-dots ((sequence image-sequence)
				  &key
				  (skip 8)
				  (magnify 32)
				  (lengthen 2.7)
				  (n-angles 32)
				  (normalized nil)
				  (power nil)
				  ->)
  ;;; to set the dimensions of the image to be drawn into.
  ;;; image is the image of which the orientational energy will be found.
  (let ((out-dims (list (round (* magnify (ceiling (y-dim sequence) skip)))
			(round (* magnify (ceiling (x-dim sequence) skip))))))
    (with-result-image ((result ->) out-dims 'plot-orientation-dots)
      (with-local-images ((current-energy (make-image (dimensions sequence))))
	;; loop over all angles, and draw lines from the current to the past
	;; oriented energies.
	(loop for i from 1 to n-angles 
	      for current-angle = (* 2 (print i) (/ pi n-angles))
	      do
	      ;; get the appropriate oriented energy measure to the appropriate order.
	      (get-energy-from-fourier-coeffs sequence current-angle :-> current-energy)
	      (if normalized (div current-energy (frame 0 sequence) :-> current-energy))
	      (if power (pow current-energy power :-> current-energy))
	      (put-dots current-energy current-angle skip magnify lengthen out-dims :-> result))))))




;;; energy1 is an image of oriented energy in the direction angle1 
;;; at every point.  Similarly for energy2 and angle2.
;;; This function draws a line segment from angle1 to angle2 
;;; on each of the many polar plots which are begin drawn simultaneously.
;;; Skip, magnify, lengthen have the same role as for vector field displays
;;; of image pairs.  out-dims in the dimensions of the output image.
(defun draw-orient-vector (energy1 angle1 energy2 angle2
				   skip magnify lengthen out-dims &key ->)
  (with-result-image ((result ->) out-dims 'draw-orient-vector)  
    (loop for y from 0 below (y-dim energy1) by skip
	  for yprime = (* magnify (/ y skip))
	  do
	  (loop for x from 0 below (x-dim energy1) by skip
		for xprime = (* magnify (/ x skip))
		for energy1-val = (iref energy1 y x)
		for energy2-val = (iref energy2 y x)
		;; angle = 0 is along the y axis, so the sines and coses are like this:
		;; (and increasing angle is counter-clockwise)
		for xdraw1 = (floor (+ xprime (* -1.0 (sin angle1) energy1-val lengthen)))
		for ydraw1 = (floor (+ yprime (* -1.0 (cos angle1) energy1-val lengthen)))
		for xdraw2 = (floor (+ xprime (* -1.0 (sin angle2) energy2-val lengthen)))
		for ydraw2 = (floor (+ yprime (* -1.0 (cos angle2) energy2-val lengthen)))
		do
		(when (and (> xdraw1 1) (> ydraw1 1) 
			   (< xdraw1 (- (x-dim result) 2)) (< ydraw1 (- (y-dim result) 2))
			   (> xdraw2 1) (> ydraw2 1) 
			   (< xdraw2 (- (x-dim result) 2)) (< ydraw2 (- (y-dim result) 2)))
		  (draw-line result ydraw1 xdraw1 ydraw2 xdraw2 :-> result))))))


;;; This is identical to plot-orientation, except faster because it plots dots instead
;;; of lines.
;;;  Make a field of polar dot plots at various positions in the image.
;;; Each single polar plot shows the oriented energy as a 
;;; function of angle, at a given position in the image.  The plots
;;; are made at many positions.
;;; Skip, magnify and lengthen play the same role as they do
;;; in the vector-field display of image pairs.
;;; If normalized = t, then normalize the length of the plots by the 
;;; dc-oriented energy.
(defmethod plot-orientation-dots ((qsb quadrature-steerable-basis)
				  &key
				  (skip 8)
				  (magnify 32)
				  (lengthen 2.7)
				  (n-angles 32)
				  (normalized nil)
				  ->)
  ;;; to set the dimensions of the image to be drawn into.
  ;;; image is the image of which the orientational energy will be found.
  (let ((out-dims (list (round (* magnify (ceiling (y-dim qsb) skip)))
			(round (* magnify (ceiling (x-dim qsb) skip))))))
    (with-result-image ((result ->) out-dims 'plot-orientation)
      (with-local-images ((current-energy (make-image (dimensions qsb)))
			  (norm (make-image (dimensions qsb))))
	(if normalized
	    (dc-oriented-energy qsb :-> norm))
	;; loop over all angles, and draw dots at the oriented energies.
	(loop for i from 1 to n-angles 
	      for current-angle = (* 2 (print i) (/ pi n-angles))
	      do
	      ;; get the appropriate oriented energy measure to the appropriate order.
	      (square-magnitude qsb current-angle :-> current-energy)
	      (if normalized (div current-energy norm :-> current-energy))
	      (put-dots current-energy current-angle skip magnify lengthen out-dims :-> result))))))

;;; energy1 is an image of oriented energy in the direction angle1 
;;; at every point.
;;; This function puts a point along angle1 out at length energy1
;;; at every point as specified by  skip, magnify  and lengthen.
;;; Skip, magnify, lengthen have the same role as for vector field displays
;;; of image pairs.  out-dims in the dimensions of the output image.
(defun put-dots (energy1 angle1 skip magnify lengthen out-dims &key (value 1.0) ->)
  (with-result-image ((result ->) out-dims 'put-dots)
    (loop for y from 0 below (y-dim energy1) by skip
	  for yprime = (* magnify (/ y skip))
	  do
	  (loop for x from 0 below (x-dim energy1) by skip
		for xprime = (* magnify (/ x skip))
		for energy1-val = (iref energy1 y x)
		;; angle = 0 is along the y axis, so the sines and coses are like this:
		;; (and increasing angle is counter-clockwise)
		for xdraw1 = (floor (+ xprime (* -1.0 (sin angle1) energy1-val lengthen)))
		for ydraw1 = (floor (+ yprime (* -1.0 (cos angle1) energy1-val lengthen)))
		do
		(when (and (> xdraw1 1) (> ydraw1 1) 
			   (< xdraw1 (- (x-dim result) 2)) (< ydraw1 (- (y-dim result) 2)))
		  (setf (iref result ydraw1 xdraw1) value))))))


#|
(obv-compile-load "draw-line")

(load-image "/images/freeman/steer/cropsmall-tex" :-> 'crop-tex)
(crop crop-tex 23 10 32 32 :-> 'tiny-tex)
(setq tex-qsb (make-quadrature-steerable-basis tiny-tex))
(plot-orientation tex-qsb :n-angles 16 :lengthen 0.015 :skip 1)

|#


;;; GET-DC-ORIENTED-ENERGY   get the dc component of the oriented energy,
;;; passing parameters in the same format as  get-oriented-energy below.
;;; Accomodates both g2h2  and g4h4  quadrature steerablee bases.
;;;
;;; Function derived from ~freeman/lisp/steer/steerables.lisp, function get-dc-oriented-energy.
;;; With that function, the results were checked for both g2h2 and g4h4, with perfect agreement
;;;   with a numerical integration.
;;;
;;; NOTE:  This version of g2h2 dc energy was tested by comparing results
;;;   with numerical average over all angles of square-magnitude.  The results
;;;   agreed to 6 places.  wtf- Dec. 11, 1989
;;; (also, it was tested rigorously in the original version
;;; in ~freeman/lisp/steer/steerables.lisp)  This g4h4 version hasn't been 
;;; tested at all, and it needs to be re-written to not use so many temporary
;;; images--to re-use some of them.

(defmethod dc-oriented-energy ((qsb quadrature-steerable-basis) &key (sq-root nil) ->)
  (with-result-image ((result ->) (dimensions qsb))
    (let ((order (1- (min (length (image-list (even-steerable-basis qsb)))
			  (length  (image-list (odd-steerable-basis qsb)))))))
      (cond ((equal order 2) 
	     (with-local-images (
				 (a (mul (nth 2 (image-list (even-steerable-basis qsb))) (nth 2 (image-list (even-steerable-basis qsb)))))
				 (b (mul (nth 0 (image-list (even-steerable-basis qsb))) (nth 0 (image-list (even-steerable-basis qsb)))))
				 (c (add a b))
				 (total (mul c 0.375)))  
	       (mul (nth 0 (image-list (odd-steerable-basis qsb))) (nth 0 (image-list (odd-steerable-basis qsb))) :-> a)
	       (mul (nth 3 (image-list (odd-steerable-basis qsb))) (nth 3 (image-list (odd-steerable-basis qsb))) :-> b)
	       (add a b :-> c)
	       (mul c 0.3125 :-> a)
	       (add a total :-> total)
	       (mul (nth 1 (image-list (odd-steerable-basis qsb))) (nth 1 (image-list (odd-steerable-basis qsb))) :-> a)
	       (mul (nth 2 (image-list (odd-steerable-basis qsb))) (nth 2 (image-list (odd-steerable-basis qsb))) :-> b)
	       (add a b :-> c)
	       (mul c 0.5625 :-> a)
	       (add a total :-> total)
	       (mul (nth 0 (image-list (odd-steerable-basis qsb))) (nth 2 (image-list (odd-steerable-basis qsb))) :-> a)
	       (mul (nth 1 (image-list (odd-steerable-basis qsb))) (nth 3 (image-list (odd-steerable-basis qsb))) :-> b)
	       (add a b :-> c)
	       (mul c 0.375 :-> a)
	       (add a total :-> total)
	       (mul (nth 1 (image-list (even-steerable-basis qsb))) (nth 1 (image-list (even-steerable-basis qsb))) :-> a)
	       (mul a 0.5 :-> b)
	       (add b total :-> total)
	       (mul (nth 0 (image-list (even-steerable-basis qsb))) (nth 2 (image-list (even-steerable-basis qsb))) :-> a)
	       (mul a 0.25 :-> b)
	       (add b total :-> result)
	       (if sq-root (square-root result :-> result))))
	    ((equal order 4) 
	     (with-local-images (
				 (g00 (mul (nth 0 (image-list (even-steerable-basis qsb))) (nth 0 (image-list (even-steerable-basis qsb)))))
				 (g44 (mul (nth 4 (image-list (even-steerable-basis qsb))) (nth 4 (image-list (even-steerable-basis qsb)))))
				 (g11 (mul (nth 1 (image-list (even-steerable-basis qsb))) (nth 1 (image-list (even-steerable-basis qsb)))))
				 (g33 (mul (nth 3 (image-list (even-steerable-basis qsb))) (nth 3 (image-list (even-steerable-basis qsb)))))
				 (g13 (mul (nth 1 (image-list (even-steerable-basis qsb))) (nth 3 (image-list (even-steerable-basis qsb)))))
				 (g04 (mul (nth 0 (image-list (even-steerable-basis qsb))) (nth 4 (image-list (even-steerable-basis qsb)))))
				 (g22 (mul (nth 2 (image-list (even-steerable-basis qsb))) (nth 2 (image-list (even-steerable-basis qsb)))))
				 (g24 (mul (nth 2 (image-list (even-steerable-basis qsb))) (nth 4 (image-list (even-steerable-basis qsb)))))
				 (g02 (mul (nth 0 (image-list (even-steerable-basis qsb))) (nth 2 (image-list (even-steerable-basis qsb)))))
				 (a (add g00 g44))
				 (a (mul a .424414))
				 (b (add g11 g33))
				 (b (mul b .970089))
				 (c (add g24 g02))
				 (c (mul c .727567))
				 (d (mul g22 1.30962))
				 (e (mul g13 1.16411))
				 (f (mul g04 .0727567))
				 (z (add a b))
				 (y (add c d))
				 (even-sq (add z y))
				 (z (add e f))
				 (even-sq (add even-sq z))
				 (h00 (mul (nth 0 (image-list (odd-steerable-basis qsb))) (nth 0 (image-list (odd-steerable-basis qsb)))))
				 (h55 (mul (nth 5 (image-list (odd-steerable-basis qsb))) (nth 5 (image-list (odd-steerable-basis qsb)))))
				 (h11 (mul (nth 1 (image-list (odd-steerable-basis qsb))) (nth 1 (image-list (odd-steerable-basis qsb)))))
				 (h44 (mul (nth 4 (image-list (odd-steerable-basis qsb))) (nth 4 (image-list (odd-steerable-basis qsb)))))
				 (h02 (mul (nth 0 (image-list (odd-steerable-basis qsb))) (nth 2 (image-list (odd-steerable-basis qsb)))))
				 (h35 (mul (nth 3 (image-list (odd-steerable-basis qsb))) (nth 5 (image-list (odd-steerable-basis qsb)))))
				 (h22 (mul (nth 2 (image-list (odd-steerable-basis qsb))) (nth 2 (image-list (odd-steerable-basis qsb)))))
				 (h33 (mul (nth 3 (image-list (odd-steerable-basis qsb))) (nth 3 (image-list (odd-steerable-basis qsb)))))
				 (h13 (mul (nth 1 (image-list (odd-steerable-basis qsb))) (nth 3 (image-list (odd-steerable-basis qsb)))))
				 (h04 (mul (nth 0 (image-list (odd-steerable-basis qsb))) (nth 4 (image-list (odd-steerable-basis qsb)))))
				 (h24 (mul (nth 2 (image-list (odd-steerable-basis qsb))) (nth 4 (image-list (odd-steerable-basis qsb)))))
				 (h15 (mul (nth 1 (image-list (odd-steerable-basis qsb))) (nth 5 (image-list (odd-steerable-basis qsb)))))
				 (a (add h00 h55))
				 (a (mul a .0388883))
				 (b (add h11 h44))
				 (b (mul b .108023))
				 (c (add h02 h35))
				 (c (mul c .0864184))
				 (d (add h04 h15))
				 (d (mul d .0185182))
				 (e (add h13 h22))
				 (z (add h24 h33))
				 (z (add e z))
				 (e (mul z .185182))
				 (z (add a b))
				 (y (add c d))
				 (odd-sq (add z y))
				 (odd-sq (add odd-sq e)))
	       (div even-sq (sqr 1.24585) :-> even-sq)
	       (div odd-sq (sqr .39752) :-> odd-sq)
	       (add even-sq odd-sq :-> result)
	       (if sq-root (square-root result :-> result))))))))


#|
(make-cos-zone-plate '(64 64) :-> 'zone)
(make-quadrature-steerable-basis zone :-> 'zoneqsb)

(dc-oriented-energy zoneqsb :-> 'orisq)
(dc-oriented-energy zoneqsb :sq-root t :-> 'ori)

;;; test the dc-oriented-energy gives the correct result, for the G2 H2 filters
(make-image '(32 32) :-> 'total)
(load-image "/visci/images/einstein/")
(gauss-out (gauss-out (gauss-out einstein)) :-> 'small)
(make-quadrature-steerable-basis small :-> 'small-qsb)
(dotimes (i 16)
  (add (square-magnitude small-qsb (* pi (/ i 16))) total :-> total))
(div total 16 :-> 'dc-exp)
(dc-oriented-energy small-qsb :-> 'dc-theo)
(sub dc-exp dc-theo)   ;;; accurate to about 6 places
|#


;;; filter the even filter of a quadrature steerable basis along each direction in proportion to 
;;; the amount of oriented energy that there is in that direction.  Only implemented
;;; for G2H2 filters, so far.

(defmethod energy-filter ((qsb quadrature-steerable-basis) &key ->)
  (with-result-image ((result ->) (dimensions qsb))
    (let ((order (1- (min (length (image-list (even-steerable-basis qsb)))
			  (length  (image-list (odd-steerable-basis qsb)))))))
      (cond ((and (equal order 2) 
		  (separable-filter-p (car (filter-list (even-steerable-basis qsb)))) 
		  (separable-filter-p (car (filter-list (odd-steerable-basis qsb)))))
	     (with-local-images (
				 (c1 (dc-oriented-energy qsb))
				 (pair (orientation-map qsb))
				 (c2 (copy (x-component pair)))
				 (c3 (copy (y-component pair)))
				 (a (add (car (image-list (even-steerable-basis qsb))) 
					 (caddr (image-list (even-steerable-basis qsb)))))
				 (b (sub (car (image-list (even-steerable-basis qsb))) 
					 (caddr (image-list (even-steerable-basis qsb)))))
				 (c (mul (cadr (image-list (even-steerable-basis qsb))) c3)))
	       (div (mul c2 b :-> b) 2.0 :-> b)
	       (div (div (add b c  :-> b) 2.0 :-> b) c1 :-> b)
	       (div a 2.0 :-> a)
	       (add a b :-> result)))))))


(defun agc (input &key (level 1) ->)
  (with-result-image ((result ->) input)
    (with-local-images ((abs (abs-value input))
			(blurred (blur abs :level level)))
      (div input blurred :-> result))))


#|

(load-image "/images/freeman/heart/cx45")
(make-quadrature-steerable-basis cx45 :-> 'cx45qsb)
(energy-filter cx45qsb :-> 'adaptcx)
(agc adaptcx :-> 'adaptagc)
|#



;;; Local Variables:
;;; buffer-read-only: t 
;;; End:
