;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  File: steer.lisp
;;;  Author: David Heeger and Bill Freeman
;;;  Description: steerable filters and quadrature-steerable filters
;;;  Creation Date: summer '89
;;;  ----------------------------------------------------------------
;;;    Object-Based Vision and Image Understanding System (OBVIUS),
;;;      Copyright 1988, Vision Science Group,  Media Laboratory,  
;;;              Massachusetts Institute of Technology.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; revisions history:  Feb. 2, 1991  Freeman added steer-filter-list.
;;;                     March 26, 1991  freeman added 	  make-avg-steerable-filters
;;;                         and *avg-steerable-filters*.
;;;                     July 11, 1991  freeman commented-out *avg-steerable-filters*,
;;;                         because I don't use it and it slowed-down load-in.
;;;                     Oct. 8, 1991  Freeman allowed make-quadrature... to take 
;;;                         existing even and odd steerable basis.

(error "Steerable code not yet ported to OBVIUS 2.2")


(in-package 'obvius)
(export '(*default-even-steerable-filters* *default-odd-steerable-filters*
	  make-g1-steerable-filters
	  make-g4-steerable-filters make-h4-steerable-filters
	  make-avg-steerable-filters
	  steerable-basis steerable-basis-p separable-steerable-basis-p
	  make-steerable-basis even-steerable-basis odd-steerable-basis
	  filter-list image-list order
	  dimensions x-dim y-dim minimum maximum range
	  steer quadrature-steerable-basis make-quadrature-steerable-basis 
	  steer-even steer-odd sum-even sum-odd magnitude square-magnitude
	  complex-phase average-energy steer-filter-list
	  sample-1d sample-2d))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; DEFAULT STEERABLE FILTERS
;;; 2nd derivs of Gaussians 
;;; and their Hilbert transforms.

(defun sample-1d (function &key (window-size 9) (delx 1.0))
  (let ((kernel (make-array (list window-size) :element-type 'float))
	(x (- (floor window-size 2))))
    (loop for i from 0 below (array-dimension kernel 0) do
	  (setf (aref kernel i) (funcall function (* delx x)))
	  (setq x (+ 1.0 x)))
    kernel))

;;; 2nd derv of gaussian filters.   The polynomial functions:
(defun g2.poly (x)(* (exp (* x x -1.0)) (*
					 .92132
					 (+
					  (* 2.0 (expt x 2))
					  -1.0))))
(defun g2.gauss (x)(exp (* x x -1.0)))
(defun g2.diag (x)(* (exp (* x x -1.0)) (* 1.35744 x )))

;;; hilbert transforms of 2nd deriv of gaussian filters.  Polynomial forms.
(defun h3.hila (x)(* (exp (* x x -1.0)) (* 
					 .97796 
					 (+ 
					  (* -2.2544 x)
					  (expt x 3)))))
(defun h3.hilb (x) (exp (* x x -1.0)))
(defun h3.minusx (x)(* (exp (* x x -1.0))  x ))
(defun h3.minusy (x)(* (exp (* x x -1.0)) (* .97796
					     (+ 
					      (expt x 2)
					      -0.751465))))

(defun make-default-even-steerable-filters ()
  (let ((n2dgpoly (make-filter (sample-1d 'g2.poly :window-size 9 :delx 0.67)))
	(n2dggauss (make-filter (sample-1d 'g2.gauss :window-size 9 :delx 0.67)))
	(n2diag (make-filter (sample-1d 'g2.diag :window-size 9 :delx 0.67))))
    (list (make-separable-filter n2dggauss n2dgpoly 			      
				 :edge-handler "treflect")
	  (make-separable-filter n2diag n2diag 
				 :edge-handler "treflect")
	  (make-separable-filter n2dgpoly n2dggauss 
				 :edge-handler "treflect"))))

(defun make-default-odd-steerable-filters ()
  (let ((n3hila (make-filter (sample-1d 'h3.hila :window-size 9 :delx 0.67)))
	(n3hilb (make-filter (sample-1d 'h3.hilb :window-size 9 :delx 0.67)))
	(n3minusy (make-filter (sample-1d 'h3.minusy :window-size 9 :delx 0.67)))
	(n3minusx (make-filter (sample-1d 'h3.minusx :window-size 9 :delx 0.67))))
    (list (make-separable-filter n3hilb n3hila 
				 :edge-handler "treflect")
	  (make-separable-filter n3minusx n3minusy 
				 :edge-handler "treflect")
	  (make-separable-filter n3minusy n3minusx 
				 :edge-handler "treflect")
	  (make-separable-filter n3hila n3hilb 
				 :edge-handler "treflect"))))

;;; 2nd derivative of Gaussian filters  (normalized)
(defvar *default-even-steerable-filters* (make-default-even-steerable-filters))

;;; Hilbert transform of 2nd derivative of Gaussian filters  (normalized)
(defvar *default-odd-steerable-filters* (make-default-odd-steerable-filters))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; STEERABLE 1ST DERIVATIVE OF GAUSSIAN
;;; Sept. 27, 1989

;;; from loading  ~/mtca/defns.m  into mathematica, and evaluating N[normderivgauss2d[x,y,1]],
;;; we find the normalization required for the integral of 1st deriv of Gaussian to equal 1.
(defun g1.poly (x)(* (exp (* x x -1.0)) (* x -1.59577)))
(defun g1.gauss (x)(exp (* x x -1.0)))

(defun make-g1-steerable-filters ()
  (let ((n1dgpoly (make-filter (sample-1d 'g1.poly :window-size 9 :delx 0.67)))
	(n1dggauss (make-filter (sample-1d 'g1.gauss :window-size 9 :delx 0.67))))
    (list (make-separable-filter n1dggauss n1dgpoly 			      
				 :edge-handler nil)
	  (make-separable-filter n1dgpoly n1dggauss 
				 :edge-handler nil))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; DEFINE STEERABLE 4TH DERIVATIVE OF GAUSSIAN, AND STEERABLE 5TH ORDER FIT TO
;;; ITS HILBERT TRANSFORM
;;; August 27, 1989 Bill Freeman created from ~freeman/lisp/steer/steerables.lisp,
;;;                              and ~freeman/mtca/sepout.  See also Vision Science
;;;                              Technical Report #118.
;;; x-y separable Gaussian derivatives, order 4
(defun g4.4y (y)
       (* 1.24585 (exp (* y y -1.0)) (+ .75 (* y y -3.0) (expt y 4))))

(defun g4.4x (x)(exp (* x x -1.0)))

(defun g4.3x (x)(* 1.24585 x (exp (* x x -1.0))))
(defun g4.3y (y)(* (exp (* y y -1.0)) (+ (* y -1.5) (expt y 3))))

(defun g4.2x (x)(* 1.116176 (exp (* x x -1.0)) (+ (* x x) -0.5)))
(defun g4.2y (y)(* 1.116176 (exp (* y y -1.0)) (+ (* y y) -0.5)))

(defun g4.1y (y)(* 1.24585 y (exp (* y y -1.0))))
(defun g4.1x (x)(* (exp (* x x -1.0)) (+ (* x -1.5) (expt x 3))))

(defun g4.0x (x)(* 1.24585 (exp (* x x -1.0)) (+ .75 (* x x -3.0) (expt x 4))))
(defun g4.0y (y)(exp (* y y -1.0)))


;;; x-y separable order 5 Hilbert transforms of order 4 Gaussian derivatives

(defun h4.5y (y)(* 0.39752 (exp (* y y -1.0)) (+ 
					       (expt y 5) 
					       (* (expt y 3) -7.5014) 
					       (* y 7.1891))))
(defun h4.5x (x)(exp (* x x -1.0)))

(defun h4.4y (y)(* 0.39752 (exp (* y y -1.0)) (+ 
					       (expt y 4) 
					       (* (expt y 2) -4.501)
					       1.4378)))
(defun h4.4x (x)(* x (exp (* x x -1.0))))

;;; ;; non-separable version of h4.3
;;; (defun h4.3yx (y x)(* 0.39752 (exp (+ (* x x -1.0) (* y y -1.0))) 
;;; 		      (+
;;; 		       (* 1.4378 y)
;;; 		       (* x x y -2.25043)
;;; 		       (* (expt y 3) -.750143)
;;; 		       (* (expt y 3) (expt x 2)))))
;;; 

;; separable (approximate) version of h4.3
(defun h4.3y (y)(* 0.39752 (exp (* y y -1.0)) (+ 
					       (expt y 3) 
					       (* y -2.225))))
(defun h4.3x (x)(* (exp (* x x -1.0))
		   (+ 
		    (sqr x)
		    -.6638)))

;;; ;; non-separable version of h4.2
;;; (defun h4.2yx (y x)(* 0.39752 (exp (+ (* y y -1.0) (* x x -1.0))) 
;;; 		      (+
;;; 		       (* 1.4378 x)
;;; 		       (* y y x -2.25043)
;;; 		       (* (expt x 3) -.750143)
;;; 		       (* (expt x 3) (expt y 2)))))
;;; 

;;; approximately separable version of h4.2
(defun h4.2x (x)(* 0.39752 (exp (* x x -1.0)) (+ 
					       (expt x 3) 
					       (* x -2.225))))
(defun h4.2y (y)(* (exp (* y y -1.0))
		   (+ 
		    (sqr y)
		    -.6638)))

(defun h4.1x (x)(* 0.39752 (exp (* x x -1.0)) (+ 
					       (expt x 4) 
					       (* (expt x 2) -4.501)
					       1.4378)))
(defun h4.1y (y)(* y (exp (* y y -1.0))))


(defun h4.0x (x)(* 0.39752 (exp (* x x -1.0)) (+ 
					       (expt x 5) 
					       (* (expt x 3) -7.5014) 
					       (* x 7.1891))))
(defun h4.0y (y)(exp (* y y -1.0)))



;;; Following is only needed when use the more exact, non-separable function
;;; definition for h4.3 and h4.2
;;; (defun sample-2d (function &key 
;;; 			   (y-window-size 9) 
;;; 			   (x-window-size 9) 
;;; 			   (delx 1.0)
;;; 			   (dely 1.0))
;;;   (let ((kernel (make-array (list y-window-size x-window-size) :element-type 'float)))
;;;     (loop for y = (* dely (- (floor y-window-size 2))) then (incf y dely)
;;; 	  for j from 0 below (array-dimension kernel 0) do
;;; 	  (loop for x = (* delx (- (floor x-window-size 2))) then (incf x delx)
;;; 		for i from 0 below (array-dimension kernel 1) do
;;; 		(setf (aref kernel j i) (funcall function y x))))
;;;     kernel))



(defun make-g4-steerable-filters ()
  (list 
   (make-separable-filter 
    (make-filter (sample-1d 'g4.0y :window-size 13 :delx 0.5))
    (make-filter (sample-1d 'g4.0x :window-size 13 :delx 0.5))
    :edge-handler "treflect")
   (make-separable-filter 
    (make-filter (sample-1d 'g4.1y :window-size 13 :delx 0.5))
    (make-filter (sample-1d 'g4.1x :window-size 13 :delx 0.5))
    :edge-handler "treflect")
   (make-separable-filter 
    (make-filter (sample-1d 'g4.2y :window-size 13 :delx 0.5))
    (make-filter (sample-1d 'g4.2x :window-size 13 :delx 0.5))
    :edge-handler "treflect")
   (make-separable-filter 
    (make-filter (sample-1d 'g4.3y :window-size 13 :delx 0.5))
    (make-filter (sample-1d 'g4.3x :window-size 13 :delx 0.5))
    :edge-handler "treflect")
   (make-separable-filter 
    (make-filter (sample-1d 'g4.4y :window-size 13 :delx 0.5))
    (make-filter (sample-1d 'g4.4x :window-size 13 :delx 0.5))
    :edge-handler "treflect")))


(defun make-h4-steerable-filters ()
  (list 
   (make-separable-filter 
    (make-filter (sample-1d 'h4.0y :window-size 13 :delx 0.5))
    (make-filter (sample-1d 'h4.0x :window-size 13 :delx 0.5))
    :edge-handler "treflect")
   (make-separable-filter 
    (make-filter (sample-1d 'h4.1y :window-size 13 :delx 0.5))
    (make-filter (sample-1d 'h4.1x :window-size 13 :delx 0.5))
    :edge-handler "treflect")
   (make-separable-filter 
    (make-filter (sample-1d 'h4.2y :window-size 13 :delx 0.5))
    (make-filter (sample-1d 'h4.2x :window-size 13 :delx 0.5))
    :edge-handler "treflect")
   (make-separable-filter 
    (make-filter (sample-1d 'h4.3y :window-size 13 :delx 0.5))
    (make-filter (sample-1d 'h4.3x :window-size 13 :delx 0.5))
    :edge-handler "treflect")
   (make-separable-filter 
    (make-filter (sample-1d 'h4.4y :window-size 13 :delx 0.5))
    (make-filter (sample-1d 'h4.4x :window-size 13 :delx 0.5))
    :edge-handler "treflect")
   (make-separable-filter 
    (make-filter (sample-1d 'h4.5y :window-size 13 :delx 0.5))
    (make-filter (sample-1d 'h4.5x :window-size 13 :delx 0.5))
    :edge-handler "treflect")))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; this is for a set of steerable filters which average along a particular direction.

;;; sample a function in 2-d and make a filter out of the result.
(defun sample-2d (function  &key 
			    (y-window-size 9) 
			    (x-window-size 9) 
			    (delx 1.0)
			    (dely 1.0))
  (let ((kernel (make-array (list y-window-size x-window-size) :element-type 'single-float)))
    (loop for y = (* dely (- (floor y-window-size 2))) then (incf y dely)
 	  for j from 0 below (array-dimension kernel 0) do
 	  (loop for x = (* delx (- (floor x-window-size 2))) then (incf x delx)
 		for i from 0 below (array-dimension kernel 1) do
 		(setf (aref kernel j i) (float (funcall function y x )))))
    kernel))

;;; a steerable filter to perform angularly adaptive local averaging
;;; returns:  Cos(theta - theta-offset)^cos-power  exp(-r^2/(2 sigma^2)),
;;; where r = sqrt(x^2 + y^2), and theta = the usual arctan(y/x).
;;; Assumes cos-power is an even integer, so it takes abs to fix a numerical problem.
;;; Note:  play with symmetries (like atan y x  vs  atan x y) until get the
;;;        desired starting angle, and direction of rotation.
(defun avg-filter (theta-offset cos-power sigma y x &key (debug nil))
  (let ((r (sqrt (+ (* x x) (* y y))))
	(theta (if (and (= 0.0 x) (= 0.0 y))
		   0.0
		   (atan x y)))
	(out nil))
    (format debug "y ~d  x ~d  theta ~d  r ~d~%"
	    y x theta r)
    (setq out (* (expt (abs (cos (- theta theta-offset))) cos-power)
		 (exp (/ (* -1.0 r r)
			 (* 2 (* sigma sigma))))))
    (format debug "out ~d ~%" out)
    out))

;;; function chooses the cosine power for the filter response based on 
;;;  the number of filters chosen for the steerable basis.
;;;  Normalize this so that, ignoring the sampling effects, the function will
;;;  have a unity DC value.  See p. 262 wtf loose red notebook.
;;;  integral cos^N theta   exp(-r^2 / (2 sigma^2)  r  d theta  d r
;;;  =  (using abramowitz and stegun, p. 77 4.3.127, and p. 302, 7.4.5)
;;;  2 pi 1/2  3/4  5/6 ...  N-1/N     times  sigma^2.
;;; In practise, norm was within 3%, for a 9x9 filter.  I think that's ok.
(defun make-avg-steerable-filters (&key 
				   (n-filts 5)
				   (sigma 2.0)
				   (del 1.0)
				   (size 9))
  (let* ((cos-power (1- n-filts))
	 (norm (norm-avg-integral cos-power sigma)))
    (loop for i from 0 below n-filts collect
	  (make-filter (sample-2d #'(lambda (y x) 
				      (/ (avg-filter (* i (/ pi n-filts)) cos-power sigma y x) norm))
				  :x-window-size size :y-window-size size
				  :delx del :dely del)
		       :edge-handler "treflect"))))


;;; an auxiliary function to compute normalization for steerable blurring filters.
(defun norm-avg-integral (order sigma)
  (if (= order 2)
      (* pi sigma sigma)
      (/ (* (1- order) (norm-avg-integral (- order 2) sigma)) order)))

;;; (defvar *avg-steerable-filters* (make-avg-steerable-filters))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; STEERABLE BASIS OBJECT

(DEFCLASS steerable-basis (viewable)
  (image-list
   filter-list)
  (:accessor-prefix ))

(defmacro steerable-basis-p (obj)
  `(typep ,obj 'steerable-basis))

(defmethod separable-steerable-basis-p ((steerable steerable-basis))
  (every '(lambda (x) (separable-filter-p x)) (filter-list steerable)))
		      
;;; the input to this can be either an image, or a list of images
(defun make-steerable-basis (image
			     &key (filter-list *default-even-steerable-filters*)
			     (display-type t)
			     ((:-> name)))
  (when (viewable-p name) (error "Can not pass exisiting viewable to make-<vbl> functions"))
  (let* ((image-list (cond ((image-p image)
			    (loop for filter in filter-list
				  collect (apply-filter filter image)))
			   ((listp image)
			    image)
			   (t (error "Input must be either an image or a list or images"))))
	 (steerable (make-instance 'steerable-basis
				   :filter-list filter-list
				   :image-list image-list
				   :display-type display-type)))
    (dolist (im image-list) (push steerable (superiors-of im)))
    (set-history steerable 'make-steerable-basis image :filter-list filter-list)
    (set-name steerable name)
    steerable))

(defmethod default-display-type ((steerable steerable-basis) (pane pane))
  (if (= (depth pane) 1)
      nil
      'pasteup))

(defmethod inferiors-of ((steerable steerable-basis))
  (image-list steerable))

(defmethod notify-of-inferior-destruction ((steerable steerable-basis) sub-image)
  (cerror "Destroy  both ~A and ~A."
	  "You are attempting to destroy ~A which is contained in ~A."
	  sub-image steerable)
  (destroy steerable))

(defmethod order ((steerable steerable-basis))
  (1- (length (filter-list steerable))))

(defmethod dimensions ((steerable steerable-basis))
  (dimensions (car (image-list steerable))))

(defmethod x-dim ((steerable steerable-basis))
  (x-dim (car (image-list steerable))))

(defmethod y-dim ((steerable steerable-basis))
  (y-dim (car (image-list steerable))))

(defmethod minimum ((steerable steerable-basis))
  (loop for im in (image-list steerable) 
	minimize (the single-float (minimum im))))

(defmethod maximum ((steerable steerable-basis))
  (loop for im in (image-list steerable) 
	maximize (the single-float (maximum im))))

(defmethod range ((steerable steerable-basis))
  (- (the single-float (maximum steerable)) 
     (the single-float (minimum steerable))))

(defmethod sum ((steerable steerable-basis) &key ->)
  (with-result-image ((res ->) (dimensions steerable)
		      'sum steerable)
    (loop for im in (image-list steerable) do
	  (add im res :-> res))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; STEERING FUNCTIONS FOR EACH DIFFERENT ORDER

;;; *** Bill, please write this ***
(defmethod steer ((steerable steerable-basis) angle 
		  &key ->)
  (with-result-image ((result ->) (dimensions steerable)
		      'steer steerable angle)
      (if (separable-steerable-basis-p steerable)
	  (case (order steerable)	; separable steerable
	    (1 (steer-separable-1 (image-list steerable) angle result))
	    (2 (steer-separable-2 (image-list steerable) angle result))
	    (3 (steer-separable-3 (image-list steerable) angle result))
	    (4 (steer-separable-4 (image-list steerable) angle result))
	    (5 (steer-separable-5 (image-list steerable) angle result)))
	  (case (order steerable)	; non-separable steerable
	    (1 nil)
	    (2 nil)
	    (3 (steer-3 (image-list steerable) angle result))
	    (4 (steer-4 (image-list steerable) angle result))
	    (5 nil))
	  )))

;;;;;;;; steering functions for images made with x-y separable filters:  ;;;;;;;;;

(defun steer-separable-1 (image-list angle result)
  (with-local-images ((a (make-image (dimensions (car image-list))))
		      (b (make-image (dimensions (car image-list)))))
    (cond ((image-p angle)      
	   (sin. angle :-> a)
	   (cos. angle :-> b)
	   (mul (nth 0 image-list) b :-> result) ;;; cos * image
	   (mul (nth 1 image-list) a :-> b)      ;;; sin * image
	   (sub result b :-> result))
	  (t 
	   (mul (nth 0 image-list) (cos angle) :-> a)  ;;; cos * image
	   (mul (nth 1 image-list) (sin angle) :-> b)  ;;; sin * image
	   (sub a b :-> result)))))

(defun steer-separable-2 (image-list angle result)
  (with-local-images ((a (make-image (dimensions (car image-list))))
		      (b (make-image (dimensions (car image-list))))
		      (c (make-image (dimensions (car image-list))))
		      (d (make-image (dimensions (car image-list))))
		      (e (make-image (dimensions (car image-list)))))
    (cond ((image-p angle)      
	   (sin. angle :-> a)
	   (cos. angle :-> b)
	   (square b :-> c);; cos-sqr
	   (mul (nth 0 image-list) c :-> result);; h-part
	   (square a :-> c);; sin-sqr
	   (mul (nth 2 image-list) c :-> d);; v-part
	   (add d result :-> result);; h-part + v-part
	   (mul a b :-> c);; sin-cos
	   (mul c -2.0 :-> c);; -2 sin-cos
	   (mul c (nth 1 image-list) :-> d);; d-part
	   (add result d :-> result))
	  (t 
	   (mul (nth 0 image-list) (expt (cos angle) 2) :-> a);; h-part
	   (mul (nth 1 image-list) (* -2.0 (sin angle) (cos angle)) :-> b);;d-part
	   (mul (nth 2 image-list) (expt (sin angle) 2):-> c);; v-part
	   (add a c :-> result)
	   (add b result :-> result)))))

(defun steer-separable-3 (image-list angle result)
  (with-local-images ((a (make-image (dimensions (car image-list))))
		      (b (make-image (dimensions (car image-list))))
		      (c (make-image (dimensions (car image-list))))
		      (d (make-image (dimensions (car image-list))))
		      (e (make-image (dimensions (car image-list))))
		      (f (make-image (dimensions (car image-list)))))
    (cond ((image-p angle)      
	   (sin. angle :-> a)
	   (cos. angle :-> b)
	   (square b :-> c);; cos-sqr
	   (mul c b :-> f);; cos-cube
	   (mul c a :-> d);; cos-sqr-sin
	   (mul (nth 1 image-list) d :-> e);; d2-part
	   (square a :-> c);; sin-sqr
	   (mul b c :-> d);; cos-sin-sqr
	   (mul (nth 2 image-list) d :-> d);; d1-part
	   (mul (sub d e :-> d) 3.0 :-> result)
	   (mul (nth 0 image-list) f :-> f);; h-part
	   (mul c a :-> c);; sin-cube
	   (mul (nth 3 image-list) c :-> c);; v-part
	   (sub f c :-> f);; "sub h-part v-part :-> h-part" 
	   (add f result :-> result))
	  (t 
	   (mul (nth 0 image-list) (expt (cos angle) 3) :-> a)  ;; h-part
	   (mul (nth 3 image-list) (expt (sin angle) 3) :-> b)  ;; v-part
	   (mul (nth 2 image-list) (* 3.0 (expt (sin angle) 2) (cos angle)) :-> c) ;;d1-part 
	   (mul (nth 1 image-list) (* 3.0 (sin angle) (expt (cos angle) 2)) :-> d) ;; d2-part
	   (add (sub a b :-> a) 
		(sub c d :-> c) :-> result)))))


;;; see vision science technical report for the steering formulas
(defun steer-separable-4 (image-list angle result)
  (with-local-images ((a (make-image (dimensions (car image-list))))
		      (b (make-image (dimensions (car image-list))))
		      (c (make-image (dimensions (car image-list))))
		      (d (make-image (dimensions (car image-list))))
		      (e (make-image (dimensions (car image-list))))
		      (f (make-image (dimensions (car image-list))))
		      (g (make-image (dimensions (car image-list)))))
    (cond ((image-p angle)      
	   (sin. angle :-> a)
	   (cos. angle :-> e)
	   (square a :-> b)   ;; sin^2
	   (mul b a :-> c)    ;; sin^3
	   (square b :-> d)   ;; sin^4          d
	   (mul e c :-> c)    ;; sin^3 cos      c
	   (square e :-> f)   ;; cos^2
	   (mul f b :-> b)    ;; cos^2 sin^2    b
	   (mul e f :-> g)    ;; cos^3
	   (mul g a :-> a)    ;; cos^3 sin      a
	   (square f :-> g)   ;; cos^4          g
	   (mul a -4.0 :-> a) ;; -4sincos^3       a
	   (mul b 6.0 :-> b)  ;; 6sin^2 cos^2     b
	   (mul c -4.0 :-> c) ;; -4sin^3 cos      c
	   (mul (nth 0 image-list) g :-> g);;      g
	   (mul (nth 1 image-list) a :-> a);;      a
	   (mul (nth 2 image-list) b :-> b);;      b
	   (mul (nth 3 image-list) c :-> c);;      c
	   (mul (nth 4 image-list) d :-> d);;      d
	   (add g a :-> e);; cos^4 + 4 cos^3 sin
	   (add b c :-> f);; 6 sin^2 cos^2 -4 sin^3 cos
	   (add e f :-> a);; cos^4 + 4 cos^3 sin + 6 sin^2 cos^2 -4 sin^3 cos
	   (add a d :-> result));; cos^4 + 4 cos^3 sin + 6 sin^2 cos^2 -4 sin^3 cos + sin^4
	  (t 
	   (mul (nth 0 image-list) (expt (cos angle) 4) :-> a)
	   (mul (nth 1 image-list) (* -4.0 (sin angle) (expt (cos angle) 3)) :-> b)
	   (mul (nth 2 image-list) (* 6.0 (sqr (cos angle)) (sqr (sin angle))) :-> c)
	   (mul (nth 3 image-list) (* -4.0 (cos angle) (expt (sin angle) 3)) :-> d)
	   (mul (nth 4 image-list) (expt (sin angle) 4) :-> e)
	   (add a b :-> f)
	   (add c d :-> g)
	   (add f g :-> a)
	   (add e a :-> result)))))

;;; see vision science technical report for the steering formulas
(defun steer-separable-5 (image-list angle result)
  (cond ((image-p angle)      
	 (with-local-images ((a (make-image (dimensions (car image-list))))
			     (b (make-image (dimensions (car image-list))))
			     (c (make-image (dimensions (car image-list))))
			     (d (make-image (dimensions (car image-list))))
			     (e (make-image (dimensions (car image-list))))
			     (f (make-image (dimensions (car image-list))))
			     (g (make-image (dimensions (car image-list)))))
	   (sin. angle :-> a) 
	   (cos. angle :-> b)  ;; cos-angle
	   (square a :-> c)   ;; sin^2
	   (mul c a :-> d)    ;; sin^3
	   (square c :-> e)   ;; sin^4          
	   (mul c d :-> f)    ;; sin^5          f
	   (mul b e :-> e)    ;; sin^4 cos      e
	   (square b :-> g)   ;; cos^2
	   (mul g d :-> d)    ;; cos^2 sin^3    d
	   (mul g b :-> g)    ;; cos^3
	   (mul g c :-> c)    ;; cos^3 sin^2    c
	   (mul g b :-> g)    ;; cos^4          
	   (mul g a :-> a)    ;; cos^4 sin      a
	   (mul g b :-> g)    ;; cos^5          g
	   (mul a -5.0 :-> a) ;; -5 cos^4 sin
	   (mul c 10.0 :-> c) ;; 10 cos^3 sin^2
	   (mul d -10.0 :-> d) ;; -10 cos^2 sin^3
	   (mul e 5.0 :-> e)  ;; 5 cos sin^4
	   (mul f -1.0 :-> f) ;; -sin^5
	   (mul (nth 0 image-list) g :-> g);;      g
	   (mul (nth 1 image-list) a :-> a);;      a
	   (mul (nth 2 image-list) c :-> c);;      c
	   (mul (nth 3 image-list) d :-> d);;      d
	   (mul (nth 4 image-list) e :-> e);;      e
	   (mul (nth 5 image-list) f :-> f);;      f
	   (add g a :-> b)            ;; cos^5 - 5 cos^4 sin
	   (add c d :-> result)       ;; 10 cos^3 sin^2 + -10 cos^2 sin^3
	   (add b result :-> result)  ;; cos^5 - 5 cos^4 sin + 10 cos^3 sin^2 + -10 cos^2 sin^3
	   (add e f :-> b)            ;; 5 cos sin^4 - sin^5
	   (add b result :-> result)));; cos^5 - 5 cos^4 sin + 10 cos^3 sin^2 + 
                                      ;;    -10 cos^2 sin^3 + 5 cos sin^4 - sin^5
	(t 
	 (with-local-images ((a (make-image (dimensions (car image-list))))
			     (b (make-image (dimensions (car image-list))))
			     (c (make-image (dimensions (car image-list)))))
	   (mul (nth 0 image-list) (expt (cos angle) 5) :-> a)
	   (mul (nth 1 image-list) (* -5.0 (sin angle) (expt (cos angle) 4)) :-> b)
	   (add a b :-> c)
	   (mul (nth 2 image-list) (* 10.0 (expt (cos angle) 3) (sqr (sin angle))) :-> a)
	   (mul (nth 3 image-list) (* -10.0 (sqr (cos angle)) (expt (sin angle) 3)) :-> b)
	   (add a b :-> result)
	   (add c result :-> result)
	   (mul (nth 4 image-list) (* 5.0 (cos angle) (expt (sin angle) 4)) :-> a)
	   (mul (nth 5 image-list) (* -1.0 (expt (sin angle) 5)) :-> b)
	   (add a b :-> c)
	   (add c result :-> result)))))



;;;;;;;; steering functions for images made with x-y non-separable filters:  ;;;;;;;;;

;;; non-separable, 3rd order polynomial filters.  See ~freeman/mtca/nonsepsteer.m
;;; for the formulae.
(defun steer-3 (image-list angle result)
  (with-local-images ((a (make-image (dimensions (car image-list))))
		      (b (make-image (dimensions (car image-list))))
		      (c (make-image (dimensions (car image-list))))
		      (d (make-image (dimensions (car image-list))))
		      (e (make-image (dimensions (car image-list)))))
    (cond ((image-p angle)      
	   (sin. angle :-> a)
	   (cos. angle :-> b)
	   (mul (square b :-> c) b :-> c);; cos cubed
	   (sub (mul c 2 :-> c) b :-> e);; coefficient of 0th image
	   (mul (nth 0 image-list) e :-> result) 
	   (mul (add a b :-> c) .7071 :-> c)  ;;; 0.707107*Cos[theta] + 0.707107*Sin[theta] 
	   (mul c (square c :-> d) :-> d)  ;;; above, cubed
	   (sub (mul d 2.0 :-> e) c :-> e);; coefficieint of 1st image
	   (add (mul (nth 1 image-list) e :-> e) result :-> result);; running total
	   (mul (square a :-> c) a :-> c);; sin cubed
	   (sub (mul c 2 :-> c) a :-> e);; coefficient of 2nd image
	   (add (mul (nth 2 image-list) e :-> e) result :-> result);; running total
	   (mul (sub a b :-> c) .7071 :-> c)  ;;; 0.707107*Sin[theta] - 0.707107*Cos[theta] 
	   (mul c (square c :-> d) :-> d)  ;;; above, cubed
	   (sub (mul d 2.0 :-> e) c :-> e);; coefficieint of 3rd image
	   (add (mul (nth 3 image-list) e :-> e) result :-> result));; result
	  (t 
	   (mul (nth 0 image-list) 
		(- (* (expt (cos angle) 3) 2.0) (cos angle)) :-> result)
	   (mul (nth 1 image-list) 
		(- (* (expt (cos (- angle (/ pi 4))) 3) 2.0)
		   (cos (- angle (/ pi 4))))
		:-> a)
	   (add a result :-> result)
	   (mul (nth 2 image-list) 
		(- (* (expt (cos (- angle (/ pi 2))) 3) 2.0)
		   (cos (- angle (/ pi 2))))
		:-> a)
	   (add a result :-> result)
	   (mul (nth 3 image-list) 
		(- (* (expt (cos (- angle (* 3 (/ pi 4)))) 3) 2.0)
		   (cos (- angle (* 3 (/ pi 4)))))
		:-> a)
	   (add a result :-> result)))))

(defun steer-4 (image-list angle result)
  (with-local-images ((a (make-image (dimensions (car image-list))))
		      (b (make-image (dimensions (car image-list))))
		      (c (make-image (dimensions (car image-list))))
		      (d (make-image (dimensions (car image-list))))
		      (e (make-image (dimensions (car image-list))))
		      (f (make-image (dimensions (car image-list)))))
    (cond ((image-p angle)      
	   (sin. angle :-> a)
	   (cos. angle :-> b)
	   (square b :-> c);; cos squared
	   (square c :-> d);; cos 4th
	   (add (mul c -2.4 :-> c) (mul d 3.2 :-> d) :-> d)
;;; - 2.4*Cos[theta]^2 + 3.2*Cos[theta]^4
	   (add d 0.2 :-> e);; coefficient of 0th image
	   (mul (nth 0 image-list) e :-> result) 

	   (add (mul b .809017 :-> d) (mul a .587785 :-> c) :-> f);; cos + sin
	   (square f :-> c);; sin + cos squared
	   (square c :-> d);; sin + cos 4th
	   (add (mul c -2.4 :-> c) (mul d 3.2 :-> d) :-> d) 
	   (add d 0.2 :-> e);; coefficient of 1st image
	   (add (mul (nth 1 image-list) e :-> e) result :-> result)

	   (add (mul b 0.309017 :-> d) (mul a 0.951057 :-> c) :-> f);; cos + sin
	   (square f :-> c);; sin + cos squared
	   (square c :-> d);; sin + cos 4th
	   (add (mul c -2.4 :-> c) (mul d 3.2 :-> d) :-> d) 
	   (add d 0.2 :-> e);; coefficient of 2nd image
	   (add (mul (nth 2 image-list) e :-> e) result :-> result)

	   (add (mul b -0.309017 :-> d) (mul a 0.951057 :-> c) :-> f);; cos + sin
	   (square f :-> c);; sin + cos squared
	   (square c :-> d);; sin + cos 4th
	   (add (mul c -2.4 :-> c) (mul d 3.2 :-> d) :-> d) 
	   (add d 0.2 :-> e);; coefficient of 2nd image
	   (add (mul (nth 3 image-list) e :-> e) result :-> result)

	   (add (mul b -0.809017 :-> d) (mul a 0.587785 :-> c) :-> f);; cos + sin
	   (square f :-> c);; sin + cos squared
	   (square c :-> d);; sin + cos 4th
	   (add (mul c -2.4 :-> c) (mul d 3.2 :-> d) :-> d) 
	   (add d 0.2 :-> e);; coefficient of 2nd image
	   (add (mul (nth 4 image-list) e :-> e) result :-> result))
	  (t 
	   (mul (nth 0 image-list) 
		(+ 0.2 (* -2.4 (expt (cos angle) 2)) (* 3.2 (expt (cos angle) 4)))
		:-> result)
	   (mul (nth 1 image-list) 
		(+ 0.2 (* -2.4 (expt (cos (- angle (/ pi 5))) 2)) 
		   (* 3.2 (expt (cos (- angle (/ pi 5))) 4)))
		:-> a)
	   (add a result :-> result)
	   (mul (nth 2 image-list) 
		(+ 0.2 (* -2.4 (expt (cos (- angle (* 2 (/ pi 5)))) 2)) 
		   (* 3.2 (expt (cos (- angle (* 2 (/ pi 5)))) 4)))
		:-> a)
	   (add a result :-> result)
	   (mul (nth 3 image-list) 
		(+ 0.2 (* -2.4 (expt (cos (- angle (* 3 (/ pi 5)))) 2)) 
		   (* 3.2 (expt (cos (- angle (* 3 (/ pi 5)))) 4)))
		:-> a)
	   (add a result :-> result)
	   (mul (nth 4 image-list) 
		(+ 0.2 (* -2.4 (expt (cos (- angle (* 4 (/ pi 5)))) 2)) 
		   (* 3.2 (expt (cos (- angle (* 4 (/ pi 5)))) 4)))
		:-> a)
	   (add a result :-> result)))))





;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; PASTEUP PICTURES FOR STEERABLE BASIS VIEWABLE

(defmethod init-picture ((pic pasteup) (steerable steerable-basis))
  (setf (layout pic) (list 1 (1+ (order steerable))))
  (let ((dims (list (* (+ (border pic) (y-dim steerable)) 
		       (list-y-dim (layout pic)))
		    (* (+ (border pic) (x-dim steerable))
		       (list-x-dim (layout pic))))))
    (setf (data pic) (new-array dims :element-type 'bltable-byte))
    (setf (dimensions pic) 
	  (new-array-dimensions dims :element-type 'bltable-byte)))
  (setf (auto-scale-compound-images pic) *auto-scale-compound-images*)
  (setf (viewable pic) steerable)
  (pushnew pic (pictures-of steerable))
  pic)

(defmethod present ((pic pasteup) (steerable steerable-basis))
  (when (or (not (listp (layout pic)))
	    (not (and (>= (x-dim pic) (* (list-x-dim (layout pic)) 
					 (+ (x-dim steerable) (border pic))))
		      (>= (y-dim pic) (* (list-y-dim (layout pic)) 
					 (+ (y-dim steerable) (border pic)))))))
    (free-array (data pic))
    (init-picture pic steerable))
  (with-slots (data pedestal scale pane x-offset y-offset current) pic
    (let ((row (round (+ (/ (- (y-dim pane) (y-dim pic)) 2) y-offset)))
	  (col (round (+ (/ (- (x-dim pane) (x-dim pic)) 2) x-offset))))
      (when (not (current-p pic))
	(rescale steerable pic)
	(setf current (current steerable)))
      (clear-pane pane)
      (if (auto-scale-compound-images pic)
	  (pane-title-bar pane (format nil "~s auto-scaled" (name steerable)))
	  (pane-title-bar pane 
			  (format nil "(~S - ~a) * ~a" (name steerable)
				  (prin1-to-string pedestal) 
				  (prin1-to-string scale))))
      (set-pane-cmap pane (colormap pic))
      (blt data (x-dim pic) (y-dim pic) 0 0 pane row col 
	   (x-dim pic) (y-dim pic) *pix-src* :current (current pic)))))

(defmethod rescale ((steerable steerable-basis) (pic pasteup))
  (with-slots (scale pedestal data) pic
    (let* ((new-scale 
	    (or scale 
		(and *auto-scale-images* (/-0 255.0 (range steerable) 1.0))
		*gray-scale*))
	   (new-pedestal 
	    (or pedestal
		(and *auto-scale-images* (minimum steerable))
		*gray-pedestal*)))
      (loop for x from 0 below (list-x-dim (layout pic)) do
	    (loop for y from 0 below (list-y-dim (layout pic)) 
		  while (< (+ x (* y (list-x-dim (layout pic))))
			   (1+ (order steerable))) do
		  (status-line-message (format nil "Rescaling: (~d, ~d)" y x))
		  (rescale-compound-image-into-8bit 
		   (nth (+ x (* y (list-x-dim (layout pic)))) (image-list steerable))
		   data (screen-of (picture-pane pic))
		   (auto-scale-compound-images pic)
		   new-pedestal new-scale 
		   (* y (+ (border pic) (y-dim steerable)))
		   (* x (+ (border pic) (x-dim steerable))))))
      (setf scale new-scale
	    pedestal new-pedestal))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; QUADRATURE STEERABLE BASIS OBJECT

(DEFCLASS quadrature-steerable-basis (viewable)
  (even-steerable-basis
   odd-steerable-basis)
  (:accessor-prefix ))

;;; note: no display type for these guys for now
;;; look at the even- or odd-steerable-basis inferiors 
;;; to see the basis images.
;;; "image" can be an image, or a list of even and odd steerable bases.
(defun make-quadrature-steerable-basis 
    (image &key (even-filters *default-even-steerable-filters*)
	   (odd-filters *default-odd-steerable-filters*)
	   (display-type nil)
	   ((:-> name)))
  (when (viewable-p name) (error "Can not pass exisiting viewable to make-<vbl> functions"))
  (let* ((even-steerable (if (listp image)
			     (car image)
			     (make-steerable-basis image :filter-list even-filters)))
	 (odd-steerable (if (listp image)
			    (cadr image)
			    (make-steerable-basis image :filter-list odd-filters)))
	 (steerable (make-instance 'quadrature-steerable-basis
				   :display-type display-type
				   :even-steerable-basis even-steerable
				   :odd-steerable-basis odd-steerable)))
    (push steerable (superiors-of even-steerable))
    (push steerable (superiors-of odd-steerable))
    (set-history steerable 'make-quadrature-steerable-basis image 
		 :even-filters even-filters :odd-filters odd-filters)
    (set-name steerable name)
    steerable))

;;; if you try to destroy either the even- or odd-steerable-basis,
;;; continuable error allows you to destroy the whole quadrature-steerable-basis
(defmethod notify-of-inferior-destruction ((steerable quadrature-steerable-basis) 
					   sub-steerable)
  (cerror "Destroy  both ~A and ~A."
	  "You are attempting to destroy ~A which is contained in ~A."
	  sub-steerable steerable)
  (destroy steerable))

(defmethod inferiors-of ((steerable quadrature-steerable-basis))
  (list (even-steerable-basis steerable) (odd-steerable-basis steerable)))

(defmethod dimensions ((steerable quadrature-steerable-basis))
  (dimensions (even-steerable-basis steerable)))

(defmethod order ((steerable quadrature-steerable-basis))
  (min (order (even-steerable-basis steerable))
       (order (odd-steerable-basis steerable))))

(defmethod x-dim ((steerable quadrature-steerable-basis))
  (x-dim (even-steerable-basis steerable)))

(defmethod y-dim ((steerable quadrature-steerable-basis))
  (y-dim (even-steerable-basis steerable)))

(defmethod steer-even ((steerable quadrature-steerable-basis) angle &key ->)
  (with-result-image ((res ->) (dimensions steerable)
		      'steer-even steerable angle)
    (steer (even-steerable-basis steerable) angle :-> res)))

(defmethod steer-odd ((steerable quadrature-steerable-basis) angle &key ->)
  (with-result-image ((res ->) (dimensions steerable)
		      'steer-odd steerable angle)
    (steer (odd-steerable-basis steerable) angle :-> res)))

(defmethod steer ((steerable quadrature-steerable-basis) angle &key ((:-> result)))
  (make-image-pair (list (steer-odd steerable angle) 
			 (steer-even steerable angle)) 
		   :-> result))

(defmethod sum-even ((steerable quadrature-steerable-basis) &key ((:-> result)))
  (sum (even-steerable-basis steerable) :-> result))

(defmethod sum-odd ((steerable quadrature-steerable-basis) &key ((:-> result)))
  (sum (odd-steerable-basis steerable) :-> result))
    
(defmethod magnitude ((steerable quadrature-steerable-basis) angle 
		      &key ((:-> result)))
  (magnitude (steer steerable angle) :-> result))

(defmethod square-magnitude ((steerable quadrature-steerable-basis) angle 
			     &key ((:-> result)))
  (square-magnitude (steer steerable angle) :-> result))

(defmethod complex-phase ((steerable quadrature-steerable-basis) angle 
			  &key ((:-> result)))
  (complex-phase (steer steerable angle) :-> result))

;;; In other functions, we want to be able to steer images,
;;; and have it just copy the result to the output.  The angle is irrelevant.
(defmethod steer ((im image) angle &key ->)
  (with-result-image ((res ->) (dimensions im)
		      'steer im angle)
    (copy im :-> res)))

;;; gives the integral over all angles of the orientated energy
(defmethod average-energy  ((qsb quadrature-steerable-basis) &key ->)
  (with-result-image ((result ->) (dimensions qsb)
		      'average-energy qsb)
    (cond ((and (= (order (even-steerable-basis qsb)) 2) 
		(= (order (odd-steerable-basis qsb)) 3)) 
	   (get-average-energy-2-3 (image-list (even-steerable-basis qsb))
				   (image-list (odd-steerable-basis qsb)) result)))))

(defun get-average-energy-2-3 (even-list odd-list result)
  (with-local-images (
		      (a (mul (nth 2 even-list) (nth 2 even-list)))
		      (b (mul (nth 0 even-list) (nth 0 even-list)))
		      (c (add a b))
		      (total (mul c 0.375)))  
    (mul (nth 0 odd-list) (nth 0 odd-list) :-> a)
    (mul (nth 3 odd-list) (nth 3 odd-list) :-> b)
    (add a b :-> c)
    (mul c 0.3125 :-> a)
    (add a total :-> total)
    (mul (nth 1 odd-list) (nth 1 odd-list) :-> a)
    (mul (nth 2 odd-list) (nth 2 odd-list) :-> b)
    (add a b :-> c)
    (mul c 0.5625 :-> a)
    (add a total :-> total)
    (mul (nth 0 odd-list) (nth 2 odd-list) :-> a)
    (mul (nth 1 odd-list) (nth 3 odd-list) :-> b)
    (add a b :-> c)
    (mul c 0.375 :-> a)
    (add a total :-> total)
    (mul (nth 1 even-list) (nth 1 even-list) :-> a)
    (mul a 0.5 :-> b)
    (add b total :-> total)
    (mul (nth 0 even-list) (nth 2 even-list) :-> a)
    (mul a 0.25 :-> b)
    (add b total :-> result)))


#|
;;; *** Bill, please write this ***
(defmethod total-energy ((steerable quadrature-steerable-basis) &key ->)
  (with-result-image ((result ->) (dimensions steerable)
		      'total-energy steerable)
    stuff))

;;; *** we also need to write these

(defmethod energy-vs-angle ((steerable quadrature-steerable-basis) j i
			    &key ->)
  ;; returns a discrete function
  )

(defmethod energy-peak ((steerable quadrature-steerable-basis)
			&key ->)
  ;; returns image-pair with vector-field display-type
  )
|#



;;; a steering method for filters.  This is more efficient sometimes than 
;;; applying each filter to an image, and then steering the images.  (if you
;;; only have one angle to which you are going to steer the results).
;;; This converts the filter-list to an image, steers the image, and then
;;; converts the result back to a filter.  
;;; It seems that convolutions applied with the same image size as the filter
;;; give spurious results.  You need a buffer of one pixel on a side.  The
;;; proper things is to go in and fix the filtering code, but I don't want to 
;;; touch that.  So make a bigger temporary image, and then crop it back down.
;;; Steer the filter to pi more than the actual angle, because you want to 
;;; transpose the filter when you apply it.
(defun steer-filter-list (filter-list angle &key ->)
  (with-result-filter ((result ->) (car filter-list) 'steer)
    (let* ((dim (dimensions (car filter-list)))
	   (ydim (+ 2 (car dim)))
	   (xdim (+ 2 (cadr dim))))
      (with-local-images ((imp (make-impulse (list ydim xdim)))
			  (sb (make-steerable-basis imp :filter-list filter-list))
			  (filter-image (steer sb (+ pi angle)))
			  (cropped (crop filter-image 1 1 (- ydim 2) (- xdim 2))))
	(copy (obvius::data cropped) :-> (kernel result))))))
