;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  File:  x-blt.lisp
;;;  Author: Heeger
;;;  Description: blt'ing images using LispView
;;;  Creation Date: summer '90
;;;  ----------------------------------------------------------------
;;;    Object-Based Vision and Image Understanding System (OBVIUS),
;;;      Copyright 1988, Vision Science Group,  Media Laboratory,  
;;;              Massachusetts Institute of Technology.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'obvius)

(export '())

;;; Higher level code accesses depth and data slots.  X-bltables
;;; actually keep two intermediate representations: the (unzoomed,
;;; padded) data array AND the (zoomed, padded) X-image.  The data
;;; array is accessed and modified by higher level code.  The
;;; dimensions method on an X-bltable returns the UNPADDED ZOOMED
;;; dimensions.  The unzoomed dimensions are base-dimensions.  
;;; Zooming occurs in render, which creates the X-image if it is not
;;; there.  The positioning code needs the unpadded zoomed
;;; dimensions.  Render needs the base-dimensions to determine the
;;; zoomed version.  

;;; *** The code in here is totally losing in efficiency, since
;;; creating a lispview:image from a data array involves copying the
;;; array into a foreign space (or using XPutPixel).  When the
;;; LispView people sort out the pixmap/image stuff, we should change
;;; this so that the data is a (client owned) pixmap, which holds the
;;; scaled and quantized 8bit array and this gets sent to the server
;;; in render.  Currently, we can't access the pixmap array!

(def-simple-class X-bltable (frob)
  (screen-of
   depth
   foreground				;these hold LISPVIEW:colors
   background
   base-dimensions			;unzoomed, unpadded
   dimensions				;zoomed,   unpadded
   data					;unzoomed data array
   frob-x-offset                        ;offsets of X-image relative to
   frob-y-offset                        ;zoomed image
   X-image)
  (:default-initargs
      :foreground (lispview::find-color :name :white)
    :background (lispview::find-color :name :black)
    :pane->frob-y (make-transform :coerce #'floor)
    :pane->frob-x (make-transform :coerce #'floor)))

;;; Called by system-independent code to make a new bltable, or to
;;; remake an old one that needs to be recomputed.  base-dimensions
;;; are the size of the base data array (before zooming occors).  If
;;; bltable arg is a bltable, we return it if it is the right size and
;;; depth.  NOTE: bit-arrays are in 32 bit chunks, and are padded to
;;; even word boundaries.  A new bltable never has an X-image.
(defmethod make-bltable ((screen X-screen) base-dimensions
			 &rest initargs
			 &key (depth (depth screen))
			 bltable)
  (remf initargs :bltable)
  (let* ((padded-dims (X-bltable-dimensions base-dimensions depth))
	 (element-type (cond ((= depth 8) '(unsigned-byte 8))
			     ((= depth 1) 'bit)
			     (t (error "Can't handle depth ~A" depth)))))
    ;; If old bltable is right size, depth, and lives on same screen, reuse it.
    (unless (and (typep bltable 'X-bltable)
		 (equal (array-element-type (data bltable)) element-type)
		 (equal padded-dims (dimensions (data bltable)))
		 (eq screen (screen-of bltable)))
      (when bltable (destroy bltable))
      (setq bltable (apply 'make-instance 'X-bltable
			   :screen-of screen
			   :data (allocate-array padded-dims :element-type element-type)
                           :depth depth
			   :frob-y-offset 0
			   :frob-x-offset 0
			   :base-dimensions base-dimensions
			   initargs)))
    ;; Signal to render to re-compute the X-image (data has changed!)
    (when (X-image bltable)
      (setf (lispview:status (X-image bltable)) :destroyed)
      (setf (X-image bltable) nil))
    bltable))

;;; Colors may be obvius colors (keyword or list of values).
(defmethod shared-initialize ((bltable X-bltable) slot-list &rest initargs
			      &key foreground background)
  (when foreground
    (setf (getf initargs :foreground) (convert foreground 'lv:color)))
  (when background
    (setf (getf initargs :background) (convert background 'lv:color)))
  (apply #'call-next-method bltable slot-list initargs))

;;; *** This should call (X11:ximage-bytes-per-line X-image) eventually.  To do
;;; that, we must create the X-image here (and fill the data slot later...)
(defun X-bltable-dimensions (dims depth)
  (flet ((round-up (val byte-size) (* byte-size (ceiling val byte-size))))
    (cond ((= depth 1)
	   (list (round-up (car dims) 2)		;*** is this necessary?
		 (round-up (cadr dims) 32)))
	  ((= depth 8)
	   (list (car dims)
		 (round-up (cadr dims) 4))))))

(defmethod X-display ((bltable X-bltable))
  (with-slots (X-image) bltable
    (lispview:display X-image)))

(defmethod dimensions ((image lispview::image))
  (list (y-dim image) (x-dim image)))

(defmethod y-dim ((image lispview:image))
  (lispview::region-height
   (lispview::bounding-region image)))

(defmethod x-dim ((image lispview:image))
  (lispview::region-width
   (lispview::bounding-region image)))

(defmethod static-arrays-of ((bltable X-bltable))
  (list (data bltable)))
   
(defmethod destroy ((bltable X-bltable) &key &allow-other-keys)
  (call-next-method)
  (when (data bltable)
    (free-array (data bltable)))
  (when (X-image bltable)
    (setf (lispview:status (X-image bltable)) :destroyed)
    (setf (X-image bltable) nil)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Blting methods for drawing to bltables

(defmethod clear ((bltable X-bltable)
		  &key 
		  (color (background (screen-of bltable)))
		  &allow-other-keys)
  (if (= (depth bltable) 1)
      (fill! (data bltable) 0)
      (fill! (data bltable) (lispview::pixel color))))

;;; How much of an overhang should be stored for large or zoomed
;;; images, to allow for faster dragging, in terms of multiples
;;; of the pane size.
(defvar *X-bltable-overhang* 1.6)

;;; New render method, Ciamac Moallemi, 8/15/92.  The first time this
;;; is called, it will have to create the X-image, possibly zooming
;;; the data of the bltable, and pass the X-image over the wire to the
;;; X server (this is taken care of by LispView).  Subsequent calls
;;; will be much more efficient!  If the zoom factor is changed, the
;;; X-image is destroyed, and render computes a new one.  NOTE:
;;; foreground and background are used for blting 1bit X-images to
;;; deeper screens: they are SWITCHED for OPENWINDOWS ***. For 8-bit
;;; images, if the fully zoomed image is bigger than the pane time
;;; *X-bltable-overhang*, only a window that size is stored.  This
;;; allows faster zooming without cons'ing up massive amounts of
;;; memory, but it also slows down dragging.  For non-8-bit images,
;;; the behavior of the old render is duplicated, which is non-optimal
;;; but it would be really hairy to write sub/supersampling routines
;;; for 1-bit images.
(defmethod render ((pane X-pane) (bltable X-bltable) y-offset x-offset zoom)
  (with-slots (X-image pane->frob-y pane->frob-x
		       foreground background dimensions base-dimensions
		       frob-x-offset frob-y-offset) bltable
    (let* ((full-y-size (ceiling (* (car base-dimensions) zoom)))
	   (full-x-size (ceiling (* (cadr base-dimensions) zoom)))
	   (row (+ (floor (- (y-dim pane) full-y-size) 2) y-offset))
	   (col (+ (floor (- (x-dim pane) full-x-size) 2) x-offset)))
      (cond ((/= (depth bltable) 8)
	     ;; smarter scaling routines cannot be used with 1bit images
	     (when (and X-image
			(/= (cadr dimensions) full-x-size))
	       (setf (lispview:status X-image) :destroyed)
	       (setf X-image nil))
	     ;; If no X-image, create one
	     (when (null X-image) (compute-bltable-X-image bltable zoom))
	     ;; Adjust the coordinate transform so that position message works!
	     (setf (slot-value pane->frob-y 'offset) (/ (- row) zoom))
	     (setf (slot-value pane->frob-x 'offset) (/ (- col) zoom))
	     (setf (slot-value pane->frob-y 'scale) (/ zoom))
	     (setf (slot-value pane->frob-x 'scale) (/ zoom))      	       
	     (lispview:copy-area X-image pane
				 ;; u-l corner of X-image 
				 0 0
				 ;; area of X-image to blt (clipping)
				 (x-dim bltable) (y-dim bltable)
				 ;; destination location
				 col row
				 :foreground background :background foreground))
	    (t ;; smart scaling used
	     ;; frob size should be smaller than the full zoomed
	     ;; size and the pane, in the case of the pane
	     ;; we must make sure we are we are on pixel
	     ;; boundaries
	     (let* ((new-frob-y-size
		     (min full-y-size
			  (ceiling
			   (* (1+ (ceiling
				   (* *X-bltable-overhang*
				      (y-dim pane)) zoom)) zoom))))
		    (new-frob-x-size
		     (min full-x-size
			  (ceiling
			   (* (1+ (ceiling
				   (* *X-bltable-overhang*
				      (x-dim pane)) zoom)) zoom))))
		    (new-frob-y-offset 0)
		    (new-frob-x-offset 0)
		    (first-image-row 0)
		    (first-image-col 0))
	       ;; Adjust the coordinate transform so that position message works!
	       (setf (slot-value pane->frob-y 'offset) (/ (- row) zoom))
	       (setf (slot-value pane->frob-x 'offset) (/ (- col) zoom))
	       (setf (slot-value pane->frob-y 'scale) (/ zoom))
	       (setf (slot-value pane->frob-x 'scale) (/ zoom))

	       ;; destroy existing X-image when wrong size or wrong
	       ;; position (i.e. not covering pane)
	       (when (and X-image
			  (or (/= new-frob-y-size (y-dim bltable))
			      (/= new-frob-x-size (x-dim bltable))
			      (and (/= new-frob-y-size full-y-size)
				   (or (< (abs row) frob-y-offset)
				       (and (> row 0)
					    (/= frob-y-offset 0))
				       (> (+ frob-y-offset (y-dim bltable))
					  full-y-size)
				       (and (> (- (y-dim pane) row)
					       (+ frob-y-offset (y-dim bltable)))
					    (< (+ frob-y-offset (y-dim bltable))
					       full-y-size))))
			      (and (/= new-frob-x-size full-x-size)
				   (or (< (abs col) frob-x-offset)
				       (and (> col 0)
					    (/= frob-x-offset 0))
				       (> (+ frob-x-offset (x-dim bltable))
					  full-x-size)
				       (and (> (- (x-dim pane) col)
					       (+ frob-x-offset (x-dim bltable)))
					    (< (+ frob-x-offset (x-dim bltable))
					       full-x-size))))))
		 (setf (lispview:status X-image) :destroyed)
		 (setf X-image nil))

	       ;; when there is no X-image, generate one
	       (unless X-image	       
		 ;; if necessary, calculate frob offsets
		 (when (/= new-frob-y-size full-y-size)
		   (setq first-image-row (floor
					  (- (- row)
					     (floor (- new-frob-y-size (y-dim pane))
						    2))
					  zoom))
		   (setq new-frob-y-offset
			 (ceiling (* first-image-row zoom)))
		   (cond ((or (< first-image-row 0)
			      (< new-frob-y-offset 0))
			  (setq first-image-row 0)
			  (setq new-frob-y-offset 0))
			 ((> (+ new-frob-y-offset new-frob-y-size) full-y-size)
			  (setq first-image-row
				(floor (- full-y-size new-frob-y-size) zoom))
			  (setq new-frob-y-offset
				(ceiling (* first-image-row zoom))))))
		 (when (/= new-frob-x-size full-x-size)
		   (setq first-image-col (floor
					  (- (- col)
					     (floor (- new-frob-x-size (x-dim pane))
						    2))
					  zoom))
		   (setq new-frob-x-offset
			 (ceiling (* first-image-col zoom)))
		   (cond ((or (< first-image-col 0)
			      (< new-frob-x-offset 0))
			  (setq first-image-col 0)
			  (setq new-frob-x-offset 0))
			 ((> (+ new-frob-x-offset new-frob-x-size) full-x-size)
			  (setq first-image-col
				(floor (- full-x-size new-frob-x-size) zoom))
			  (setq new-frob-x-offset
				(ceiling (* first-image-col zoom))))))
		 (compute-bltable-sub-X-image bltable zoom
					      first-image-row first-image-col
					      new-frob-y-offset new-frob-x-offset
					      new-frob-y-size new-frob-x-size))
	       ;; copy over the X-image
	       (lispview:copy-area
		X-image pane
		0 0	; u-l corner of X-image 
		(x-dim bltable) (y-dim bltable)	; area of X-image to blt
		(+ col frob-x-offset) (+ row frob-y-offset) ; destination location
		:foreground background
		:background foreground)))))))

#|
;;; This is the old render method, it keeps around an X-image that is
;;; the size of the full zoomed image.  Losing, since it makes zooming
;;; slow and conses huge amounts of memory.  The first time this is
;;; called, it will have to create the X-image, possibly zooming the
;;; data of the bltable, and pass the X-image over the wire to the X
;;; server (this is taken care of by LispView).  Subsequent calls will
;;; be much more efficient!  If the zoom factor is changed, the
;;; X-image is destroyed, and render computes a new one.  NOTE:
;;; foreground and background are used for blting 1bit X-images to
;;; deeper screens!  *** SWITCHED for OPENWINDOWS ***
(defmethod render ((pane X-pane) (bltable X-bltable) y-offset x-offset zoom)
  (with-slots (X-image pane->frob-y pane->frob-x
		       foreground background dimensions base-dimensions) bltable
    ;; If X-image exists and is wrong size (due to zooming), destroy it.
    (when (and X-image
	       (/= (cadr dimensions)
		   (ceiling (* (cadr base-dimensions) zoom))))
      (setf (lispview:status X-image) :destroyed)
      (setf X-image nil))
    ;; If no X-image, create one
    (when (null X-image) (compute-bltable-X-image bltable zoom))
    (let ((row (+ (floor (- (y-dim pane) (y-dim bltable)) 2) y-offset))
	  (col (+ (floor (- (x-dim pane) (x-dim bltable)) 2) x-offset)))
      ;; Adjust the coordinate transform so that position message works!
      (setf (slot-value pane->frob-y 'offset) (/ (- row) zoom))
      (setf (slot-value pane->frob-x 'offset) (/ (- col) zoom))
      (setf (slot-value pane->frob-y 'scale) (/ zoom))
      (setf (slot-value pane->frob-x 'scale) (/ zoom))
      (lispview:copy-area X-image pane
			  0 0		;u-l corner of X-image 
			  (x-dim bltable) (y-dim bltable) ;area of X-image to blt (clipping)
			  col row	;destination location
			  :foreground background :background foreground))))
|#

;;;; First arg should just be one of the frobs, which are all assumed
;;; to have the same dimensions, depths, foreground and background.
;;; This function is basically a concatenation of stuff from (render
;;; x-bltable) and lv::dd-copy-area.
(defmethod fast-display-seq ((bltable x-bltable) pane frobs
			     x-offset y-offset zoom
			     frame-delay seq-delay test-fn
			     &aux gc)
  ;; Compute frobs if necessary:
  (with-status-message "Computing X-images"
    (loop for frob in frobs
	  for X-image = (x-image frob)
	  do
	  (when (and X-image
		     (/= (cadr (dimensions frob))
			 (ceiling (* (cadr (base-dimensions frob)) zoom))))
	    (setf (lispview:status X-image) :destroyed)
	    (setf X-image nil))
	  (unless X-image (compute-bltable-X-image frob zoom))))
  (status-message "Displaying movie ...")
  (setq gc (lv::graphics-context (lv::display pane)))
  (lv::with-graphics-context (gc :foreground (background bltable)
				 :background (foreground bltable))
    (let* ((lv-display (lv::display pane))
	   (dst-xvo (lv::device pane))
	   (dsp (lv::xview-object-dsp (lv::device lv-display)))
	   (xgc (lv::xview-object-xid (lv::device gc)))
	   (dst-xid (lv::xview-object-xid dst-xvo))
	   (src-xids (mapcar #'(lambda (f) (lv::xview-object-xid (lv::device (x-image f))))
			     frobs))
	   (width  (x-dim bltable))	;assume all frobs the same dimensions
	   (height (y-dim bltable))
	   (to-x (+ (floor (- (x-dim pane) (x-dim bltable)) 2) x-offset))
	   (to-y (+ (floor (- (y-dim pane) (y-dim bltable)) 2) y-offset))
	   (from-depth (lv::xview-drawable-depth (lv::device (x-image bltable))))
	   (to-depth (lv::xview-drawable-depth dst-xvo))
	   copy-plane-p)
      (cond ((= from-depth to-depth) (setq copy-plane-p nil))
	    ((= from-depth 1) (setq copy-plane-p t))
	    (t (error "Copying from drawable of depth ~D not supported"
		      from-depth)))
      (loop for count from 0
	    while (funcall test-fn count) do
	    (loop for src-xid in src-xids do
		  (XView::with-xview-lock
		      (if copy-plane-p
			  (X11:XCopyPlane dsp src-xid dst-xid xgc
					  0 0 width height to-x to-y 1)
			  (X11:XCopyArea dsp src-xid dst-xid xgc
					 0 0 width height to-x to-y))
		    (X11:XFlush dsp)
		    (frame-sleep frame-delay)))
	    (mp:process-allow-schedule)	;check for events!
	    (when (> seq-delay 0) (frame-sleep seq-delay))))))

;;; Make the X-image.  Set the dimensions slot (must be consistent
;;; with X-image size!). Zoom the data of the bltable. *** This should
;;; re-use the X-image when the new version of LispView allows this...
(defmethod compute-bltable-X-image ((bltable X-bltable) zoom)
  (with-slots (data screen-of X-image dimensions base-dimensions depth) bltable
    (let* (padded-dims)
      (setf dimensions (list (ceiling (* (car base-dimensions) zoom))
			     (ceiling (* (cadr base-dimensions) zoom))))
      (setq padded-dims (X-bltable-dimensions dimensions depth))
      (cond ((= zoom 1) ;*** wasteful!
	     (setq X-image (make-instance 'lispview:image
					  :depth (depth bltable)
					  :data data)))
	    ((= depth 1) ;(eq (array-element-type data) 'bit)
	     (with-static-arrays ((new-data (similar data
						     :dimensions padded-dims
						     :initial-element 0)))
	       (if (> zoom 1)
		   (internal-replicate-1bit data (x-dim data) (y-dim data)
					    new-data (cadr padded-dims)
					    (round zoom))
		   (internal-subsample-1bit data (x-dim data) (y-dim data)
					    new-data (cadr padded-dims)
					    (round (/ 1 zoom))))
	       (setq X-image (make-instance 'lispview:image
					    :depth (depth bltable)
					    :data new-data))))
	    ((= depth 8) ;(equal (array-element-type data) '(unsigned-byte 8))
	     (with-static-arrays ((new-data
				   (similar data :dimensions padded-dims :initial-element
					    (lispview::pixel (background screen-of)))))
	       (if (> zoom 1)
		   (internal-replicate-8bit data (x-dim data) (y-dim data)
					    0 0 ;source starting corner
					    new-data (x-dim new-data)
					    0 0 ;dest starting corner
					    (cadr padded-dims) (car padded-dims)
					    (round zoom))
		   (internal-subsample-8bit data (x-dim data) (y-dim data)
					    new-data (cadr padded-dims) 
					    (round (/ 1 zoom))))
	       (setq X-image (make-instance 'lispview:image
					    :depth (depth bltable)
					    :data new-data))))
	    (t (error "Can't handle Screen/Bltable depths of ~A" (depth bltable)))))))

;;; For 8-bit images only.  Make the X-image, set the frob-offset and
;;; dimensions slot.  Copy and zoom the proper region of the image into
;;; the X-image.
(defmethod compute-bltable-sub-X-image ((bltable X-bltable) zoom
					   first-image-row first-image-col
					   new-frob-y-offset new-frob-x-offset
					   new-frob-y-size new-frob-x-size)
  (with-slots (data screen-of X-image dimensions base-dimensions
		    depth frob-y-offset frob-x-offset) bltable
    (let* (padded-dims)
      (setf dimensions (list new-frob-y-size new-frob-x-size))
      (setf frob-y-offset new-frob-y-offset)
      (setf frob-x-offset new-frob-x-offset)
      (setq padded-dims (X-bltable-dimensions dimensions depth))
      (with-static-arrays ((new-data
			    (similar data :dimensions padded-dims :initial-element
				     (lispview::pixel (background screen-of)))))
	(cond ((= zoom 1)
	       (internal-paste-8bit-region data (x-dim data) (y-dim data)
					   first-image-col
					   first-image-row  ;source starting corner
					   new-data (x-dim new-data)
					   0 0 ;dest starting corner
					   (cadr padded-dims) (car padded-dims)))
	       ((> zoom 1)
		(internal-replicate-8bit data (x-dim data) (y-dim data)
					 first-image-col
					 first-image-row  ;source starting corner
					 new-data (x-dim new-data)
					 0 0 ;dest starting corner
					 (cadr padded-dims) (car padded-dims)
					 (round zoom)))
	      (t;; (< zoom 1)
	       (internal-subsample-8bit-region data (x-dim data) (y-dim data)
					       first-image-col
					       first-image-row  ;source starting corner
					       new-data (x-dim new-data)
					       0 0 ;dest starting corner
					       (cadr padded-dims) (car padded-dims)
					       (round (/ zoom)))))
	(setq X-image (make-instance 'lispview:image
				     :depth (depth bltable)
				     :data new-data))))))

;;; We ignore the zoom parameter here!  We should do this in the
;;; generic case too, since it is appropriate for hardcopy screens too! ****.
(defmethod draw-float-array ((bltable X-bltable) float-array
		      pedestal scale zoom
		      y-offset x-offset)
  (declare (ignore zoom))
  (with-slots (data) bltable
    (setq scale (/-0 1.0 scale 1.0))
    (unwind-protect
	 (cond ((= (depth bltable) 1)	;bitmap
		(internal-dither-into-1bit float-array (x-dim float-array) (y-dim float-array)
					   data (x-dim data)
					   (float pedestal) (float (* scale 2.0))
					   (round x-offset) (round y-offset)))
	       ((and (= (depth bltable) 8) (typep (screen-of bltable) '8bit-X-screen))
		(let* ((screen (screen-of bltable))
		       (ditherp (gray-dither screen))
		       (lut (gray-lut screen))
		       (lut-size (length (gray-lut screen))))
		  (funcall (if (or (null ditherp)
				   (and (numberp ditherp) (> lut-size ditherp)))
			       'internal-f-into-8bit-lut ;these have same arglist!
			       'internal-dither-into-8bit-lut)
			   float-array (x-dim float-array) (y-dim float-array) 
			   data (x-dim data)
			   (float pedestal) (float (* scale lut-size))
			   (round x-offset) (round y-offset) lut lut-size)))
	       (t (error "Can't handle Screen/Bltable depths of ~A/~A"
			 (depth (screen-of bltable)) (depth bltable)))))
  bltable))

(defmethod draw-lines ((bltable X-bltable) y0 x0 y1 x1 &rest args)
  (apply 'draw-lines (slot-value bltable 'X-image) y0 x0 y1 x1 args))


;;; X-dependent Code
(defmethod overlay-bitmap ((screen X-screen) pic sub-pic sub-vbl)
  (let ((color-pixel (lispview::pixel (if (typep (foreground sub-pic) 'lv:color)
					  (foreground sub-pic)
					  (find-true-color :name (foreground sub-pic)
							   :if-not-found :warn)))))
    (overlay-bitmap-data (data sub-vbl) (data (system-dependent-frob pic)) color-pixel)))

#|
;;; 1-17-90 -DH

;;; Test code to display generalized images (of arbitrary type).
;;; Should change the name of this function to draw-array.  In this
;;; version, array is not necessarily a float array.

(defmethod draw-float-array ((bltable t) array
		      pedestal scale gamma zoom
		      y-offset x-offset)
  (unwind-protect
       (cond ((lucid-float-arrays-p array)
	      (cond ((= (depth bltable) 1) ;bitmap
		     (internal-dither-into-1bit array (x-dim array) (y-dim array)
						(data bltable) (x-dim (data bltable))
						(float pedestal) (float (/-0 2.0 scale 1.0))
						(round x-offset) (round y-offset)))
		    ((= (depth bltable) 8)
		     (float-array-into-8bit array (data bltable) pedestal scale zoom y-offset x-offset))
		    (t (error "Can't handle Screen/Bltable depths of ~A" (depth bltable)))))
	     (t
	      (cond ((= (depth bltable) 1)
		     (error "General array to bitmap not implemented"))
		    ((= (depth bltable) 8)
		     (general-array-into-8bit (data bltable) pedestal scale zoom y-offset x-offset))
		    (t (error "Can't handle Screen/Bltable depths of ~A" (depth bltable)))))))
  bltable)

(defun float-array-into-8bit (float-array 8bit-array pedestal scale zoom y-offset x-offset)
  (let* ((screen (screen-of bltable)
	 (ditherp (gray-dither screen))
         (lut (gray-lut screen))
	 (lut-size (total-size lut))))
    (funcall (if (or (null ditherp)
		     (and (numberp ditherp) (> lut-size ditherp)))
		 'internal-f-into-8bit-lut ;these have same arglist!
		 'internal-dither-into-8bit-lut)
	     float-array (x-dim float-array) (y-dim float-array) 
	     8bit-array (x-dim 8bit-array)
	     (float pedestal) (float (* scale lut-size))
	     (round x-offset) (round y-offset) lut lut-size))
  8bit-array)

(defun general-array-into-8bit (float-array 8bit-array pedestal scale zoom y-offset x-offset)
  (let* ((lut (gray-lut (screen-of bltable)))
	 (lut-size (total-size lut)))
    ;;; *** write a loop here to do the rescaling ***
    )
  8bit-array)
|#


;;; Local Variables:
;;; buffer-read-only: t 
;;; End:
