;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  File:  x-draw.lisp
;;;  Author: Heeger/Simoncelli
;;;  Description: x-windows using LISPVIEW (lispview)
;;;  Creation Date: summer '90
;;;  ----------------------------------------------------------------
;;;    Object-Based Vision and Image Understanding System (OBVIUS),
;;;      Copyright 1988, Vision Science Group,  Media Laboratory,  
;;;              Massachusetts Institute of Technology.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'obvius)
(export '())

;;; An object which accepts drawing commands and can be blt'ed to the
;;; screen.  Screen-of holds an obvius::screen object.  This is used
;;; only to get the background color for clearing.  *** should this
;;; have a foreground color?
(def-simple-class X-drawable (frob lispview:image)
  (screen-of))

;;; Called by system-independent code.  Similar to make-bltable.
(defmethod make-drawable ((screen X-screen) dimensions
			  &key (depth (depth screen)) drawable)
  (let* ((actual-dims (X-bltable-dimensions dimensions depth)))
    (unless (and (typep drawable 'X-drawable)
		 (equal actual-dims (dimensions drawable))
		 (= depth (depth drawable)))
      (when drawable (destroy drawable)) ;*** Wasteful
      (setq drawable
	    (make-instance 'X-drawable
			   :screen-of screen
			   :height (car actual-dims)
			   :width (cadr actual-dims)
			   :depth depth)))
    (clear drawable :color (background screen))
    drawable))

(defmethod X-display ((drawable X-drawable))
  (lispview:display drawable))

(defmethod dimensions ((drawable X-drawable))
  (let ((rgn (lispview:bounding-region drawable)))
    (list (lispview:region-height rgn) (lispview:region-width rgn))))

(defmethod x-dim ((drawable X-drawable))
  (lispview:region-width (lispview:bounding-region drawable)))

(defmethod y-dim ((drawable X-drawable))
  (lispview:region-height (lispview:bounding-region drawable)))

(defmethod depth ((drawable X-drawable))
  (lispview:depth drawable))

(defmethod destroy ((drawable X-drawable) &key &allow-other-keys)
  (call-next-method)
  (setf (lispview:status drawable) :destroyed))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; system dependent stuff to draw to X-drawables

(defmethod font ((pane X-pane))
  (lispview:font (lispview:graphics-context (X-display pane))))

(defmethod font-height ((font lv:font))
  (+ (lispview:font-ascent font) (lispview:font-descent font)))

(defmethod string-width ((font lv:font) string)
  (lv:string-width font string))

;;; **** Why don't we just call the lispview function directly?  We
;;; could augment them with methods that worked on postscript panes.
(defmethod draw-text ((drawable X-drawable) y x string &rest keys)
  (apply #'lispview:draw-string drawable x (- y 2) string keys))

(defmethod draw-line ((drawable X-drawable) from-y from-x to-y to-x &rest keys)
  (apply #'lispview:draw-line drawable from-x from-y to-x to-y keys))

;;;; NOTE: this is actually faster and more cons-efficient than
;;;; calling the current version of lispview:draw-lines, since that
;;;; requires consing lists of vectors.  Call it with a LispVIew object!
(defmethod draw-lines ((X-thing lispview::drawable) y0 x0 y1 x1
		       &key
		       gc foreground line-width line-style
		       x-offset y-offset)
  (declare (type (array fixnum (*)) x0 x1 y0 y1)
	   (fixnum x-offset y-offset))
  (when foreground
    (unless (typep foreground 'lv:color)
      (setq foreground (find-true-color :name foreground :if-not-found :warn))))
  (let ((X-display (lispview:display X-thing)))
    (unless gc (setq gc (lispview:graphics-context X-display)))
    (lispview:with-graphics-context (gc :foreground foreground
					:line-style line-style
					:line-width line-width)
      (lispview:with-output-buffering X-display
	(cond ((or x-offset y-offset)
	       (setq x-offset (or x-offset 0))
	       (setq y-offset (or y-offset 0))
	       (dotimes (i (length y0))
		 (lispview:draw-line X-thing
				     (+ (aref x0 i) x-offset)
				     (+ (aref y0 i) y-offset)
				     (+ (aref x1 i) x-offset)
				     (+ (aref y1 i) y-offset)
				     :gc gc)))
	      (t (dotimes (i (length y0))
		   (lispview:draw-line X-thing
				       (aref x0 i) (aref y0 i)
				       (aref x1 i) (aref y1 i)
				       :gc gc))))))))

(defmethod draw-circles ((X-thing lispview::drawable) yarr xarr
			 &key gc foreground line-width line-style
			 fill-p radius)
  (declare (type (array fixnum (*)) xarr yarr))
  (when foreground
    (unless (typep foreground 'lv:color)
      (setq foreground (find-true-color :name foreground :if-not-found :warn))))
  (let ((X-display (lispview:display X-thing)))
    (unless gc (setq gc (lispview:graphics-context X-display)))
    (lispview:with-graphics-context (gc :foreground foreground
					:line-style line-style
					:line-width line-width)
      (lispview:with-output-buffering X-display
	(dotimes (i (length yarr))
	  (lispview:draw-arc X-thing
			     (- (aref xarr i) radius) (- (aref yarr i) radius)
			     (* 2 radius) (* 2 radius)
			     0 360
			     :fill-p fill-p :gc gc))))))

(defmethod draw-squares ((X-thing lispview::drawable) yarr xarr
			 &key gc foreground line-style line-width
			 fill-p size)
  (declare (type (array fixnum (*)) xarr yarr))
  (when foreground
    (unless (typep foreground 'lv:color)
      (setq foreground (find-true-color :name foreground :if-not-found :warn))))
  (let ((X-display (lispview:display X-thing)))
    (unless gc (setq gc (lispview:graphics-context X-display)))
    (lispview:with-graphics-context (gc :foreground foreground
					:line-style line-style
					:line-width line-width)
      (lispview:with-output-buffering X-display
	(dotimes (i (length yarr))
	  (lispview:draw-rectangle X-thing
				   (- (aref xarr i) size) (- (aref yarr i) size)
				   (* 2 size) (* 2 size)
				   :fill-p fill-p :gc gc))))))

(defmethod draw-circle ((drawable X-drawable) y-center x-center radius &rest keys)
  (apply #'lispview:draw-arc drawable
	 (- x-center radius) (- y-center radius)
	 (* 2 radius) (* 2 radius) 0 360
	 :fill-p t keys))

(defmethod draw-rect ((drawable X-drawable) y0 x0 y1 x1 &rest keys)
  (apply #'lispview:draw-rectangle drawable
	 x0 y0 (- x1 x0) (- y1 y0) :fill-p t keys))

(defmethod clear ((drawable X-drawable) &key (y0 0) (x0 0) 
		  (y1 (car (dimensions drawable)))
		  (x1 (cadr (dimensions drawable)))
		  (color (background (screen-of drawable))))
  (draw-rect drawable y0 x0 y1 x1 :foreground color))

(defmethod render ((pane X-pane) (drawable X-drawable) y-offset x-offset zoom)
  (declare (ignore zoom))
  (with-slots (pane->frob-y pane->frob-x) drawable
    (let ((corner-y (+ y-offset (floor (- (y-dim pane) (y-dim drawable)) 2)))
	  (corner-x (+ x-offset (floor (- (x-dim pane) (x-dim drawable)) 2))))
      (setf (slot-value pane->frob-y 'offset) (- corner-y))
      (setf (slot-value pane->frob-x 'offset) (- corner-x))
      ;; NOTE: foreground and background are used for blting 1bit X-drawables to
      ;; deeper screens!
      (lispview:copy-area drawable pane
			  0 0 (x-dim drawable) (y-dim drawable) corner-x corner-y
			  :foreground (foreground pane)
			  :background (background pane)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; draw-graph for X-drawables

;;; frob is a lispview:canvas (window or image).
;;; *** Why not just dispatch on lispview:canvas?
;;; *** this is SERIOUSLY UGLY!! -EPS
(defmethod draw-graph ((drawable X-drawable)
		       data graph-type
		       graph->frob-y graph->frob-x data->frob-y data->frob-x
		       y-range x-range y-axis x-axis
		       y-tick-step x-tick-step y-tick-length x-tick-length
		       y-tick-gap x-tick-gap y-tick-format-string x-tick-format-string
		       y-label x-label
		       &key
		       (gc (lispview:graphics-context (X-display drawable)))
		       (font (lispview:font gc))
		       color axis-color line-width
		       plot-symbol fill-symbol-p symbol-size
		       x-offset y-offset)
  (funcall 'x-draw-graph
	   drawable data graph-type
	   graph->frob-y graph->frob-x data->frob-y data->frob-x
	   y-range x-range y-axis x-axis
	   y-tick-step x-tick-step y-tick-length x-tick-length
	   y-tick-gap x-tick-gap y-tick-format-string x-tick-format-string
	   y-label x-label
	   :gc gc :font font :color color :axis-color axis-color
	   :line-width line-width
	   :plot-symbol plot-symbol :fill-symbol-p fill-symbol-p :symbol-size symbol-size
	   :x-offset x-offset :y-offset y-offset))

(defmethod draw-graph ((drawable X-pane)
		       data graph-type
		       graph->frob-y graph->frob-x data->frob-y data->frob-x
		       y-range x-range y-axis x-axis
		       y-tick-step x-tick-step y-tick-length x-tick-length
		       y-tick-gap x-tick-gap y-tick-format-string x-tick-format-string
		       y-label x-label
		       &key
		       (gc (lispview:graphics-context (X-display drawable)))
		       (font (font drawable))
		       color axis-color line-width
		       plot-symbol fill-symbol-p symbol-size
		       x-offset y-offset)
  (funcall 'x-draw-graph
	   drawable data graph-type
	   graph->frob-y graph->frob-x data->frob-y data->frob-x
	   y-range x-range y-axis x-axis
	   y-tick-step x-tick-step y-tick-length x-tick-length
	   y-tick-gap x-tick-gap y-tick-format-string x-tick-format-string
	   y-label x-label
	   :gc gc :font font :color color :axis-color axis-color
	   :line-width line-width
	   :plot-symbol plot-symbol :fill-symbol-p fill-symbol-p :symbol-size symbol-size
	   :x-offset x-offset :y-offset y-offset))

(defun x-draw-graph (drawable
		     data graph-type
		     graph->frob-y graph->frob-x data->frob-y data->frob-x
		     y-range x-range y-axis x-axis
		     y-tick-step x-tick-step y-tick-length x-tick-length
		     y-tick-gap x-tick-gap y-tick-format-string x-tick-format-string
		     y-label x-label
		     &key
		     (gc (lispview:graphics-context (X-display drawable)))
		     (font (lispview:font gc))
		     color axis-color line-width
		     plot-symbol fill-symbol-p symbol-size
		     x-offset y-offset)
  (declare (ignore y-offset x-offset y-label x-label))
  (unless (typep color 'lv:color)
    (setq color (find-true-color :name color :if-not-found :warn)))
  (unless (typep axis-color 'lv:color)
    (setq axis-color (find-true-color :name axis-color :if-not-found :warn)))  
  (lispview:with-graphics-context (gc :foreground axis-color)
    (lispview::with-output-buffering (X-display drawable)
      (when x-axis
	(plot-x-axis drawable graph->frob-x (transform-point graph->frob-y x-axis)
		     x-range x-tick-step x-tick-length x-tick-gap
		     x-tick-format-string font))
      (when y-axis
	(plot-y-axis drawable graph->frob-y (transform-point graph->frob-x y-axis)
		     y-range y-tick-step y-tick-length y-tick-gap
		     y-tick-format-string font))))
  (lispview:with-graphics-context
      (gc :line-width line-width
	  :foreground color
	  :clip-mask (lispview:make-region
		      :top (transform-point graph->frob-y (cadr y-range))
		      :left (transform-point graph->frob-x (car x-range))
		      :bottom (transform-point graph->frob-y (car y-range))
		      :right (1+ (transform-point graph->frob-x (cadr x-range)))))
    (let ((x-axis-pos
	   (transform-point graph->frob-y (or x-axis (apply 'clip 0 y-range)))))
      (plot-data graph-type data drawable
		 data->frob-y data->frob-x x-axis-pos
		 plot-symbol fill-symbol-p symbol-size
		 gc))))


;;; Local Variables:
;;; buffer-read-only: t 
;;; End:
