/*
    gfun.c -- Dispatch for generic functions.
*/
/*
    Copyright (c) 1990, Giuseppe Attardi.

    ECoLisp is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    See file '../Copyright' for full details.
*/


#include "config.h"

extern bool member_eq();		/* from list.d	*/
extern int  hash_eql();			/* from hash.d  */

object siScompute_applicable_methods;
object siScompute_effective_method;
object siSgeneric_function_method_combination;
object siSgeneric_function_method_combination_args;

siLallocate_gfun(int narg, object name, object arg_no, object ht)
{
	object x, y;
	int n, i;

	check_arg(3);

	if (!FIXNUMP(arg_no) ||
	    (number_minusp(arg_no) == TRUE))
	  FEerror("~S is not a non-negative number.", 1, arg_no);

	if (type_of(ht) != t_hashtable)
		FEwrong_type_argument(Shash_table, ht);

	x = alloc_object(t_gfun);
	x->gf.gf_spec_how = NULL; /* for GC sake */
	x->gf.gf_name = name;
	x->gf.gf_meth_ht = ht;
	n = fix(arg_no);
	x->gf.gf_arg_no = n;
	x->gf.gf_spec_how = (object *)alloc_relblock(sizeof(object)*n,
						     sizeof(object));
	for (i = 0;  i < n;  i++)
		x->gf.gf_spec_how[i] = OBJNULL;
	x->gf.gf_gfun = Cnil;
	VALUES(0) = x;
	RETURN(1);
}

siLgfun_name(int narg, object x)
{
	check_arg(1);

	if (type_of(x) != t_gfun)
		FEwrong_type_argument(Sdispatch_function, x);
	VALUES(0) = x->gf.gf_name;
	RETURN(1);
}

siLgfun_name_set(int narg, object x, object name)
{
	check_arg(2);

	if (type_of(x) != t_gfun)
		FEwrong_type_argument(Sdispatch_function, x);
	x->gf.gf_name = name;
	VALUES(0) = x;
	RETURN(1);
}

siLgfun_method_ht(int narg, object x)
{
	check_arg(1);

	if (type_of(x) != t_gfun)
		FEwrong_type_argument(Sdispatch_function, x);
	VALUES(0) = x->gf.gf_meth_ht;
	RETURN(1);
}

siLgfun_method_ht_set(int narg, object x, object y)
{
	check_arg(2);

	if (type_of(x) != t_gfun)
		FEwrong_type_argument(Sdispatch_function, x);
	if (type_of(y) != t_hashtable)
		FEwrong_type_argument(Shash_table, y);
	x->gf.gf_meth_ht = y;
	VALUES(0) = x;
	RETURN(1);
}

siLgfun_spec_how_ref(int narg, object x, object y)
{
	int i;
	check_arg(2);

	if (type_of(x) != t_gfun)
		FEwrong_type_argument(Sdispatch_function, x);
	if (!FIXNUMP(y) ||
	    (i = fix(y)) < 0 || i >= x->gf.gf_arg_no)
		FEerror("~S is an illegal spec_how index.", 1, y);
	VALUES(0) = x->gf.gf_spec_how[i];
	RETURN(1);
}

siLgfun_spec_how_set(int narg, object x, object y, object spec)
{
	int i;
	check_arg(3);

	if (type_of(x) != t_gfun)
		FEwrong_type_argument(Sdispatch_function, x);
	if (!FIXNUMP(y) || (i = fix(y)) >= x->gf.gf_arg_no)
		FEerror("~S is an illegal spec_how index.", 1, y);
	x->gf.gf_spec_how[i] = spec;
	VALUES(0) = spec;
	RETURN(1);
}

siLgfun_instance(int narg, object x)
{
	check_arg(1);

	if (type_of(x) != t_gfun)
		FEwrong_type_argument(Sdispatch_function, x);
	VALUES(0) = x->gf.gf_gfun;
	RETURN(1);
}

siLgfun_instance_set(int narg, object x, object y)
{
	check_arg(2);

	if (type_of(x) != t_gfun)
		FEwrong_type_argument(Sdispatch_function, x);
	if (type_of(y) != t_instance)
		FEwrong_type_argument(Sinstance, y);
	x->gf.gf_gfun = y;
	VALUES(0) = x;
	RETURN(1);
}

siLgfunp(int narg, object x)
{
	check_arg(1);
	if (type_of(x) == t_gfun)
		VALUES(0) = Ct;
	else
		VALUES(0) = Cnil;
	RETURN(1);
}


/*
 * variation of gethash from hash.d, which takes an array of objects as key
 * It also assumes that entries are never removed except by clrhash.
 */

struct htent *
get_meth_hash(object *keys, int argno, object hashtable)
{
	int hsize;
	struct htent *e, *htable;
	object hkey, tlist;
	register int i = 0;
	int k, n; /* k added by chou */
	bool b = 1;

	hsize = hashtable->ht.ht_size;
	htable = hashtable->ht.ht_self;
	for (n = 0; n < argno; n++)
	  i += (int)keys[n] / 4; /* instead of:
				   i += hash_eql(keys[n]);
				   i += hash_eql(Cnil);
				 */
	for (i %= hsize, k = 0; k < hsize;  i = (i + 1) % hsize, k++) {
	  e = &htable[i];
	  hkey = e->hte_key;
	  if (hkey == OBJNULL)
	    return(e);
	  for (n = 0, tlist = hkey; b && (n < argno);
	       n++, tlist = CDR(tlist))
	    b &= (keys[n] == CAR(tlist));
	  if (b)
	    return(&htable[i]);
	}
}

siLmethod_ht_get(int narg, object keylist, object table)
{
	struct htent *e;
	check_arg(2);
	{  int i, argn = length(keylist);
	   object keys[argn];	/* __GNUC__ */

	   for (i = 0; i < argn; i++, keylist = CDR(keylist))
	     keys[i] = CAR(keylist);
	   e = get_meth_hash(keys, argn, table);
	   if (e->hte_key == OBJNULL)
	     VALUES(0) = Cnil;
	   else
	     VALUES(0) = e->hte_value;
	   RETURN(1);
	 }
      }

set_meth_hash(object *keys, int argno, object hashtable, object value)
{
	struct htent *e;
	object keylist, *p;
	
	if (hashtable->ht.ht_nent + 1 >= fix(hashtable->ht.ht_rhthresh))
		extend_hashtable(hashtable);
	e = get_meth_hash(keys, argno, hashtable);
	if (e->hte_key == OBJNULL)
		hashtable->ht.ht_nent++;
	keylist = Cnil;
	for (p = keys + argno; p > keys; p--) keylist = CONS(p[-1], keylist);
	e->hte_key = keylist;
	e->hte_value = value;
}

gcall(int narg, object fun, object *args)
{
	object func;

	{ int i, spec_no;
	  struct htent *e;
	  object *spec_how = fun->gf.gf_spec_how;
	  object argtype[narg]; /* __GNUC__ */
	  extern object TYPE_OF();

	  if (narg < fun->gf.gf_arg_no)
	    FEerror("Generic function ~S requires more than ~R argument~:p.",
		    2, fun->gf.gf_name, MAKE_FIXNUM(narg));
	  for (i = 0, spec_no = 0; i < fun->gf.gf_arg_no; i++, spec_how++) {
	    if (*spec_how != Cnil)
	      argtype[spec_no++] = ((*spec_how == Ct) ||
				    !member_eq(args[i], *spec_how)) ?
				      TYPE_OF(args[i]) :
					args[i];
	  }

	  e = get_meth_hash(argtype, spec_no, fun->gf.gf_meth_ht);

	  if (e->hte_key == OBJNULL)  { 
	    /* method not cached */
	    register object gf = fun->gf.gf_gfun;
	    object methods, meth_comb, arglist = Cnil;

	    i = narg;
	    while (i-- > 0)
	      arglist = CONS(args[i], arglist);
	    funcall(3, siScompute_applicable_methods, gf, arglist);
	    methods = VALUES(0);
	    funcall(2, siSgeneric_function_method_combination, gf);
	    meth_comb = VALUES(0);
	    funcall(2, siSgeneric_function_method_combination_args, gf);
	    funcall(5, siScompute_effective_method, gf, methods,
		    meth_comb, VALUES(0));
	    func = VALUES(0);	/* result of funcall */
	  
	    /* update cache */
	    set_meth_hash(argtype, spec_no, fun->gf.gf_meth_ht, func);
	  } else
	    /* method is already cached */
	    func = e->hte_value;
	}
	switch (type_of(func)) {

	case t_cfun:
	  return(APPLY(narg, *func->cf.cf_self, args));

	case t_cclosure:
	  { int i; CSTACK(narg+1);
	    CPUSH(func->cc.cc_env);
	    for (i = 0; i < narg; i++)
	      CPUSH(*args++);
#ifdef CCALL
	    return(CCALL(narg+1, func->cc.cc_self));
#else
	    return(APPLY(narg+1, func->cc.cc_self, CSTACK_BOT));
#endif CCALL
	  }
	case t_cons:
	  return(apply(narg, func, args));

	default:
	  FEinvalid_function(func);
	}
      }

init_gfun()
{
	siScompute_applicable_methods =
	  make_si_ordinary("COMPUTE-APPLICABLE-METHODS");
	enter_mark_origin(&siScompute_applicable_methods);
	siScompute_effective_method =
	  make_si_ordinary("COMPUTE-EFFECTIVE-METHOD");
	enter_mark_origin(&siScompute_effective_method);
	siSgeneric_function_method_combination =
	  make_si_ordinary("GENERIC-FUNCTION-METHOD-COMBINATION");
	enter_mark_origin(&siSgeneric_function_method_combination);
	siSgeneric_function_method_combination_args =
	  make_si_ordinary("GENERIC-FUNCTION-METHOD-COMBINATION-ARGS");
	enter_mark_origin(&siSgeneric_function_method_combination_args);

	make_si_function("ALLOCATE-GFUN", siLallocate_gfun);
	make_si_function("GFUN-NAME", siLgfun_name);
	make_si_function("GFUN-NAME-SET", siLgfun_name_set);
	make_si_function("GFUN-METHOD-HT", siLgfun_method_ht);
	make_si_function("GFUN-METHOD-HT-SET", siLgfun_method_ht_set);
	make_si_function("GFUN-SPEC-HOW-REF", siLgfun_spec_how_ref);
	make_si_function("GFUN-SPEC-HOW-SET", siLgfun_spec_how_set);
	make_si_function("GFUN-INSTANCE", siLgfun_instance);
	make_si_function("GFUN-INSTANCE-SET", siLgfun_instance_set);
	make_si_function("GFUNP", siLgfunp);
	make_si_function("METHOD-HT-GET", siLmethod_ht_get);
}
