/*
    pathname.d -- Pathnames.
*/
/*
    Copyright (c) 1984, Taiichi Yuasa and Masami Hagiya.
    Copyright (c) 1990, Giuseppe Attardi.

    ECoLisp is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    See file '../Copyright' for full details.
*/

/*
	O.S. DEPENDENT

	This file contains those functions that interpret namestrings.
*/

#include "config.h"

/******************************* EXPORTS ******************************/

object Vdefault_pathname_defaults;
object Kwild;
object Knewest;
object Kstart;
object Kend;
object Kjunk_allowed;

/******************************* ------- ******************************/

object Khost;
object Kdevice;
object Kdirectory;
object Kname;
object Ktype;
object Kversion;
object Kdefaults;

object Kroot;
object Kcurrent;
object Kparent;
object Kper;

object
make_pathname(host, device, directory, name, type, version)
object host, device, directory, name, type, version;
{
	object x;

	x = alloc_object(t_pathname);
	x->pn.pn_host = host;
	x->pn.pn_device = device;
	x->pn.pn_directory = directory;
	x->pn.pn_name = name;
	x->pn.pn_type = type;
	x->pn.pn_version = version;
	return(x);
}


/* added by E. Wang */
object
tilde_expand(directory)
object directory;
{
	object head, prefix;
	extern object homedir_pathname();
	
	if (endp(directory))
		goto RET;
	head = CAR(directory);
	if (head == Kroot || head == Kcurrent || head == Kparent)
		goto RET;
	head = coerce_to_string(head);
	if (head->st.st_fillp == 0 || head->st.st_self[0] != '~')
		goto RET;
	prefix = homedir_pathname(head)->pn.pn_directory;
	directory = append(prefix, CDR(directory));
 RET:
	return directory;
}

static object
make_one(s, end)
char *s;
int end;
{
	object x;

	x = alloc_simple_string(end);
	x->st.st_self = alloc_contblock(end+1);
	memcpy(x->st.st_self, s, end);
	x->st.st_self[end] = '\0';
	return(x);
}

/* !!!!! Bug Fix. NLG */
object
parse_namestring(s, start, end, ep)
object s;
int start, end, *ep;
{
	int i, j, k, d;
	object path, *plast = &path, name, type;

	path = Cnil;
	for (i = j = start;  i < end;  ) {
		if (isspace(s->st.st_self[i]))
			break;
#ifdef unix
		if (IS_DIR_SEPARATOR(s->st.st_self[i])) {
#endif
			if (j == 0 && i == 0) {
				i++;
				plast = &CDR(*plast = CONS(Kroot, Cnil));
				j = i;
				continue;
			}
#ifdef unix
			/* BUG FIX by Grant J. Munsey */
			if (i == j) {
				i++;
				j = i;
				continue;
			}
			/* END OF BUG FIX */
			if (i-j == 1 && s->st.st_self[j] == '.') {
				plast = &CDR(*plast = CONS(Kcurrent, Cnil));
			} else if (i-j==2 && s->st.st_self[j]=='.' && s->st.st_self[j+1]=='.') {
				plast = &CDR(*plast = CONS(Kparent, Cnil));
			} else
				plast = &CDR(*plast = CONS(
					make_one(&s->st.st_self[j], i-j), Cnil));
#endif
			i++;
			j = i;
		} else
			i++;
	}
	*ep = i;
	if (i == j) {
		/*  no file and no type  */
		name = Cnil;
		type = Cnil;
		goto L;
	}
	for (k = j, d = -1;  k < i;  k++)
		if (s->st.st_self[k] == '.')
			d = k;
	if (d == -1) {
		/*  no file type  */
#ifdef unix
		if (i-j == 1 && s->st.st_self[j] == '*')
#endif
			name = Kwild;
		else
			name = make_one(&s->st.st_self[j], i-j);
		type = Cnil;
	} else if (d == j) {
		/*  no file name  */
		name = Cnil;
#ifdef unix
		if (i-d-1 == 1 && s->st.st_self[d+1] == '*')
#endif
			type = Kwild;
		else
			type = make_one(&s->st.st_self[d+1], i-d-1);
	} else {
		/*  file name and file type  */
#ifdef unix
		if (d-j == 1 && s->st.st_self[j] == '*')
#endif
			name = Kwild;
		else
			name = make_one(&s->st.st_self[j], d-j);
#ifdef unix
		if (i-d-1 == 1 && s->st.st_self[d+1] == '*')
#endif
			type = Kwild;
		else
			type = make_one(&s->st.st_self[d+1], i-d-1);
	}
L:
	path = tilde_expand(path);
	return(make_pathname(Cnil, Cnil, path, name, type, Cnil));

NO:
	*ep = i;
	return(OBJNULL);
}

object
coerce_to_pathname(x)
object x;
{
	object y;
	int e;

L:
	switch (type_of(x)) {
	case t_symbol:
	case t_string:
                /* !!!!! Bug Fix. NLG */
		y = parse_namestring(x, 0, x->st.st_fillp, &e);
		if (y == OBJNULL || e != x->st.st_fillp)
			goto CANNOT_COERCE;
		return(y);

	case t_pathname:
		return(x);

	case t_stream:
		switch ((enum smmode)x->sm.sm_mode) {
		case smm_input:
		case smm_output:
		case smm_probe:
		case smm_io:
			x = x->sm.sm_object1;
			/*
				The file was stored in sm.sm_object1.
				See open.
			*/
			goto L;

		case smm_synonym:
			x = symbol_value(x->sm.sm_object0);
			goto L;

		default:
			goto CANNOT_COERCE;
		}

	default:
	CANNOT_COERCE:
		FEerror("~S cannot be coerced to a pathname.", 1, x);
	}
}

object
default_device(host)
object host;
{
	return(Cnil);
	/*  not implemented yet  */
}

object
merge_pathnames(path, defaults, default_version)
object path, defaults, default_version;
{
	object host, device, directory, name, type, version;

	if (Null(path->pn.pn_host))
		host = defaults->pn.pn_host;
	else
		host = path->pn.pn_host;
	if (Null(path->pn.pn_device))
		if (Null(path->pn.pn_host))
			device = defaults->pn.pn_device;
		else if (path->pn.pn_host == defaults->pn.pn_host)
			device = defaults->pn.pn_device;
		else
			device = default_device(path->pn.pn_host);
	else
		device = path->pn.pn_device;
	if (Null(path->pn.pn_directory))
		directory = defaults->pn.pn_directory;
	else
		directory = path->pn.pn_directory;
	if (Null(path->pn.pn_name))
		name = defaults->pn.pn_name;
	else
		name = path->pn.pn_name;
	if (Null(path->pn.pn_type))
		type = defaults->pn.pn_type;
	else
		type = path->pn.pn_type;
	version = Cnil;
	/*
		In this implimentation, version is not counted
	*/
	return(make_pathname(host,device,directory,name,type,version));
}

/*
	Namestring(x) converts a pathname to a namestring.
*/
object
namestring(x)
object x;
{
	int i, j;
	object l, y;

	i = 0;
	l = x->pn.pn_directory;
	if (endp(l))
		goto L;
	y = CAR(l);
	if (y == Kroot) {
#ifdef unix
		token->st.st_self[i++] = '/';
#endif
		l = CDR(l);
	}
	for (;  !endp(l);  l = CDR(l)) {
		y = CAR(l);
#ifdef unix
		if (y == Kcurrent) {
			token->st.st_self[i++] = '.';
			token->st.st_self[i++] = '/';
			continue;
		} else if (y == Kparent) {
			token->st.st_self[i++] = '.';
			token->st.st_self[i++] = '.';
			token->st.st_self[i++] = '/';
			continue;
		}
#endif
		y = coerce_to_string(y);
		for (j = 0;  j < y->st.st_fillp;  j++)
			token->st.st_self[i++]
			= y->st.st_self[j];
#ifdef unix
		token->st.st_self[i++] = '/';
#endif
	}
L:
	y = x->pn.pn_name;
	if (Null(y))
		goto M;
	if (y == Kwild) {
#ifdef unix
		token->st.st_self[i++] = '*';
#endif
		goto M;
	}
	if (type_of(y) != t_string)
		FEerror("~S is an illegal pathname name.", 1, y);
	for (j = 0;  j < y->st.st_fillp;  j++)
		token->st.st_self[i++] = y->st.st_self[j];
M:
	y = x->pn.pn_type;
	if (Null(y))
		goto N;
	if (y == Kwild) {
		token->st.st_self[i++] = '.';
#ifdef unix
		token->st.st_self[i++] = '*';
#endif
		goto N;
	}
	if (type_of(y) != t_string)
		FEerror("~S is an illegal pathname name.", 1, y);
	token->st.st_self[i++] = '.';
	for (j = 0;  j < y->st.st_fillp;  j++)
		token->st.st_self[i++] = y->st.st_self[j];
N:
	token->st.st_fillp = i;
	token->st.st_self[i] = '\0';
	return(copy_simple_string(token));
}

object
coerce_to_namestring(object x)
{
	object y;
	int e;

L:
	switch (type_of(x)) {
	case t_symbol:
		y = alloc_simple_string(x->s.s_fillp);
		/* By Nick Gall */
		y->st.st_self = alloc_relblock(x->s.s_fillp+1, sizeof(char));
		memcpy(y->s.s_self, x->st.st_self, x->s.s_fillp+1);
		return(y);

	case t_string:
		if (x->st.st_self[0] != '~')
		  return(x);
		/* added by E. Wang */
		return(namestring(coerce_to_pathname(x)));

	case t_pathname:
		return(namestring(x));

	case t_stream:
		switch ((enum smmode)x->sm.sm_mode) {
		case smm_input:
		case smm_output:
		case smm_probe:
		case smm_io:
			x = x->sm.sm_object1;
			/*
				The file was stored in sm.sm_object1.
				See open.
			*/
			goto L;

		case smm_synonym:
			x = symbol_value(x->sm.sm_object0);
			goto L;

		default:
			goto CANNOT_COERCE;
		}

	default:
	CANNOT_COERCE:
		FEerror("~S cannot be coerced to a namestring.", 1, x);
	}
}

Lpathname(int narg, object name)
{
	check_arg(1);
	check_type_or_pathname_string_symbol_stream(&name);
	VALUES(0) = coerce_to_pathname(name);
	RETURN(1);
}

@(defun parse_namestring (thing
	&o host
	   (defaults `symbol_value(Vdefault_pathname_defaults)`)
	&k start end junk_allowed
	&a x y)
	int s, e, ee;
@
	check_type_or_pathname_string_symbol_stream(&thing);
	check_type_or_pathname_string_symbol_stream(&defaults);
	defaults = coerce_to_pathname(defaults);
	x = thing;
L:
	switch (type_of(x)) {
	case t_symbol:
	case t_string:
		get_string_start_end(x, start, end, &s, &e);
		for (;  s < e && isspace(x->st.st_self[s]);  s++)
			;
		y
                  /* !!!!! Bug Fix. NLG */
		= parse_namestring(x,
                                   s,
				   e - s,
				   &ee);
		if (Null(junk_allowed)) {
			for (;  ee < e - s;  ee++)
				if (!isspace(x->st.st_self[s + ee]))
					break;
			if (y == OBJNULL || ee != e - s)
				FEerror("Cannot parse the namestring ~S~%\
from ~S to ~S.",
					3, x, start, end);
		} else
			if (y == OBJNULL)
				@(return Cnil `MAKE_FIXNUM(s + ee)`)
		start = MAKE_FIXNUM(s + ee);
		break;

	case t_pathname:
		y = x;
		break;

	case t_stream:
		switch ((enum smmode)x->sm.sm_mode) {
		case smm_input:
		case smm_output:
		case smm_probe:
		case smm_io:
			x = x->sm.sm_object1;
			/*
				The file was stored in sm.sm_object1.
				See open.
			*/
			goto L;

		case smm_synonym:
			x = symbol_value(x->sm.sm_object0);
			goto L;

		default:
			goto CANNOT_PARSE;
		}

	default:
	CANNOT_PARSE:
		FEerror("Cannot parse the namestring ~S.", 1, x);
	}
	if (host != Cnil && y->pn.pn_host != Cnil &&
	    host != y->pn.pn_host)
		FEerror("The hosts ~S and ~S do not match.",
			2, host, y->pn.pn_host);
	@(return y start)
@)

@(defun merge_pathnames (path
	&o (defaults `symbol_value(Vdefault_pathname_defaults)`)
 	   (default_version Knewest))
@
	check_type_or_pathname_string_symbol_stream(&path);
	check_type_or_pathname_string_symbol_stream(&defaults);
	path = coerce_to_pathname(path);
	defaults = coerce_to_pathname(defaults);
	@(return `merge_pathnames(path, defaults, default_version)`)
@)

@(defun make_pathname (&key host device directory name
			    type version defaults
		       &aux x)
@
	if (Null(defaults)) {
		defaults
		= symbol_value(Vdefault_pathname_defaults);
		defaults = coerce_to_pathname(defaults);
		defaults
		= make_pathname(defaults->pn.pn_host,
			        Cnil, Cnil, Cnil, Cnil, Cnil);
	} else
		defaults = coerce_to_pathname(defaults);
	x = make_pathname(host, device, directory, name, type, version);
	x = merge_pathnames(x, defaults, Cnil);
	@(return x)
@)

Lpathnamep(int narg, object pname)
{
	check_arg(1);

	if (type_of(pname) == t_pathname)
		VALUES(0) = Ct;
	else
		VALUES(0) = Cnil;
	RETURN(1);
}

Lpathname_host(int narg, object pname)
{
	check_arg(1);

	check_type_or_pathname_string_symbol_stream(&pname);
	pname = coerce_to_pathname(pname);
	VALUES(0) = pname->pn.pn_host;
	RETURN(1);
}

Lpathname_device(int narg, object pname)
{
	check_arg(1);

	check_type_or_pathname_string_symbol_stream(&pname);
	pname = coerce_to_pathname(pname);
	VALUES(0) = pname->pn.pn_device;
	RETURN(1);
}

Lpathname_directory(int narg, object pname)
{
	check_arg(1);

	check_type_or_pathname_string_symbol_stream(&pname);
	pname = coerce_to_pathname(pname);
	VALUES(0) = pname->pn.pn_directory;
	RETURN(1);
}

Lpathname_name(int narg, object pname)
{
	check_arg(1);

	check_type_or_pathname_string_symbol_stream(&pname);
	pname = coerce_to_pathname(pname);
	VALUES(0) = pname->pn.pn_name;
	RETURN(1);
}

Lpathname_type(int narg, object pname)
{
	check_arg(1);

	check_type_or_pathname_string_symbol_stream(&pname);
	pname = coerce_to_pathname(pname);
	VALUES(0) = pname->pn.pn_type;
	RETURN(1);
}

Lpathname_version(int narg, object pname)
{
	check_arg(1);

	check_type_or_pathname_string_symbol_stream(&pname);
	pname = coerce_to_pathname(pname);
	VALUES(0) = pname->pn.pn_version;
	RETURN(1);
}

Lnamestring(int narg, object pname)
{
	check_arg(1);

	check_type_or_pathname_string_symbol_stream(&pname);
	VALUES(0) = coerce_to_namestring(pname);
	RETURN(1);
}

Lfile_namestring(int narg, object pname)
{
	check_arg(1);

	check_type_or_pathname_string_symbol_stream(&pname);
	pname = coerce_to_pathname(pname);
	VALUES(0) = namestring(make_pathname(Cnil, Cnil, Cnil,
					     pname->pn.pn_name,
					     pname->pn.pn_type,
					     pname->pn.pn_version));
	RETURN(1);
}

Ldirectory_namestring(int narg, object pname)
{
	check_arg(1);

	check_type_or_pathname_string_symbol_stream(&pname);
	pname = coerce_to_pathname(pname);
	VALUES(0) = namestring(make_pathname(Cnil, Cnil,
					     pname->pn.pn_directory,
					     Cnil, Cnil, Cnil));
	RETURN(1);
}

Lhost_namestring(int narg, object pname)
{
	check_arg(1);

	check_type_or_pathname_string_symbol_stream(&pname);
	pname = coerce_to_pathname(pname);
	pname = pname->pn.pn_host;
	if (Null(pname) || pname == Kwild)
		pname = make_simple_string("");
	VALUES(0) = pname;
	RETURN(1);
}

@(defun enough_namestring (path
	&o (defaults `symbol_value(Vdefault_pathname_defaults)`))
@
	check_type_or_pathname_string_symbol_stream(&path);
	check_type_or_pathname_string_symbol_stream(&defaults);
	defaults = coerce_to_pathname(defaults);
	path = coerce_to_pathname(path);
	path
	= make_pathname(equalp(path->pn.pn_host, defaults->pn.pn_host) ?
			Cnil : path->pn.pn_host,
	                equalp(path->pn.pn_device,
			       defaults->pn.pn_device) ?
			Cnil : path->pn.pn_device,
	                equalp(path->pn.pn_directory,
			       defaults->pn.pn_directory) ?
			Cnil : path->pn.pn_directory,
	                equalp(path->pn.pn_name, defaults->pn.pn_name) ?
			Cnil : path->pn.pn_name,
	                equalp(path->pn.pn_type, defaults->pn.pn_type) ?
			Cnil : path->pn.pn_type,
	                equalp(path->pn.pn_version,
			       defaults->pn.pn_version) ?
			Cnil : path->pn.pn_version);
	@(return `namestring(path)`)
@)

init_pathname()
{
	Vdefault_pathname_defaults =
	make_special("*DEFAULT-PATHNAME-DEFAULTS*",
		     make_pathname(Cnil, Cnil, Cnil, Cnil, Cnil, Cnil));

	Kwild = make_keyword("WILD");
	Knewest = make_keyword("NEWEST");

	Kstart = make_keyword("START");
	Kend = make_keyword("END");
	Kjunk_allowed = make_keyword("JUNK-ALLOWED");

	Khost = make_keyword("HOST");
	Kdevice = make_keyword("DEVICE");
	Kdirectory = make_keyword("DIRECTORY");
	Kname = make_keyword("NAME");
	Ktype = make_keyword("TYPE");
	Kversion = make_keyword("VERSION");
	Kdefaults = make_keyword("DEFAULTS");

	Kroot = make_keyword("ROOT");
	Kcurrent = make_keyword("CURRENT");
	Kparent = make_keyword("PARENT");
	Kper = make_keyword("PER");
}

init_pathname_function()
{
	make_function("PATHNAME", Lpathname);
	make_function("PARSE-NAMESTRING", Lparse_namestring);
	make_function("MERGE-PATHNAMES", Lmerge_pathnames);
	make_function("MAKE-PATHNAME", Lmake_pathname);
	make_function("PATHNAMEP", Lpathnamep);
	make_function("PATHNAME-HOST", Lpathname_host);
	make_function("PATHNAME-DEVICE", Lpathname_device);
	make_function("PATHNAME-DIRECTORY", Lpathname_directory);
	make_function("PATHNAME-NAME", Lpathname_name);
	make_function("PATHNAME-TYPE", Lpathname_type);
	make_function("PATHNAME-VERSION", Lpathname_version);
	make_function("NAMESTRING", Lnamestring);
	make_function("FILE-NAMESTRING", Lfile_namestring);
	make_function("DIRECTORY-NAMESTRING", Ldirectory_namestring);
	make_function("HOST-NAMESTRING", Lhost_namestring);
	make_function("ENOUGH-NAMESTRING", Lenough_namestring);
}
