;;;;  Copyright (c) 1984, Taiichi Yuasa and Masami Hagiya.
;;;;  Copyright (c) 1990, Giuseppe Attardi.
;;;;
;;;;    This program is free software; you can redistribute it and/or
;;;;    modify it under the terms of the GNU Library General Public
;;;;    License as published by the Free Software Foundation; either
;;;;    version 2 of the License, or (at your option) any later version.
;;;;
;;;;    See file '../Copyright' for full details.

;;;; CMPIF  Conditionals.

(in-package 'compiler)

(defun c1if (args &aux info f)
  (when (or (endp args) (endp (cdr args)))
        (too-few-args 'if 2 (length args)))
  (unless (or (endp (cddr args)) (endp (cdddr args)))
          (too-many-args 'if 3 (length args)))
  (setq f (c1fmla-constant (car args)))
  (case f
        ((T) (c1expr (second args)))
        ((NIL) (if (endp (cddr args)) (c1nil) (c1expr (third args))))
        (otherwise
         (setq info (make-info))
         (list 'IF info
               (c1fmla f info)
               (c1expr* (second args) info)
               (if (endp (cddr args)) (c1nil) (c1expr* (third args) info)))))
  )

(defun c1fmla-constant (fmla &aux f)
  (cond
   ((consp fmla)
    (case (car fmla)
          (AND (do ((fl (cdr fmla) (cdr fl)))
                   ((endp fl) t)
		 (setq f (c1fmla-constant (car fl)))
		 (case f
		   ((T))
		   ((NIL) (return nil))
		   (t (if (endp (cdr fl))
			  (return f)
			  (return (list* 'AND f (cdr fl))))))))
          (OR (do ((fl (cdr fmla) (cdr fl)))
                  ((endp fl) nil)
		(setq f (c1fmla-constant (car fl)))
		(case f
		  ((T) (return t))
		  ((NIL))
		  (t (if (endp (cdr fl))
			 (return f)
			 (return (list* 'OR f (cdr fl))))))))
          ((NOT NULL)
           (when (endp (cdr fmla)) (too-few-args 'not 1 0))
           (unless (endp (cddr fmla))
                   (too-many-args 'not 1 (length (cdr fmla))))
           (setq f (c1fmla-constant (second fmla)))
           (case f
                 ((T) nil)
                 ((NIL) t)
                 (t (list 'NOT f))))
          (t fmla)))
   ((symbolp fmla) (if (constantp fmla)
                       (if (symbol-value fmla) t nil)
                       fmla))
   (t t))
  )

(defun c1fmla (fmla info)
  (if (consp fmla)
      (case (car fmla)
            (AND (case (length (cdr fmla))
                   (0 (c1t))
                   (1 (c1fmla (second fmla) info))
                   (t (cons 'FMLA-AND
                            (mapcar #'(lambda (x) (c1fmla x info))
                                    (cdr fmla))))))
            (OR (case (length (cdr fmla))
                   (0 (c1nil))
                   (1 (c1fmla (second fmla) info))
                   (t (cons 'FMLA-OR
                            (mapcar #'(lambda (x) (c1fmla x info))
                                    (cdr fmla))))))
            ((NOT NULL)
                  (when (endp (cdr fmla)) (too-few-args 'not 1 0))
                  (unless (endp (cddr fmla))
                          (too-many-args 'not 1 (length (cdr fmla))))
                  (list 'FMLA-NOT (c1fmla (second fmla) info)))
            (t (c1expr* fmla info)))
      (c1expr* fmla info))
  )

(defun c2if (fmla form1 form2
                  &aux (Tlabel (next-label)) Flabel)
  (cond ((and (eq (car form2) 'LOCATION)
              (null (third form2))
              (eq *destination* 'TRASH)
              (not (eq *exit* 'RETURN)))
         (let ((exit *exit*)
               (*unwind-exit* (cons Tlabel *unwind-exit*))
               (*exit* Tlabel))
              (CJF fmla Tlabel exit))
         (wt-label Tlabel)
         (c2expr form1))
        (t
         (setq Flabel (next-label))
         (let ((*unwind-exit* (cons Flabel (cons Tlabel *unwind-exit*)))
               (*exit* Tlabel))
              (CJF fmla Tlabel Flabel))
         (wt-label Tlabel)
         (let ((*unwind-exit* (cons 'JUMP *unwind-exit*))
	       (*temp* *temp*))
	   (c2expr form1))
         (wt-label Flabel)
         (c2expr form2)))
  )

;;; If fmla is true, jump to Tlabel.  If false, do nothing.
(defun CJT (fmla Tlabel Flabel)
  (case (car fmla)
    (FMLA-AND (do ((fs (cdr fmla) (cdr fs)))
                  ((endp (cdr fs))
                   (CJT (car fs) Tlabel Flabel))
                (let* ((label (next-label))
                       (*unwind-exit* (cons label *unwind-exit*)))
                  (CJF (car fs) label Flabel)
                  (wt-label label))))
    (FMLA-OR (do ((fs (cdr fmla) (cdr fs)))
                 ((endp (cdr fs))
                  (CJT (car fs) Tlabel Flabel))
               (let* ((label (next-label))
                      (*unwind-exit* (cons label *unwind-exit*)))
                 (CJT (car fs) Tlabel label)
                 (wt-label label))))
    (FMLA-NOT (CJF (second fmla) Flabel Tlabel))
    (LOCATION
     (case (third fmla)
       ((T) (unwind-no-exit Tlabel) (wt-nl) (wt-go Tlabel))
       ((NIL))
       (t (let ((*destination* (list 'JUMP-TRUE Tlabel)))
            (c2expr* fmla)))))
    (t (let ((*destination* (list 'JUMP-TRUE Tlabel))) (c2expr* fmla))))
  )

;;; If fmla is false, jump to Flabel.  If true, do nothing.
(defun CJF (fmla Tlabel Flabel)
  (case (car fmla)
    (FMLA-AND (do ((fs (cdr fmla) (cdr fs)))
                  ((endp (cdr fs)) (CJF (car fs) Tlabel Flabel))
                (declare (object fs))
                (let* ((label (next-label))
                       (*unwind-exit* (cons label *unwind-exit*)))
                  (CJF (car fs) label Flabel)
                  (wt-label label))))
    (FMLA-OR (do ((fs (cdr fmla) (cdr fs)))
                 ((endp (cdr fs)) (CJF (car fs) Tlabel Flabel))
               (declare (object fs))
               (let* ((label (next-label))
                      (*unwind-exit* (cons label *unwind-exit*)))
                 (CJT (car fs) Tlabel label)
                 (wt-label label))))
    (FMLA-NOT (CJT (second fmla) Flabel Tlabel))
    (LOCATION
     (case (third fmla)
       ((T))
       ((NIL) (unwind-no-exit Flabel) (wt-nl) (wt-go Flabel))
       (t (let ((*destination* (list 'JUMP-FALSE Flabel)))
            (c2expr* fmla)))))
    (t (let ((*destination* (list 'JUMP-FALSE Flabel))) (c2expr* fmla))))
  )

(defun c1and (args)
  (cond ((endp args) (c1t))
        ((endp (cdr args)) (c1expr (car args)))
        (t (let ((info (make-info))) (list 'AND info (c1args args info))))))

(defun c2and (forms)
  (do ((forms forms (cdr forms))
       (form))
      ((endp (cdr forms))
       (c2expr (car forms)))
    (declare (list forms))
    (setq form (first forms))
    (cond ((eq (car form) 'LOCATION)
           (case (third form)
             ((T))
             ((NIL) (unwind-exit nil 'JUMP))
             (t (wt-nl "if(" (third form) "==Cnil){")
                (unwind-exit nil 'JUMP) (wt "}")
                )))
          ((eq (car form) 'VAR)
           (wt-nl "if(")
           (wt-var (car (third form)))	; (second (third form)) ccb
           (wt "==Cnil){")
           (unwind-exit nil 'JUMP) (wt "}"))
          (t
           (let* ((label (next-label))
                  (*unwind-exit* (cons label *unwind-exit*)))
             (let ((*destination* (list 'JUMP-TRUE label)))
               (c2expr* form))
             (unwind-exit nil 'JUMP)
             (wt-label label))))
      ))

(defun c1or (args)
  (cond ((endp args) (c1nil))
        ((endp (cdr args)) (c1expr (car args)))
        (t (let ((info (make-info)))
                (list 'OR info (c1args args info))))))

(defun c2or (forms &aux temp)
  (do ((forms forms (cdr forms))
       (form))
      ((endp (cdr forms))
       (c2expr (car forms)))
    (declare (list forms))
    (setq form (first forms))
    (cond ((eq (car form) 'LOCATION)
	   (case (third form)
	     ((T) (unwind-exit t 'JUMP))
	     ((NIL))
	     (t (wt-nl "if(" (third form) "!=Cnil){")
		(unwind-exit (third form) 'JUMP) (wt "}"))))
	  ((eq (car form) 'VAR)
	   (wt-nl "if(")
	   (wt-var (car (third form)))	; (second (third form)) ccb
	   (wt "!=Cnil){")
	   (unwind-exit (cons 'VAR (third form)) 'JUMP) (wt "}"))
	  ((and (eq (car form) 'CALL-GLOBAL)
		(get (third form) 'PREDICATE))
	   (let* ((label (next-label))
		  (*unwind-exit* (cons label *unwind-exit*)))
	     (let ((*destination* (list 'JUMP-FALSE label)))
	       (c2expr* form))
	     (unwind-exit t 'JUMP)
	     (wt-label label)))
	  #+locative
	  ((and (eq (car form) 'CALL-GLOBAL)
		(eq (get (third form) 'PROCLAIMED-RETURN-TYPE) 'NULL))
	   (let ((*destination* 'TRASH))
	     (c2expr* form)))
	  (t
	   (let* ((label (next-label))
		  (*unwind-exit* (cons label *unwind-exit*)))
	     ;; to hanlde (or (foo) ..), since foo may be inlined,
	     ;; we force the result into VALUES(0)
	     (let ((*destination* 'RETURN)) (c2expr* form))
	     (wt-nl "if(VALUES(0)==Cnil)") (wt-go label)
	     (unwind-exit '(VALUE 0) 'JUMP)
	     (wt-label label))))
    )
  )

(defun set-jump-true (loc label)
  (unless (null loc)
    (cond ((eq loc t))
          ((eq (car loc) 'INLINE-COND)
           (wt-nl "if(")
           (wt-inline-loc (third loc) (fourth loc))
           (wt ")"))
          ((member (car loc) '(CALL CALL-LOCAL) :test #'eq)
	   (wt-nl loc ";")
           (wt-nl "if(VALUES(0)!=Cnil)"))
          (t (wt-nl "if((" loc ")!=Cnil)")))
    (unless (eq loc t) (wt "{"))
    (unwind-no-exit label)
    (wt-nl) (wt-go label)
    (unless (eq loc t) (wt "}")))
  )

(defun set-jump-false (loc label)
  (unless (eq loc t)
    (cond ((null loc))
          ((eq (car loc) 'INLINE-COND)
           (wt-nl "if(!(")
           (wt-inline-loc (third loc) (fourth loc))
           (wt "))"))
          ((member (car loc) '(CALL CALL-LOCAL) :test #'eq)
	   (wt-nl loc ";")
           (wt-nl "if(VALUES(0)==Cnil)"))
          (t (wt-nl "if((" loc ")==Cnil)")))
    (unless (null loc) (wt "{"))
    (unwind-no-exit label)
    (wt-nl) (wt-go label)
    (unless (null loc) (wt "}")))
  )

(defun c1ecase (args) (c1case args t))  

(defun c1case (args &optional (default nil))
  (when (endp args) (too-few-args 'case 1 0))
  (let* ((info (make-info))
         (key-form (c1expr* (car args) info))
         (clauses nil))
    (dolist (clause (cdr args))
      (cmpck (endp clause) "The CASE clause ~S is illegal." clause)
      (case (car clause)
            ((NIL))
            ((T OTHERWISE)
             (when default
                   (cmperr (if (eq default 't)
                               "ECASE had an OTHERWISE clause."
                               "CASE had more than one OTHERWISE clauses.")))
             (setq default (c1progn (cdr clause)))
             (add-info info (second default)))
            (t (let* ((keylist
                       (cond ((consp (car clause))
                              (mapcar #'(lambda (key) (if (symbolp key) key
                                                          (add-object key)))
                                      (car clause)))
                             ((symbolp (car clause)) (list (car clause)))
                             (t (list (add-object (car clause))))))
                      (body (c1progn (cdr clause))))
                 (add-info info (second body))
                 (push (cons keylist body) clauses)))))
    (list 'CASE info key-form (nreverse clauses) (or default (c1nil)))))

(defun c2case (key-form clauses default
               &aux loc (*inline-blocks* 0))
  (setq key-form (second (first (inline-args (list key-form)))))
  (if (and (consp key-form) (eq (car key-form) 'VAR))
      (setq loc key-form)
      (progn
	(setq loc (list 'LCL (next-lcl)))
	(wt-nl "{object " loc "= " key-form ";")
	))
  (dolist (clause clauses)
    (let* ((label (next-label))
           (keylist (first clause))
           (local-label nil))
      (do ()
          ((<= (length keylist) 5))
        (unless local-label (setq local-label (next-label)))
        (wt-nl "if(")
        (dotimes (i 5)
          (cond ((symbolp (car keylist))
                 (wt "(" loc "== ")
                 (case (car keylist)
                   ((T) (wt "Ct"))
                   ((NIL) (wt "Cnil"))
                   (otherwise (wt "VV[" (add-symbol (car keylist)) "]")))
                 (wt ")"))
                (t (wt "eql(" loc ",VV[" (car keylist) "])")))
          (when (< i 4) (wt-nl "|| "))
          (pop keylist))
        (wt ")")
        (wt-go local-label))

      (wt-nl "if(")
      (do ()
          ((endp keylist))
        (cond ((symbolp (car keylist))
               (wt "(" loc "!= ")
               (case (car keylist)
                 ((T) (wt "Ct"))
                 ((NIL) (wt "Cnil"))
                 (otherwise (wt "VV[" (add-symbol (car keylist)) "]")))
               (wt ")"))
              (t (wt "!eql(" loc ",VV[" (car keylist) "])")))
        (unless (endp (cdr keylist)) (wt-nl "&& "))
        (pop keylist))
      (wt ")")
      (wt-go label)
      (when local-label (wt-label local-label))
      (let ((*unwind-exit* (cons 'JUMP *unwind-exit*))) (c2expr (cdr clause)))
      (wt-label label)))

  (if (eq default 't)
      (wt-nl "FEerror(\"The ECASE key value ~s is illegal.\",1," loc ");")
      (c2expr default))

  (unless (eq loc key-form) (wt "}"))
  (close-inline-blocks))

;;; ----------------------------------------------------------------------

(setf (get 'if 'c1special) #'c1if)
(setf (get 'if 'c2) #'c2if)
(setf (get 'and 'c1) #'c1and)
(setf (get 'and 'c2) #'c2and)
(setf (get 'or 'c1) #'c1or)
(setf (get 'or 'c2) #'c2or)

(setf (get 'jump-true 'set-loc) #'set-jump-true)
(setf (get 'jump-false 'set-loc) #'set-jump-false)

(setf (get 'case 'c1) #'c1case)
(setf (get 'ecase 'c1) #'c1ecase)
(setf (get 'case 'c2) #'c2case)
