% This file containts the predicates needed to process "functional"
% contraints like "order" (phonology) and "combine" (semantics) which may 
% appear within signs.  Some constraints should not be processed until all 
% Expectations associated with an edge have been satisfied.

% process(Constraints_In, Constraints_Out, Expectations)

process([order(A,B)|Rest], Result, []) :- !,
   call(order(A,B)),
   process(Rest, Result, []).

process([collect(A,B,C,D,E)|Rest], Result, []) :- !,
   call(collect(A,B,C,D,E)),
   process(Rest, Result, []).

process([collect(A,B,C,D)|Rest], Result, []) :- !, % Rogers compatibility
   call(collect(A,B,C,D)),
   process(Rest, Result, []).

process([combine(A,B,C,D)|Rest], Result, []) :- !,
   call(combine(A,B,C,D)),
   process(Rest, Result, []).

process([select1(A,B)|Rest], Result, []) :- !, % Select a single element
   call(select1(A,B)),
   process(Rest, Result, []).

process([nfp(X,Y,Z,W,V,U)|Rest], Result,  []) :- !,
   call(nfp(X,Y,Z,W,V,U)),
   process(Rest,Result,[]).

process([C|Rest], [C|Result], []) :- !, process(Rest, Result, []).

process(C, C, _).

% Some function calls can be processed immediately, and thus we define this
% version of "process" which does not worry about whether they are still 
% expectations or not.

process([concat(A,B,C)|Rest], Result) :- !,
   call(concat(A,B,C)),
   process(Rest, Result).

process([C|Rest], [C|Result]) :- process(Rest, Result).

process([], []).


% The Constituent Order Principle, references the "order" function 

% This version of "order" function requires:
%	syn:loc:lex,
%	syn:loc:head:maj:n

order(/*Dtrs*/ [[_H,Head], [_C,Comps], [_A,Adjs], [_F,Fillers]], Phonology) :-
  order(Head, Comps, Adjs, Fillers, Phonology), !.

order(Head, Comps, [], [], Phonology) :- %%%% Head and Comp Daughters
  path(Head, syn:loc:lex, plus) ->        % if it is a lexical head
    order_comps(Comps, Result),           % then use the comp_dtr ordering
    path(Head, phon, HeadP),              % get the phonology of the head
    concat(HeadP, Result, Phonology)      % and have it precede its comps
  ; order_comps([Head|Comps], Phonology). % otherwise treat head as most
                                          %  oblique complement

order(Head, [], [Adj], [], Phonology)  :- %%%% Head and Adjunct Daughter
  path(Head, phon, HeadP),                 % (should be modified if adverbs
  path(Adj, phon, AdjP),                   % are to be added)
  ( (path(Adj, syn:loc:lex, plus)          % if lexical adjunct or 
    ; path(Adj, syn:loc:head:maj:n, plus)  %  an adjective or a noun
    ) -> concat(AdjP, HeadP, Phonology)    % have it precede its head
  ; concat(HeadP, AdjP, Phonology) ).      % otherwise have it follow

order(Head, [], [], [Fill], Phonology):- %%%% Head and Filler Daughter
   path(Head, phon, HeadP),
   path(Fill, phon, FillP),
   (path(Head,syn:loc:lex,minus) ->	% lexical head precedes filler
      concat(FillP, HeadP, Phonology)  
    ; concat(HeadP, FillP, Phonology)).	% non-lexical follows it

order_comps([], []).
  
order_comps([Sign|Comps], Phonology) :-
  path(Sign, phon, Phon),
  order_comps(Comps, Result),
  concat(Result,Phon,Phonology).

% Semantics Principle - essentially, an implementation of 
%                       successively-combine-semantics and
%                       collect indices
%
% Usually very grammar-specific; this version of "combine" function expects:
%	sem:cont:quant
%	sem:cont:rlt
%	sem:cont:scope

% When no adjuncts or complements, return the CONT of the head

combine(Sem, [], [], Sem) :- !.   

% When there is a quantified complement, return a quantified sign 
% similar to the sign described in Pollard & Sag, 1987, p.109
% In our simple implementation, we do not explicitly store the "type"
% of a sign.  Thus, the only way that we can determine if a sign is
% a quantified sign or not, is by whether or not it has an atomic value for
% its sem|cont|quant attribute

combine(HC, [C|Comps], [], Result) :-   % if comp. dtrs then no adj. dtrs
  path(C,sem:cont:quant,Q),
  nonvar(Q),                            % requires sem:cont:quant
  !,                                    %          sem:cont:scope
  path(C,sem:cont,CC),                  %    and   sem:cont:rlt
  (path(HC,rlt,Reln),var(Reln),CC=HC -> % if contents unify and head is not a 
    R=CC                                % VP, then this is the result  
  ; copy_term(Result,R),                % otherwise we create sign for sem:cont
    path(R,quant,CC),                   % and combine contents of the head 
    path(R,scope,HC),                    % and the complement
    path(R,index:var,X),		% index:var is used for referencing...
    path(HC,index:var,X)
  ), 
  combine(R, Comps, [], Result).        % proceed for other complements

/*

Mon Mar 29 14:10:36 1993  Changed according to Fred's bug description and fix:

There is a bug in the constraints.pl file that will cause the system to
hang when analysing some sentences.  Instead of reading

  combine(Sem, [_C|Comps], [], Sem) :- !,
    combine(Sem, Comps, [], Sem).

the third clause of combine should be

  combine(Sem, [_C|Comps], [], Result) :- !,
    combine(Sem, Comps, [], Result).

The bug caused a cyclic datastructure to be created, and whenever 
you attempt to assert a cyclic structure to the database, the system
will hang.

- Fred


*/

combine(Sem, [_C|Comps], [], Result) :- !,
  combine(Sem, Comps, [], Result).

combine(_, [], [A], AC) :-  % content of the head-adjunct structure is the
  path(A,sem:cont,AC).      % content of the adjunct

% For collecting indices (concatenate sem:indices values of the daughters)

collect(I, [], [], [], I) :-!.           % requires all lexical entries
                                         % to have a list as a value of
collect(HI, [], [A], [], NewI):-!,	 % sem:indices
  path(A, sem:indices,I),
  concat(I,HI,NewI).

collect(HI, [], [], [F], NewI):-!,	
  path(F, sem:indices,I),
  concat(I,HI,NewI).
 
collect(Indices, [C|Comps], [], [], Result) :- 
  path(C, sem:indices, I),
  concat(I,Indices,NewI),
  collect(NewI, Comps, [], [], Result).

% For selecting fillers...
%
% select(+,-)    return a sublist of length one

select([X|Rest],L) :- L=[X] ; select(Rest,L).

% For selecting a single adjunct

select1([X|Rest],T):- T=X ; select1(Rest,T).

% Nonlocal Feature Principle -- nfp/6
%
% INHER value of the phrase for each NONLOC feature becomes the "union"
% of INHER values of the daughter less TO_BIND value of the head

% This version expects these list-values features:
%	syn:non_loc:inher:slash
%	syn:non_loc:inher:rel
%	syn:non_loc:inher:que
% Structure of syn:non_loc:to_bind is the same, expect that lists can have
% at most one element.

nfp(HInh, HToB, [], [Adj], [], PInh):-!,	% nfp/6
  nfp([Adj], HInh, HToB, PInh).

nfp(HInh, HToB, [], [], [F], PInh):-!,
  nfp([F], HInh, HToB, PInh).

nfp(HInh, HToB, Comps, [], [], PInh):-      
  nfp(Comps, HInh, HToB, PInh).

nfp([], HInh, _, HInh):-!.  % head is the only daughter (rule 2) - no "binding"

nfp(Dtrs, HInh, HToB, PInh):- 			% nfp/4
  get_inh(Dtrs, DInh),
  conc_val(HInh,DInh,Inh),
  nfp(Inh,HToB,PInh).

get_inh([],[[slash,[]], [rel,[]], [que,[]]]).
get_inh([D|Dtrs],Inh):-
  get_inh(Dtrs,DtrInh),
  path(D, syn:non_loc:inher, DInh),  
  conc_val(DInh,DtrInh,Inh).

conc_val([],[],[]).
conc_val([[F,V1]|T1], [[F,V2]|T2], [[F,V3]|T3]):- % INHER and TO_BIND must
  concat(V1,V2,V3),                               % have 3 subfeatures
  conc_val(T1,T2,T3).

nfp([],[],[]).					% nfp/3
nfp([[F,I]|T1], [[F,T]|T2], [[F,P]|T3]):-
  union_less(T,I,P),
  nfp(T1,T2,T3).

union_less([],I,I).		% TO-BIND element must unify with at least one
union_less([E],[E|T],T):-!.     % member of INHER; otherwise fails!
union_less([E],[_|T],T1):-
  union_less([E],T,T1).

