% TreeTool Interface - to be most portable, create a file that will be
%                      used as input to the TreeTool program.
%
%         Note: "treetool" was formerly called "tugtree", which explains
%               the naming convention.

gparse :-                         % Pass TreeTool just the valid parse.
  open('.HPSGchart',write,Stream),
  (clause(edge(_,_,_),_) -> dumpit(parse,Stream)
  ; write('Cannot find reconstructed sign, phrasal flag should be on.'), nl
  ). 

gchart :-                         % Pass TreeTool the whole chart.
  open('.HPSGchart',write,Stream),
  dumpit(chart,Stream).

dumpit(parse, Stream) :-
  edge(PSign, Constituents, Num),    % created by write_psign
  dump1(PSign, Constituents, Num, Stream),
  fail.

% Reconstruct Phrasal Signs if possible

dumpit(chart, Stream) :-
  edge(_, Sign, Constituents,_,_,_,Num),
  makePSign(Sign, Constituents, []),
  dump1(Sign, Constituents, Num, Stream),
  fail.

dumpit(_,Stream) :- 
  flush_output(Stream), 
  close(Stream),
  unix(shell('treetool -i .HPSGchart &')).   % Sicstus and Quintus

dump1(PSign, Cs, Num, Stream) :-
  tugtree(PSign,Tree),
  write(Stream,fa_structure(Num,Tree,[],Cs)),
  write(Stream,'.'),
  nl(Stream), !.

% Transforms phrasal sign into the form accepted by TreeTool
% this version expects the following attributes:
%	syn:loc:subcat
%	syn:loc:head
%	syn:non_loc:inher
%	sem:cont
% Could be modified!

tugtree(PSign,Tree) :-
  path(PSign,phon,Phon),
  give_head(PSign,Head),
  abbrev_head(Head,AHead),
  path(PSign,dtrs,Dtrs),
  tugdtrs(Dtrs,Subtrees),
  path(PSign,syn:loc:subcat,SubCat),
  abbrev_subcat(SubCat,ASub),
  path(PSign,syn:non_loc:inher, Inh),
  abbrev_inh(Inh,AInh),
  path(PSign,sem:cont,Cont),
  abbrev_cont(Cont,ACont),
  Tree =.. [tree, Phon:AHead:ASub:AInh:ACont | Subtrees],
  !.

tugdtrs([],[nil]).

tugdtrs(PSign, [H|C]) :-
  path(PSign, head_dtr, HDtr),
  tugtree(HDtr, H),
  path(PSign, comp_dtrs, CDtrs),
  path(PSign, adj_dtrs, ADtrs),
  path(PSign, filler_dtrs, FDtrs),
  concat(CDtrs, ADtrs, Dtrs),
  concat(Dtrs,FDtrs,Dtrs1),
  tugdtrlist(Dtrs1, C).

tugdtrlist([],[]).

tugdtrlist([Dtr|Rest], [D|R]) :- 
  tugtree(Dtr,D),
  tugdtrlist(Rest,R).

%  Used to abbreviate signs for display on terminal on when passed to TreeTool.
%  Very messy and very much a hack!

abbrev_head(Var, Var) :- var(Var), !.

abbrev_head([[_,Value]|Rest],[V|R]) :- !,
  abbrev_head(Value,V),
  abbrev_head(Rest,R).

abbrev_head(X, X).

abbrev_subcat([], []).

abbrev_subcat([PSign|Rest], [H|R]) :-
  give_head(PSign, Head),
  abbrev_head(Head, H),
  abbrev_subcat(Rest, R).

abbrev_inh(Inh,AInh):-
  collect_inh(Inh,L),
  AInh =.. [inh|L].

collect_inh([],[]).

collect_inh([[_F,V]|T], [V|T1]) :-
  collect_inh(T,T1).

% abbreviates semantic "content"; requires the following sem:cont subfeatures
%	scope, quant, rlt, index:var, index:restriction

abbrev_cont(Cont,scope(AC)):-
  path(Cont,scope,S),
  nonvar(S),!,
  abbrev_cont(S,AC).

abbrev_cont(Cont,AC):-
  path(Cont,rlt,Rlt),
  nonvar(Rlt),!,
  Rlt = [[_,Name]|Args],
  get_args(Args,A),
  AC =.. [Name|A].

abbrev_cont(Cont,quant(Q,V)):-
  path(Cont,quant,Q),
  atomic(Q),!,
  path(Cont,index:var,V).

abbrev_cont(Cont,AC):-
  path(Cont,index:restriction,Rst),
  nonvar(Rst),!,
  Rst = [[_,Name]|Args],
  get_args(Args,A),
  AC =.. [Name|A].

abbrev_cont(_,no_semantics).

get_args([],[]).

get_args([[_,A]|Args],[A|As]):-
  get_args(Args,As).

abbrev_slash([],[]).

abbrev_slash([S],H):-!,
  give_head(S,H).

abbrev_slash([S1,S2],[H1,H2]):- % SLASH list may not have more that 2 elements
  give_head(S1,H1),
  give_head(S2,H2).

give_head(I_Sign,TrimmedHead) :-
  path(I_Sign,syn:loc:head,Head),
  trim_head(Head,TrimmedHead).

trim_head([],[]).
trim_head([X|_],'_') :- var(X), !.   
trim_head([[_,Value]|Rest], TrimmedRest) :- 
  var(Value),!,
  trim_head(Rest, TrimmedRest).
trim_head([[Attribute,_]|Rest],TrimmedRest):-         % some attributes are
  member(Attribute,[heads,adjuncts,agr,inv,aux]),!,   % always skipped
  trim_head(Rest,TrimmedRest).
trim_head([[Attribute,lset(D,X)]|Rest],[[Attribute, Y]|TrimmedRest]) :-
  list2set(D,X,Y),
  trim_head(Rest, TrimmedRest).
trim_head([[Attribute,Value]|Rest],[[Attribute, Value]|TrimmedRest]) :-
  atomic(Value),
  trim_head(Rest, TrimmedRest).
trim_head([[form,Stuff]|Rest],[Wheat|TrimmedRest]) :-
  determine_form(Stuff,Wheat),!,
  trim_head(Rest, TrimmedRest).
trim_head([[Attribute,Value]|Rest],[[Attribute, TrimmedValue]|TrimmedRest]) :-
  !,
  trim_head(Value, TrimmedValue),
  trim_head(Rest, TrimmedRest).
trim_head(X,X).

determine_form([],[]).
determine_form([[_,Value]|_],[form,Value]) :-
  var(Value).
determine_form([[_,[]]|OtherForms],Wheat) :-
  !,determine_form(OtherForms,Wheat).
determine_form([[Attribute,Value]|_],[Attribute,Value]).
