/*

Top level printing routine:


*/


pdag(D0, STDForm) :- 
    findunifs(D0, VT, Root), !, 
    unfold_value_tree(VT),
    rename_indexes(Root),
    mike2standard_format(Root, STDForm).


/*

Parameters for ordering and deletion of feature structures

*/

% Things we don't want to see

shadowed_attribute(pos).
shadowed_attribute(all).
shadowed_attribute(beg).
shadowed_attribute(end).

% The last two probably redundant

% Preferred ordering of attributes 
% We state a single ordering for all (nonshadowed) attributes assuming 
% that this will be good for any FS.

attribute_ordering([phon, syn, sem, dom, dtrs,
		    fun, argdtrs, 
		    maj, case, head, lex, args, 
		    loc, nonloc]).


/*

extract_values(Atts, AVM, NewAVM)

NewAVM is just like AVM, including wrt conindexing except that it is
only specified for attributes in Atts.

The body of the first clause is there because of the need to maintain
coindexing.

*/

extract_values([], AVM, NewAVM) :- 
    (var(AVM) -> AVM = NewAVM;
    		tail(AVM, NewAVM)).
extract_values([H|T], AV, [H=Value|AV1]) :-
    eccs_delete(H = Value, AV, AV2),
    extract_values(T, AV2, AV1).

    

/* fs_labels(FS, Ls)

FS has Ls as it labels at the top level

*/

fs_labels(Var, []) :- 
    var(Var), !.
fs_labels([], []) :- !.
fs_labels([F=_|T], R) :-
    shadowed_attribute(F), !,
    fs_labels(T, R).
fs_labels([F=_|T], [F|R]) :-
    fs_labels(T, R).


/* order_labels(Ls, L1s)

L1s are labels sorted by the attribute ordering stated above.

*/

order_labels(Ls, L1s) :-
    attribute_ordering(Ord),
    order_labels(Ord, Ls, L1s).



order_labels(_, [], []) :- !.  
order_labels([], Ls, Ls) :- !.
order_labels([L|Ord], Ls, [L|L1s]) :-
    eccs_delete(L, Ls, L2s), !,
    order_labels(Ord, L2s, L1s).
order_labels([L|Ord], Ls, L1s) :-
    order_labels(Ord, Ls, L1s).

/*

findunifs --- the right way!

JC Wed Jan 30 11:30:53 1991

At last, I've worked out how to do this properly.  The real insight
here is that what drives the whole of this process is the values that
we find in graphs or terms, it is they that drive the computation,
rather than the structure of the graph or term.  This means that we
use the following strategy given some object with reentrancies
implicit in variable sharing, we build a binary tree of all values
(variable or atomic) found in that object, using the prolog standard
ordering of terms to drive the tree building.  As we do this, we can
associate arbitrary info with the values.  In particular we can i)
create partial output structures on the fly (so folding in the process
of ordering attributes) and ii) put paths into equivalence classes
(allowing some later process to determine the ``best path'' to some
value).

The first group of predicates here do the tree building.  One could
imagine folding them into findunifs for efficiency.

*/

/*

binary_tree_of_values(Value, VarTree, Structure)

Value is positioned at an appropriate point (given by the Prolog
standard ordering) in VarTree and associated in that tree with
Structure.

*/

binary_tree_of_values(Value, VarTree, Struct) :-
    var(VarTree), !,			% never been here before 
    VarTree = tree(Value,Struct, _NewTree).
binary_tree_of_values(Value, tree(Val1, Struct1, SubTree), Struct) :-
    compare(Rel, Value, Val1),
    order_tree_of_values(Rel, Value, tree(Val1, Struct1, SubTree), Struct).

order_tree_of_values((=), Value, tree(Value, Struct, _), Struct) :- !,
    (var(Value); atomic(Value)), !.	% ?? just for sanity
order_tree_of_values((<), Value, tree(_, _, t(Pre, _)), Struct) :- !,
    binary_tree_of_values(Value, Pre, Struct).  
order_tree_of_values((>), Value, tree(_, _, t(_, Succ)), Struct) :- !,
    binary_tree_of_values(Value, Succ, Struct).

/*

findunifs(AVM, VarTree)

VarTree is the binary tree of values for AVM

*/

findunifs(Value, VarTree, Root) :-
    findunifs(Value, Root, VarTree, []).

/*

findunifs(Value, Root, VarTree, Path)

VarTree is the binary tree of values in AVM.  Path is the reversed
list of attributes which, followed from the root of the original dag
leads to Value. Root is a partial specification of the output form of
AVM.

*/

/* On finding a value, insert it in the tree and add the path to the
list of paths.  In the case of variables, leave a new variable hanging
around.


The fourth clause here seems to me to be needed in order to catch
cases where whole sequences are identified.  It might be commented in,
if this was deemed to be appropriate behaviour. Depends on whether one
wishes to be explicit about the extensionality of operators like
concat and union, I guess.

*/


findunifs(Var, Root, VarTree, Path) :-
    var(Var), !,
    binary_tree_of_values(Var, VarTree, Paths >> NewVar),
    eccs_note_path(path(Root, Path), Paths),
    findconstraints(Var, NewVar, VarTree).
findunifs([], Root, VarTree, Path) :-
    !,
    binary_tree_of_values('$NULL_SEQ', VarTree, Paths >> '$NULL_SEQ'),
    eccs_note_path(path(Root, Path), Paths).
findunifs(Atom, Root, VarTree, Path) :-
    atomic(Atom), !,
    binary_tree_of_values(Atom, VarTree, Paths >> Atom),
    eccs_note_path(path(Root, Path), Paths).  
/* findunifs(AVM, Root, VarTree, Path) :-
 * binary_tree_of_values(AVM, NewRoot, Paths >> NewRoot),
 * (Paths = [] -> fail; eccs_note_path(path(Root, Path), Paths)), !.  
*/ 
findunifs([H|T], Root, VarTree, Path) :-
    var(H), !,
    binary_tree_of_values(H, VarTree, Paths >> NewRoot),
    eccs_note_path(path(Root, ['$HEAD'|Path]), Paths),
    findunifs_seq(T, Root, VarTree, ['$TAIL'|Path]).
findunifs([A=V|R], Root, VT, Path) :-
    !, tail([A=V|R], Tail),
    binary_tree_of_values(Tail, VT, Paths >> NewRoot),
    eccs_note_path(path(Root, Path), Paths),
    (fu_been_here_before(Paths) -> true;
    		order_and_filter_attributes([A=V|R], Filtered, NewRoot),
    findunifs_av_pairs(Filtered, NewRoot, VT, [])). % reset path
findunifs([H|T], Root, VT, Path) :-
    findunifs_seq([H|T], Root, VT, Path).

fu_been_here_before(Paths) :- \+ Paths = [_].

findunifs_av_pairs(Var, Root, _, _) :-
    var(Var), !.
findunifs_av_pairs([A=V|R], Root, VT, Path) :-
    findunifs(V, Root, VT, [A|Path]),
    findunifs_av_pairs(R, Root, VT, Path).

findunifs_seq(Var, Root, VT, Path) :-
    var(Var), !, 
    binary_tree_of_values(Var, VT, Ps >> NewVar),
    eccs_note_path(path(Root, Path), Ps).
findunifs_seq([], Root, VT, Path) :-
    !,
    binary_tree_of_values('$END_SEQ', VT, Ps >> '$END_SEQ'),
    eccs_note_path(path(Root, Path), Ps).
findunifs_seq([H|T], Root, VT, Path) :-
    findunifs(H, Root, VT, ['$HEAD'|Path]),
    findunifs_seq(T, Root, VT, ['$TAIL'|Path]).

/*

Make a note of the current path in the list of paths associated with
each node.

*/ 
eccs_note_path(path(Root, Path), Var) :-
    var(Var), !, Var = [path(Root, Path)|_].  
eccs_note_path(path(Root, Path), [path(R1, P1)|_]) :-
    Root == R1, Path == P1, !.  
eccs_note_path(path(Root, Path), [_|T]) :-
    eccs_note_path(path(Root, Path), T).


/*

findconstraints(NewVar, VarTree)

*/

findconstraints(Var, NewVar, VarTree) :-
    frozen(Var, Goal), !,
    analyse_constraint(Goal, NewVar, VarTree).
findconstraints(Var, NewVar, VarTree).

/*

Thu Aug  6 11:43:54 1992 JC

With modules in SICStus, frozen goals may be of the form

Module:Goal

rather than just Goal.  The third clause is added to take 
account of this.  

*/

analyse_constraint(true, NewVar, VarTree) :- !.
analyse_constraint(dif(_V, Neg), NewVar, VarTree) :- !,
    atomic(Neg),
    NewVar = '$NEG'(Neg).
analyse_constraint(_Module:dif(_V, Neg), NewVar, VarTree) :- 
    atomic(Neg),
    NewVar = '$NEG'(Neg).

/*

unfold_value_tree(VT)

The statements in VT are evaluated and, as a side effect, build give
rise to the tree associated with the empty path from Root, cf
findunifs above and follow_path below.

*/

unfold_value_tree(empty_tree) :- !.  
unfold_value_tree(tree(_, Ps >> SubDag, Tree)) :-
    value_format_and_place(Ps, SubDag),
    unfold_value_tree(Tree).
unfold_value_tree(t(Pre, Succ)) :- 
    unfold_value_tree(Pre),
    unfold_value_tree(Succ).


/*

This is where we decide whether we coindex and where coindexed
insatiated structures should be placed.

*/

value_format_and_place([P], Var) :-	% single path to var
    var(Var),
    !, 
    P = path(Dag, Path),
    follow_path(Path, Dag, '$UNINSTANTIATED').
value_format_and_place([P], Value) :- !, % single path to nonvar 
    P = path(Dag, Path),
    follow_path(Path, Dag, Value).
value_format_and_place(Ps, Var) :-	% many paths to var
    var(Var),
    !,
    follow_paths(Ps, '$INDEX'(N)).
value_format_and_place(Ps, Atom) :-	% many paths to atom 
    atomic(Atom), !,
    follow_paths(Ps, Atom).
value_format_and_place(Ps, Value) :-	% many paths to complex AVM
    order_eq_paths(Ps, [H|T]),
    follow_paths([H], '$INDEX'(N, Value)),
    follow_paths(T, '$INDEX'(N)).

/*

follow_paths(Ps, Value)

Follow the paths Ps given in the form path(Root, RevPath) to Value

*/



follow_paths([], _) :- !.
follow_paths([P|Ps], Value) :-
    P = path(Dag, Path),
    follow_path(Path, Dag, Value),
    follow_paths(Ps, Value).

/*

follow_path(Atts, Dag, Value)

following the reversed path specified by Atts in Dag leads to Value.
The main wrinkle here has to do with instantiation.  The ordering of
attributes within all terms that ultimately instantiate Dag has
already been defined by order_and_filter_attributes in findunifs
above.  However, we can't tell what this ordering is and have to wait
until Dag is appropriately instantiated before we proceed.  Hence the
freeze in the auxiliary predicate merge_value_into_dag.  If we don't
do this, we will screw up because we can instantiate an avm whose
ordering doesn't matter (under the user defined attribute ordering) in
two ways which are graph unification equivalent but not term
unifiable.  We know that the frozen instantiation conditions will
ultimately be met, because under order_and_filter_attributes, all
attributes that figure at a particular level of structure are
represented in the partial output generated by that routine.


*/

follow_path(RevPath, Dag, Value) :-
    eccs_reverse(RevPath, Path),
    follow_path1(Path, Dag, Value).

follow_path1([], Dag, Dag1) :- !,
    merge_value_into_dag(Dag, Dag1).
follow_path1(['$HEAD'|R], [Dag|_], Value) :- !,
    follow_path1(R, Dag, Value).  
follow_path1(['$TAIL'|R], [_|Dag], Value) :- !,
    follow_path1(R, Dag, Value).
follow_path1([Att|Atts], Dag, SubDag) :-
    delayed_memberchk(Att = Val, Dag),
    follow_path1(Atts, Val, SubDag).


/*

merge_value_into_dag(DPreserve, D)

Unify D and DPreserve, ensuring that no modification is done to the
ordering given in DPreserve.

*/



merge_value_into_dag(Var, Dag) :-
    var(Var), !,
    Dag = Var.
merge_value_into_dag(D, []) :- !.
merge_value_into_dag(D, [A=Value|R]) :- !,
    merge_av_pairs_into_dag(D, [A=Value|R]).
merge_value_into_dag([H|T], D) :-
    merge_value_into_dag_list([H|T], D).
merge_value_into_dag(Atom, Atom) :-
    atomic(Atom).

merge_av_pairs_into_dag(D, [A=Value|R]) :-
    delayed_memberchk(D, A = Value),
    merge_av_pairs_into_dag(D, R).

delayed_memberchk(A = Value, D) :-
    freeze(D, ( D = [A1 = V1|R],
    (A ==A1 -> Value = V1; 
	       delayed_memberchk(A = Value, R)))).

merge_value_into_dag_list(D, []) :- !.
merge_value_into_dag_list([H|T], [F|R]) :-
    freeze(F, merge_value_into_dag(H, F)),
    merge_value_into_dag_list(T, R).





/*

order_and_filter_attributes(Old, New, OutAVM)

New is a smaller version of Old, with any shadowed attributes removed
from the top level. OutAVM is a partial specification of the feature
structure that will be output.  We know all the feature structures
that figure at this level of structure (wherever it occurs). One thing
we can't assume is that we always end up with a unique ordering for
any given permutation of labels.  This is because there may be labels
which are unordered with respect to one another. The ordering we do
get will not contradict any of the attribute ordering statements.

*/

order_and_filter_attributes(Old, New, OutAVM) :-
    fs_labels(Old, Labels), 
    order_labels(Labels, LOrdered),	% ordered and without shadowed attributes 
    extract_values(LOrdered, Old, New),
    build_partial_output(LOrdered, OutAVM).


build_partial_output([], []) :- !.
build_partial_output([Att|Atts], [Att = _|R]) :-
    build_partial_output(Atts, R).
  

/*     

order_eq_paths(Ps, Ordered)

the preferred path of the Ps is the head of Order, the remainder of Ps
being Order's tail.

Note that this is not as bad as the equivalent sorting problem, as we
only have to find the most preferred and reorder the list of that is
comes first rather than doing that *and* having to ensure that the
tail of the list is also sorted.  In other words, we can always get by
with pairwise comparisons.

*/

order_eq_paths([H|T], [Best|Rest])  :-
    order_eq_paths(T, H, [], Rest, Best).

order_eq_paths([], P, Rest, Rest, P) :- !.
order_eq_paths([H|Tail], SoFar, Rest0, Rest, Best) :-
    better_path(H, SoFar), !,
    order_eq_paths(Tail, H, [SoFar|Rest0], Rest, Best).
order_eq_paths([H|Tail], SoFar, Rest0, Rest, Best) :-
    order_eq_paths(Tail, SoFar, [H|Rest0], Rest, Best).


better_path(path(_, Better), path(_, Worse)) :- 
    eccs_memberchk(dom, Better), fail.
better_path(path(_, Better), path(_, Worse)) :- 
    eccs_memberchk(argdtrs, Worse), fail.
better_path(path(_, Better), path(_, Worse)) :- 
    eccs_memberchk(fun, Worse), fail.
better_path(path(_, Better), path(_, Worse)) :- 
    eccs_memberchk(argdtrs, Better), !.
better_path(path(_, Better), path(_, Worse)) :- 
    eccs_memberchk(fun, Better), !.

    

/*

rename_indexes(Dag)

is now just a process of instantiation.

*/

rename_indexes(D) :-
	rename_indexes(D, 0, _).

rename_indexes(X, N, N) :-
        atomic(X), !.
rename_indexes(X, N, N) :-
        var(X), !.
rename_indexes('$NEG'(_), N, N) :- !.	% assume negations are all atomic
rename_indexes('$INDEX'(I), N, N) :-			% seen
    	nonvar(I), !.
rename_indexes('$INDEX'(I, _), N, N) :-			% seen
    	nonvar(I), !.
rename_indexes('$INDEX'(I), N0, N) :-	% unseen
    	var(I), !,
	I is N0 +1,
	I = N.
rename_indexes('$INDEX'(I, L), N0, N) :-		% unseen
    	var(I), !,
	I is N0 +1,
	rename_indexes(L, I, N).
rename_indexes([L = Val0|T0], N0, N) :- !,
        rename_indexes(Val0, N0, N1),
	rename_indexes(T0, N1, N).
rename_indexes([H0|T0], N0, N) :-
	rename_indexes(H0, N0, N1),
        rename_indexes(T0, N1, N).

/*
print(Dag, Tab, Col)						     

Tab over Tab spaces and then print Dag.  Col is the running column count.
Dag is the result of findunifs/2 and so is ground.
*/

pp_print(R) :-
	print(R, 0, 0), !.

print(X, _, _) :-
        atomic(X), !,
	write_atomic(X).
print(X, _, _) :-
        var(X), !,
	write(X).
print('$NEG'(N), Tab, _) :- !,		% assume negations are atomic
    tab(Tab),
    write('~'), 
    write(N).
print('$INDEX'(N), Tab, _) :-
	tab(Tab),
	write(@), write(N).
print('$INDEX'(N, L), Tab, Col) :-
	tab(Tab),
	write(N), write('@'),
	prlength(N, I),
	NewCol is Col + 1 + I,
	print(L, 0, NewCol).
print([Label = Val|T], Tab, Col) :- !,
        print_feature_structure([Label = Val|T], Tab, Col).
print([H|T], Tab, Col) :-
        print_sequence([H|T], Tab, Col).

/*

The first of these is prettier but is illegal reader syntax.

*/

%write_atomic('$NULL_SEQ') :- !,
%    write('<>').
write_atomic('$NULL_SEQ') :- !,
    write('[]').
write_atomic('$UNINSTANTIATED') :- !,
    write('[]').
write_atomic(X) :- !,
    write(X).

print_feature_structure([Label = Val|T], Tab, Col) :-
	tab(Tab), write('['),
	NCol is Col + 1,
	print_feature_structure0([Label = Val|T], 0, NCol),
        write(']').

print_feature_structure0([Label = Val], Tab, Col) :- !,
	tab(Tab), write(Label), 
	write(' = '),
	prlength(Label, N),
	NTab is Col + N + 3,
	print(Val, 0, NTab).
print_feature_structure0([Label = Val|T], Tab, Col) :-
	tab(Tab), write(Label), 
	write(' = '),
	prlength(Label, N),
	NTab is Col + N + 3,
	print(Val, 0, NTab),
	write(','), nl,
	print_feature_structure0(T, Col, Col).

print_sequence(L, Tab, Col) :-
        list_of_atoms(L), !,
	tab(Tab), write('['),
	write_list_of_atoms(L),
        write(']').
print_sequence(L, Tab, Col) :-
        list_of_NODEs(L), !,
	tab(Tab), write('['),
	write_list_of_NODEs(L),
        write(']').
print_sequence([H|T], Tab, Col) :-
	tab(Tab), write('['),
	NCol is Col + 1,
	write_list_of_signs([H|T],0,NCol),
        write(']').

list_of_atoms([]).
list_of_atoms(Atom) :-
    atomic(Atom), !.			% allow for atoms in tails
list_of_atoms([H|T]) :-
  atomic(H),
  list_of_atoms(T).


list_of_NODEs([]).
list_of_NODEs('$END_SEQ').
list_of_NODEs('$INDEX'(N)).
list_of_NODEs(['$INDEX'(N)|T]) :-
  list_of_NODEs(T).

write_list_of_atoms([]).
write_list_of_atoms([H|'$END_SEQ']) :-
    write_atomic(H).
write_list_of_atoms([X|'$INDEX'(N)]) :- !,
  write_atomic(X), write('|@'), write(N).
write_list_of_atoms([X]) :- !,
  write_atomic(X).
write_list_of_atoms([H|T]) :-
  write_atomic(H), write(', '),
  write_list_of_atoms(T).



write_list_of_signs([X], Tab, Col) :-
	print(X, Tab, Col).
write_list_of_signs([X|'$END_SEQ'], Tab, Col) :- !,
	print(X, Tab, Col).
write_list_of_signs([X|'$INDEX'(N)], Tab, Col) :- !,
	print(X, Tab, Col),
	write('|@'), write(N).
write_list_of_signs([H|T], Tab, Col) :-
	print(H, Tab, Col), write(','), nl,
	write_list_of_signs(T, Col, Col).


write_list_of_NODEs([]).
write_list_of_NODEs('$END_SEQ').
write_list_of_NODEs(['$INDEX'(M)|'$END_SEQ']) :-
  write('@'), write(M).
write_list_of_NODEs(['$INDEX'(M)|'$INDEX'(N)]) :- !,
  write('@'), write(M), write('|@'), write(N).
write_list_of_NODEs(['$INDEX'(N)]) :- !,
  write('@'), write(N).
write_list_of_NODEs(['$INDEX'(N)|T]) :-
  write('@'), write(N), write(', '),
  write_list_of_NODEs(T).



% Utilities

tail([_H|T], T) :- (var(T); T = []), !.
tail([_|T], Tail) :- tail(T, Tail).

prlength(Atom, L) :- atomic(Atom), name(Atom, N), length(N, L), !.

reverse(L1, L2) :-
	reverse(L1, [], L2).

reverse([], L, L).
reverse([H|T], L, R) :-
	reverse(T, [H|L], R).
 


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

mike_print_lexical_entry(lexical_entry, Orth, Dag) :-
    pdag(Dag, STDForm),
    eccs_output(captioned(Orth, STDForm)).
    
mike_print_lexical_entry(lexical_entry, Orth, Dag, captioned(Caption, STDForm)) :- 
    pdag(Dag, STDForm),
    eccs_concat_list(['The lexical entry: "', Orth, '"'], Caption).
    


mike_print_template(template, Name, temp(Dag, _), captioned(Caption, STDForm)) :-
    pdag(Dag, STDForm),
    eccs_concat_list(['The template: "', Name, '"'], Caption).


mike_print_grammar_rule(Type, Name, Rule) :-
    mike_print_grammar_rule(Type, Name, Rule, STDForm),
    eccs_output(captioned(Name, STDForm)).

mike_print_grammar_rule(lp_statement, Name, prec(F0, X0, Y0), 
	captioned(Caption, sequence([FSTD, XSTD, YSTD]))) :-
  findunifs(F0, VarTree, FRoot),
  findunifs(X0, VarTree, XRoot),
  findunifs(Y0, VarTree, YRoot),
  unfold_value_tree(VarTree),
  rename_indexes([FRoot, XRoot, YRoot]),
  mike2standard_format(FRoot, FSTD),
  mike2standard_format(XRoot, XSTD),
  mike2standard_format(YRoot, YSTD),
  eccs_concat_list(['The lp statement: "', Name, '"'], Caption).
  

mike_print_grammar_rule(rule, Name, rule(Mom, Cons), captioned(Caption, STDForm)) :-
    eccs_sys_call(Cons),
    pdag(Mom, STDForm),
    eccs_concat_list(['The grammar rule: "', Name, '"'], Caption).

mike_draw_fs(_Args, Caption, Dag, captioned(Caption, STDForm)) :-
    pdag(Dag, STDForm).

eccs_default_printing_routine(_, Dag) :-
    pdag(Dag), nl.


/*

mike2standard_format(Root, STDForm)

The dag Root has the standard  form STDForm

*/

mike2standard_format([], []).
mike2standard_format([F = V|R], avm(L)) :- !,
    mike2standard_format_avm([F = V|R], L).
mike2standard_format([F|R], sequence(L)) :- 
    mike2standard_format_seq([F|R], L).
mike2standard_format('$NEG'(N), neg(N1)) :- !,
    mike2standard_format(N, N1).
mike2standard_format('$INDEX'(N), tag(N)) :- !.
mike2standard_format('$INDEX'(N, L), tagged_avm(N, L1)) :- !,
    mike2standard_format_avm(L, L1).
mike2standard_format('$NULL_SEQ', sequence([])) :- !.
mike2standard_format('$UNINSTANTIATED', uninstantiated) :- !.
mike2standard_format(A, atomic(A)) :-
    eccs_sys_atomic(A).

mike2standard_format_avm([], []).
mike2standard_format_avm([F = V|R], [F = V1|L]) :-
    mike2standard_format(V, V1),
    mike2standard_format_avm(R, L).

mike2standard_format_seq([], []).
mike2standard_format_seq('$INDEX'(N), [tag(N)]).
mike2standard_format_seq('$END_SEQ', []).
mike2standard_format_seq([F|R], [H|T]) :-
    mike2standard_format(F, H),
    mike2standard_format_seq(R, T).
