/*

Monday, September 3, 1990 mkr

Reader for the usual type of feature structure notation.  However, this
also allows lists of atoms, indexes or other feature structures. 
Indexes follow Jo Calder's notations.  There are no restrictions on the
use of indexes.  (If there is, it's a bug.)  This is meant to be used in
conjunction with the pretty printer of Sat, 1 Sep 1990.  They should
read and write each other's input and output.

This reader is considerably simpler than the corresponding printer.  The
problem of making things reentrant is trivial compared to the problem of
finding reentrancies and simplifying the input accordingly.
*/  

eccs_term_reader(_, Name, Type, Definition) :-
    eccs_sys_read(X),
    (X == end_of_file, X = Name -> X = Definition; 
	    well_formed_grammar_object(X, Name, Type, Definition)).

/*

well_formed_grammar_object(X, Name, Type, Definition)

*/

well_formed_grammar_object(word(X, Name), Name, lexical_entry, X).
well_formed_grammar_object(prec(N, X, Y, Z), N, lp_statement, prec(X, Y, Z)).
well_formed_grammar_object(rule(Name, X, Cons), Name, rule, rule(X, Cons)).
well_formed_grammar_object(temp(X, Name), Name, template, temp(X, Name)).
well_formed_grammar_object(abb(Abb, Path), Abb, path_abbreviation, abb(Abb, Path)).


mike_rule_compiler(rules, rule, Name, _, rule(X, Cons), rule(Y, Cons), []) :-
    !,
    read_dag(X,Y).
mike_rule_compiler(rules, lp_statement, Name, _, prec(Fun0, X0, Y0), prec(Fun, X, Y), []) :- 
    !,
    read_dag(Fun0,Fun,[],L0),
    read_dag(X0,X,L0,L1),
    read_dag(Y0,Y,L1,_).
mike_template_compiler(templates, template, Name, _, temp(X, Name), temp(Y, Name), []) :-
    !,
    read_dag(X,Y).

mike_lex_compiler(words, lexical_entry, Name, _, X, Y, []) :-
  read_dag(X,Y).

read_dag(D0, D) :-
  read_dag(D0, D, [], _).

/*
read_dag(Input, Output, AList, NewAList)

Input is the term which is being transformed by the reader.  Basically,
closed ended lists encoding feature structures get turned into open
ended lists and coindexed substructures get unified.  Output is the
result of the transformation.  AList is an association list of
"equations" of the form N = D where N is always an integer and D is a
feature structure.  NewAList is the updated alist.  read_dag simply
processed each feature-value pair of a feature structure by calling
read_dag_value to transform the value.
*/

read_dag(@N, D, List, List) :-
  assoc((N = D), List), !.
read_dag(@N, D, List, [(N=D)|List]) :- !.
read_dag(N@D0, D, List0, List) :-
  assoc((N = D), List0), !,
  read_dag(D0, D1, List0, List),
  D <=> D1.
read_dag(N@D0, D, List0, [(N=D)|List]) :- !,
  read_dag(D0, D, List0, List).
read_dag([], _, List, List).
read_dag([F = ~Atom|T0], [F = Var|T], List0, List) :-
  atomic(Atom), !,
  dif(Var, Atom),
  read_dag(T0, T, List0, List).
read_dag([F = V0 | T0], [F = V | T], List0, List) :-
  read_dag_value(V0, V, List0, List1),
  read_dag(T0, T, List1, List).

    

/*
read_dag_value(Input, Output, AList, NewAList)

The arguments are as for read_dag except that Input is any type of
feature value including atoms, (Prolog) variables, indexes, indexed
feature structures, feature structures and arbitrary lists.  Obviously,
there are lots of subcases.

The treatment of indexes is simple.  In all cases, the association list
is checked for the index.  If it is there, then the value associated
with it is unified with the dag indexed.  If it's not there, then the
index and the dag indexed are added to the alist.  (If the index occurs
alone, then a variable is added to the alist.)
*/

read_dag_value(V, V, L, L) :-
  atomic(V), !.
read_dag_value(V, V, L, L) :-
  var(V), !.
read_dag_value(@N, D, List, List) :-
  assoc((N = D), List), !.
read_dag_value(@N, D, List, [(N=D)|List]) :- !.
read_dag_value(N@D0, D, List0, List) :-
  assoc((N = D), List0), !,
  read_dag_value(D0, D1, List0, List),
  D <=> D1.
read_dag_value(N@D0, D, List0, [(N=D)|List]) :- !,
  read_dag_value(D0, D, List0, List).
read_dag_value([F = V0|T], V, L0, L) :- !,
  read_dag([F = V0|T], V, L0, L).
read_dag_value([H|T], V, L0, L) :- !,
  read_list([H|T], V, L0, L).

/*
read_list/4

The arguments are as for read_dag except that Input is a list of values. 
read_list just calls read_dag_value on each element of the list since
anything can appear in a list. 
*/

read_list([],[],L,L).
read_list([H0|@N0],[H|N],L0,L) :- !,
  read_dag_value(H0,H,L0,L1),
  read_dag_value(@N0,N,L1,L).
read_list([H0|T0],[H|T],L0,L) :-
  read_dag_value(H0,H,L0,L1),
  read_list(T0,T,L1,L).

/*
We can cheat here with the association list access since the key N
is an integer which is always ground.  So, we can do a real unification
instead of ==.
*/

assoc((N = D), [(N = D)|_]) :- !.
assoc((N = D), [_|T]) :-
  assoc((N = D), T).
