/*

Heavily Modified Thu Sep 29 15:36:25 1988, JC and MKR.

*/

/*

Checking classes of unification problems.

We will assume that given a list of equations to string unify that if there
is an equation which is in one of the acceptable classes, then choosing
it first instead of some other will not ultimately change the
possibility of finding an answer.  So, for example, if we have a list
of equations [A,B,C] and A is acceptable and then as a result of A, B
is acceptable but then C isn't, then we give up at this point rather
than try the three elements in some other order.  This assumption is
to make the behaviour linear and to avoid factorial behaviour when the
set of equations is not in an acceptable class (cf. below).

One thing that can be done is that all the acceptable equalities are
found first and then done all in one go.  Then we know that we can
never back up beyond this point.  After these are all done, then we
check the remaining list and do all of those that are acceptable and
so on recursively until the list is empty or we come to a point where
we have a nonempty list remaining no element of which is acceptable.  At
that point, there is nothing that we can do short of writing a message
or just plunging ahead and hoping we terminate.  This strategy isn't
factorial but does incur the cost of researching the sublists and
checking the equation classes.

The first acceptable class has one of the strings completely ground.
Then it doesn't matter what the second string is.  (This is the case
for both subsumption checking and the processing of lexical entries so
we can just unify any way we like for these).  This is a subclass of
P0.  P0 itself says that no variable occurs more than once in either
of the strings.  P0.5 says that no variable occurs in both strings.
Therefore, P0 is a subset of P0.5 for if a variable occurs only once
then it cannot appear in both strings.  Another special class that we
can make use of (although we have no proof) says that that no
nondisjunctive variable (including negative variables Thu Sep 29
15:37:59 1988) occurs in both strings but that disjunctive variables
can occur any number of times in either string.  Basically, we can
treat disjunctive variables as if they are not variables at all and
just check for P0.5.

Therefore, we will define a predicate acceptable_class/2 which is true
if the (compiled) strings are in the modified notion of P0.5 just described.

*/

% ac_unify_eq_list(Eq) is true if the list of equations Eq can be
% unified in some order such that each equation is in P0.5.

ac_unify_eq_list([]) :- !.
ac_unify_eq_list(Eq) :-
	acceptable_class_sort(Eq,Safe,UnSafe),
	Safe \== [],
	sunify_eq_list(Safe),
	ac_unify_eq_list(UnSafe).

% acceptable_class_sort(X,Y,Z) is true if X is a list of equations and
% Y is the list of the members of X that are in P0.5 and Z is the list
% of the members of X that are not.

acceptable_class_sort([],[],[]).
acceptable_class_sort([L=R|T],[L=R|Safe],UnSafe) :-
	acceptable_class(L,R), !,
	acceptable_class_sort(T,Safe,UnSafe).
acceptable_class_sort([L=R|T],Safe,[L=R|UnSafe]) :-
	acceptable_class_sort(T,Safe,UnSafe).

% acceptable_class(X,Y) is true if the pair <X,Y> of compiled strings
% is in the modified version of P0.5 discussed above.

acceptable_class(S,_) :-
	xvar(S), !.
acceptable_class(_,T) :-
	xvar(T), !.
acceptable_class([],[]) :- !.
acceptable_class(S,T) :-
	acceptable_class0(S,T).

acceptable_class0(S,T) :-
	get_nondvar_vars(S,SVars),
	get_nondvar_vars(T,TVars),
	eccs_verify((
	  numbervars(SVars,1,_),
	  numbervars(TVars,1,_),
	  u_intersect(SVars,TVars,Intersection),
	  \+ Intersection = [] )).

get_nondvar_vars([],[]).
get_nondvar_vars([S1|S],[S1|Vars]) :-
	\+ dvar(S1),		% not a disjunctive variable
	xvar(S1), !,		% but is a variable
	get_nondvar_vars(S,Vars).
get_nondvar_vars([S1|S],Vars) :-
	get_nondvar_vars(S,Vars).

