/*
File:	/mnt/jo/Protolexicon/qmorphtables.pl
Date:	Wed Apr 27 14:35:02 1988
By:	Mike Reape with changes by Jo Calder
*/

% compile a set of morphology tables into internal format and
% calculate the subsumption relation over them.

eccs_q_compile_tables :-
    eccs_file_of_type(File, morph_tables),
    eccs_debug(1, eccs_message([starting, to, compile, morphological, tables, '$nl$',
    	      here, are, the, subsumption, relations, between, tables])),
    eccs_get_stats(Heap, CPU),
    eccs_db_retractall(table_ids, _, _),
    TopTable = table('$top$',[],[],Spe,[],m(Cl,St,Rt,[],[]),[],[]),
    eccs_db_retractall(table, '$top$', TopTable),
    eccs_store_in_database(File, table, '$top$', _, TopTable, []),
    findall(Id,
	eccs_get_from_database(table, Id, _),
	IdList),
    eccs_list_to_set(IdList,IdSet),
    eccs_store_in_database(File, table_ids, table_ids, _, IdSet, []),
    eccs_comp_tbls,
    eccs_comp_tbl_sub,
    eccs_comp_tops,
    eccs_comp_subs,
    eccs_comp_pres,
    eccs_comp_sucs,
    eccs_comp_equiv,
    eccs_get_stats(Heap1, CPU1),
    Heap2 is Heap1 -Heap, CPU2 is CPU1 -CPU,
    eccs_debug(1, eccs_message([table, compilation, complete, Heap2, bytes, CPU2, cpu])),
    findall(Eq, 
	eccs_get_from_database(equiv, equiv, Eq), Eqs),
    (Eqs = []; eccs_debug(1, eccs_message([the, following, tables, are, equivalent,
			     in, entry, conditions, '$nl$', Eqs]))), !.

% compile the tables into internal format.

% erase all the previously compiled tables and then compile the new ones.

eccs_comp_tbls :-
	eccs_db_retractall(ctable, _, _),
	eccs_comp_tbls0.

% for each table, generate every most general compilation instance.
% this is nondeterminate due to the possibility that string
% unification is in general has an infinitary solution space.

eccs_comp_tbls0 :-
	eccs_get_from_database(table, _, Table),
	eccs_comp_tbl(Table),
	fail.
eccs_comp_tbls0.

% verify that the table can be compiled and print an error message if
% it can't.

eccs_comp_tbl(Table) :-
	eccs_verify(eccs_comp_tbl(Table,_)), !,
	eccs_comp_tbl0(Table).
eccs_comp_tbl(table(Id,_,_,_,_,_,_,_)) :-
	  eccs_debug(1, eccs_error(['**Table',Id,'didnt compile.'])).

% compile a table and add it to the secondary database.

eccs_comp_tbl0(Table) :-
	eccs_comp_tbl(Table,CTable0),	% compile the table
	eccs_strip(CTable0,CTable),		% cleanup the constraint list stack
	eccs_file_of_type(File, morph_tables),
	cleanup_table(CTable, File).		% add the table to the database.
eccs_comp_tbl0(_).

% compile a table.  notice that constraint lists are unified with
% their source level variables only after all notation compilation has
% been performed.  this is to keep the compilation predicates simple.
% ac_unify_eq_list is called to do the actual string unifications to
% guarantee that there is a progression of P0.5 string unifications
% corresponding to the set of string equalities.

eccs_comp_tbl(table(Id,Tem,D,Spe,Eq0,M0,LR,F0),ctable(Id,Tem,D,Spe,[],M,LR,F)) :-
	M0 = m(Cl,St0,Rt0,MS0,MR0),
	ext(St0,St),
	ext(Rt0,Rt),
	ext_alist(MS0,MS),
	ext_alist(MR0,MR),
	M = m(Cl,St,Rt,MS,MR),
	ext_list(F0,F),
	ext_eq_list(Eq0,Eq,C),
	unify_constraints_list(C),
	ac_unify_eq_list(Eq).

% if there is no prior compilation instance of a table, then add the
% current compilation instance to the secondary database.  if the
% current compilation instance subsumes any prior instances, then
% remove them and add this instance to the database.  if this instance
% is subsumed by any prior instances then don't bother to add it to
% the database.

cleanup_table(CTable1, File) :-
    eccs_get_from_databaser(ctable, _, CTable2,Ref),
    cleanup_table0(CTable1,CTable2), !,
    eccs_sys_erase(Ref),
    CTable1 = ctable(Id,_,_,_,_,_,_,_),
    eccs_store_in_database(File, ctable, Id, _, CTable1, []).
cleanup_table(CTable1, _File) :-
    eccs_get_from_database(ctable, _, CTable2),
    cleanup_table0(CTable2,CTable1), !.
cleanup_table(CTable, File) :-
    CTable = ctable(Id,_,_,_,_,_,_,_),
    eccs_store_in_database(File, ctable, Id, _, CTable, []).

cleanup_table0(CTable1,CTable2) :-
    CTable1 = ctable(Id,_,_,_,_,M1,_,_),
    CTable2 = ctable(Id,_,_,_,_,M2,_,_),
    eccs_once(m_subsumes(M1,M2)).

% m_subsumes is true if the canonical morphology parameterised
% template M1 subsumes M2.

m_subsumes(M1,M2) :-
	eccs_verify((
	  eccs_q_numbervars(M2,0,_),
	  M1 = m(Cl,S1,R1,MS1,MR1),
	  M2 = m(Cl,S2,R2,MS2,MR2),
	  eccs_q_subsumes(S1,S2),
	  eccs_q_subsumes(R1,R2),
	  eccs_q_subsumes_assoc_list(MS1,MS2),
	  eccs_q_subsumes_assoc_list(MR1,MR2) )).

% eccs_q_subsumes_assoc_list is the subsumption equivalent of
% sunify_assoc_list.  Cf.  eccs_q_subsumes.

eccs_q_subsumes_assoc_list([],_).
eccs_q_subsumes_assoc_list([K-S|M],M0) :-
	assoc(K-S0,M0),
	eccs_q_subsumes(S,S0),
	eccs_q_subsumes_assoc_list(M,M0).

% ext(X,Y) compiles a string pattern X into compiled form Y.

ext(X,X) :-
	eccs_sys_var(X), !.
ext(X+Y,Z) :- !,
	eccs_explode(X+Y,Z).
ext(+Y,Z) :- !,
	eccs_explode(X+Y,Z).
ext((X++),Z) :- !,
	eccs_explode(X+Y,Z).
ext(X0,X) :-
	eccs_sys_atom(X0), !,
	eccs_explode(X0,X).
ext([H|T],Z) :-
	eccs_explode([H|T],Z).

% ext_list compiles a list of string patterns.

ext_list([],[]).
ext_list([H0|T0],[H|T]) :-
	ext(H0,H),
	ext_list(T0,T).

% ext_alist compiles the values in an association list.

ext_alist([],[]).
ext_alist([K-M0|T0],[K-M|T]) :-
	ext(M0,M),
	ext_alist(T0,T).

% ext_eq_list translates the list of equational constraints.

ext_eq_list([],[],[]).
ext_eq_list([H0|T0],E0,C0) :-
	ext_eq(H0,E0,E,C0,C),
	ext_eq_list(T0,E,C).

% translate an equational constraint.

ext_eq((X\=Y),E,E,[C0|C],C) :- !,
	ext_ne((X\=Y),C0).
ext_eq((X=Y),En,E,Cn,C) :-
	eccs_verify(ext_eq1((X=Y),E0,C0)), !,
	ext_eq1((X=Y),E0,C0),
	( E0 = [] -> En = E ; En = [E0|E] ),
	( C0 = [] -> Cn = C ; Cn = [C0|C] ).
ext_eq((X=Y),[E0|E],E,C,C) :-
	ext_eq2((X=Y),E0).

% translate a negative constraint.

ext_ne(R0 \= L0, L0=V:C) :-
	(eccs_sys_var(L0) ; eccs_sys_atom(L0) ),
	eccs_sys_nonvar(R0),
	( R0 = {R1},
	  eccs_comma_to_list(R1,R) ;
	  R0 = `Class,
	  eccs_q_table_class(Class,R) ),
	ext_vneg(V,R,C).
ext_ne(R0 \= L0, L0=V:C) :-
	(eccs_sys_var(L0) ; eccs_sys_atom(L0) ),
	eccs_sys_nonvar(R0),
	R0 = 0,
	ext_vneg(V,[0],C).
ext_ne(A \= L0, L0=V:C) :-
	(eccs_sys_var(L0) ; eccs_sys_atom(L0) ),
	eccs_sys_atom(A),
	eccs_sys_name(A,[_]),
	ext_vneg(V,[A],C).
ext_ne(L0 \= R0, L0=V:C) :-
	(eccs_sys_var(L0) ; eccs_sys_atom(L0) ),
	eccs_sys_nonvar(R0),
	( R0 = {R1},
	  eccs_comma_to_list(R1,R) ;
	  R0 = `Class,
	  eccs_q_table_class(Class,R) ),
	ext_vneg(V,R,C).
ext_ne(L0 \= R0, L0=V:C) :-
	(eccs_sys_var(L0) ; eccs_sys_atom(L0) ),
	eccs_sys_nonvar(R0),
	R0 = 0,
	ext_vneg(V,[0],C).
ext_ne(L0 \= A, L0=V:C) :-
	(eccs_sys_var(L0) ; eccs_sys_atom(L0) ),
	eccs_sys_atom(A),
	eccs_sys_name(A,[_]),
	ext_vneg(V,[A],C).

% translate a disjunctive constraint.

ext_eq1(R0 = L0, E0, C0) :-
	(eccs_sys_var(L0) ; eccs_sys_atom(L0) ),
	eccs_sys_nonvar(R0),
	( R0 = {R1},
	  eccs_comma_to_list(R1,R) ;
	  R0 = `Class,
	  eccs_q_table_class(Class,R) ),
	ext_vdisj(L0,R,E0,C0).
ext_eq1(L0 = R0, E0, C0) :-
	(eccs_sys_var(L0) ; eccs_sys_atom(L0) ),
	eccs_sys_nonvar(R0),
	( R0 = {R1},
	  eccs_comma_to_list(R1,R) ;
	  R0 = `Class,
	  eccs_q_table_class(Class, R) ),
	ext_vdisj(L0,R,E0,C0).

% translate a genuine equational constraint.

ext_eq2(L0=R0,L=R) :-
	ext(L0,L),
	ext(R0,R).

% make a negative constraint.

ext_vneg(L,R,c(L, not(_, R))).

% make a disjunctive constraint.

ext_vdisj(L,R,E,C) :-
	ext_vdisj_separate_disjs(R,Singles,Multiples),
	( ext_vdisj_enumerate_multiples(Multiples,Rhs),
	  C = [],
	  E = (L = Rhs) ;
	  ext_vdisj_single(Singles,Rhs),
	  C = (L = Rhs),
	  E = [] ).

ext_vdisj_separate_disjs([],[],[]) :- !.
ext_vdisj_separate_disjs([H|T],[H|S],M) :-
	eccs_sys_name(H,[_]), !,
	ext_vdisj_separate_disjs(T,S,M).
ext_vdisj_separate_disjs([H|T],S,[H|M]) :-
	ext_vdisj_separate_disjs(T,S,M).

ext_vdisj_enumerate_multiples(Multiples,List) :-
	eccs_member(Word,Multiples),
	ext(Word,List).
	
ext_vdisj_single(R,L:c(L, or(_, R))).

% unify the constraints with the source level variables.

unify_constraints_list([]).
unify_constraints_list([V=V1:C1|T]) :-
  eccs_sys_var(V), !,
  V = V0:C0,
  V0 = V1,
  eccs_q_merge_constraints(C0,[[C1]|_]),
  unify_constraints_list(T).
unify_constraints_list([V0:C0=V1:C1|T]) :-
  V0 = V1,
  eccs_q_merge_constraints(C0,[[C1]|_]),
  unify_constraints_list(T).

% explode an atom or a simple string pattern into a string pattern
% (i.e., list of characters).

eccs_explode(X,Y) :-
	eccs_explode0(X,Y,[]).

eccs_explode0(Var,[Var|T],T) :-
	eccs_sys_var(Var), !.
eccs_explode0([],T,T) :- !.
eccs_explode0(Atom,List,L) :-
	eccs_sys_atom(Atom), !,
	eccs_atom_to_chars(Atom,List,L).
eccs_explode0(X+Y,L0,L) :- !,
	eccs_plus_to_list(X+Y,List),
	eccs_explode0(List,L0,L).
eccs_explode0([H|T],L0,L) :- !,
	eccs_explode0(H,L0,L1),
	eccs_explode0(T,L1,L).

% compute the the subsumption relation over the tables.

eccs_comp_tbl_sub :-
    eccs_db_retractall(subsumes, _, _),
    eccs_get_from_database(table_ids, table_ids, IdSet),
    eccs_comp_tbl_sub0(IdSet).

eccs_comp_tbl_sub0([]).
eccs_comp_tbl_sub0([H|T]) :-
	eccs_comp_tbl_sub1(H,T),
	eccs_comp_tbl_sub0(T).

eccs_comp_tbl_sub1(Id,[]).
eccs_comp_tbl_sub1(Id,[H|T]) :-
	eccs_comp_tbl_sub2(Id,H),
	eccs_comp_tbl_sub1(Id,T).

% check whether table Id1 subsumes Id2 and whether Id2 subsumes Id1.

eccs_comp_tbl_sub2(Id1,Id2) :-
	eccs_comp_tbl_sub3(Id1,Id2),
	eccs_comp_tbl_sub3(Id2,Id1).

% table Id1 subsumes table Id2 if every compilation of Id2 is subsumed
% by some compilation of Id1.  this is accomplished by not specifying
% the reference of Id1 in eccs_comp_tbl_sub5.  don't bother if Id2 didn't
% compile.  also, if Id1 does subsume Id2 then add all other
% subsumption information that you can infer from the transitive
% closure of the subsumes relationship immediately.

eccs_comp_tbl_sub3(Id1,Id2) :-
    eccs_get_from_database(subsumes, Id1, Id2), !.
eccs_comp_tbl_sub3(Id1,Id2) :-
    findall(Ref,
    	eccs_get_from_databaser(ctable, _, ctable(Id2,Tem2,D2,Spe2,C2,M2,_,_),Ref),
	Refs),
    Refs \== [],
    eccs_comp_tbl_sub4(Id1,Id2,Refs), !,
    tc_subsumes(Id1,Id2),
    eccs_file_of_type(File, morph_tables),
    eccs_store_in_database(File, subsumes, Id1, _, Id2, []),
    eccs_debug(1, eccs_message([Id1, '>', Id2])).
eccs_comp_tbl_sub3(_,_).

% table Id1 subsumes table Id2 with list of references if table Id1
% subsumes each of the references.

eccs_comp_tbl_sub4(_,_,[]).
eccs_comp_tbl_sub4(Id1,Id2,[Ref|Refs]) :-
	eccs_comp_tbl_sub5(Id1,Id2,Ref),
	eccs_comp_tbl_sub4(Id1,Id2,Refs).

% check whether Id1 subsumes compilation Ref of Id2.  The first verify
% checks for gross subsumption of the term structure, templates, disjunctive
% templates and the kb parameterised templates.  m_subsumes guarantees
% that the morphological parameterised templates are in the
% subsumption relation.

eccs_comp_tbl_sub5(Id1,Id2,Ref) :-
    eccs_get_from_database(ctable, Id1, ctable(Id1,Tem1,D1,Spe1,C1,M1,_,_)),
    eccs_get_from_databaser(ctable, Id2, ctable(Id2,Tem2,D2,Spe2,C2,M2,_,_), Ref),
    eccs_verify((
         numbervars(Tem2+D2+Spe2,0,_),
	 eccs_subset(Tem1,Tem2),
	 disj_subsumes(D1,D2,Tem2),
	 Spe1 = Spe2 )),
    m_subsumes(M1,M2).

% disj_subsumes(D1,D2,Tems2) if the set of disjunctive templates in D1
% subsumes the set of disjunctive templates D2 and the set of
% templates Tems2.

disj_subsumes([],_,_).
disj_subsumes([H|T],D,Tem) :-
	once((
	  disj_subsumes_disj(H,D) ;
	  disj_subsumes_tem(H,Tem) )),
	disj_subsumes(T,D,Tem).

disj_subsumes_disj(X,[D|_]) :-
	eccs_subset(D,X).
disj_subsumes_disj(X,[_|D]) :-
	disj_subsumes_disj(X,D).
disj_subsumes_disj(X,D0) :-
	eccs_member(D1,D0),
	eccs_member(D3,D0),
	eccs_not_eq(D1, D3),
	subtract(D0,[D1],D2),
	subtract(D2,[D3],D4),
	eccs_intersection(D1,D3,D5),
	disj_subsumes_disj(X,[D5|D4]).

disj_subsumes_tem(D,[Tem|_]) :-
	eccs_member(Tem,D).
disj_subsumes_tem(D,[_|Tems]) :-
	disj_subsumes_tem(D,Tems).

% compute the greatest element(s) in the semi-lattice.  This should
% always be '$top$'.  If it isn't then the user, has done something
% wrong.  This should probably be checked and an error printed if
% there is not a unique top element.

eccs_comp_tops :-
    eccs_db_retractall(top_table, _, _),
    eccs_get_from_database(table_ids, table_ids, IdSet),
    eccs_file_of_type(File, morph_tables),
    eccs_comp_tops0(IdSet, File).

eccs_comp_tops0([], File).
eccs_comp_tops0([H|T], File) :-
    findall(Id, subsumes(Id,H),
		IdSet),
	eccs_sys_if_then_else( IdSet = [H], eccs_store_in_database(File, top_table, top_table, _, H, []), true),
	eccs_comp_tops0(T, File).

% subsumes(X,Y) is true if X and Y are the same or we have explicitly
% computed that X subsumes Y above.

subsumes(X,X).
subsumes(X,Y) :-
    eccs_get_from_database(subsumes, X, Y).

% tc_subsumes computes the transitive closure of the subsumes relation
% based on the new information that Id1 subsumes Id2.

tc_subsumes(Id1,Id2) :-
	tc_pres(Id1,Id2),
	tc_sucs(Id1,Id2).

% If Id1 subsumes Id2 then anything that subsumes Id1 also subsumes Id2.

tc_pres(Id1,Id2) :-
    eccs_get_from_database(subsumes,Id0,Id1),
    eccs_not_eq(Id0, Id2),
    \+ eccs_get_from_database(subsumes, Id0,Id2),
    eccs_file_of_type(File, morph_tables),
    eccs_store_in_database(File, subsumes, Id0, _, Id2, []),
    eccs_debug(1, eccs_message([Id0, '>', Id2])),
    fail.
tc_pres(_,_).

% If Id1 subsumes Id2 then anything that Id2 subsumes, Id1 also subsumes.

tc_sucs(Id1,Id2) :-
    eccs_get_from_database(subsumes, Id2, Id3),
    eccs_not_eq(Id1, Id3),
	\+ eccs_get_from_database(subsumes, Id1,Id3),
    eccs_file_of_type(File, morph_tables),
    eccs_store_in_database(File, subsumes, Id1, _, Id3, []),
    eccs_debug(1, eccs_message([Id1, '>', Id3])),
    fail.
tc_sucs(_,_).

% compute the immediately subsumes relation.

eccs_comp_subs :-
    eccs_db_retractall(subs, _, _),
    eccs_get_from_database(table_ids, table_ids, IdSet),
    eccs_file_of_type(File, morph_tables),
    eccs_comp_subs0(IdSet, File).

eccs_comp_subs0([], _).
eccs_comp_subs0([H|T], File) :-
    findall(Id,subsumes(H,Id),IdSet),
    eccs_store_in_database(File, subs, H, _, IdSet, []),
    eccs_comp_subs0(T, File).

% compute the immediate successor relation.

suc(X, Y) :-
    eccs_get_from_database(suc, X, Y).

eccs_comp_sucs :-
    eccs_db_retractall(suc, _, _),
    eccs_file_of_type(File, morph_tables),
    eccs_get_from_database(table_ids, table_ids, IdSet),
    eccs_comp_sucs0(IdSet, File).

eccs_comp_sucs0([], _).
eccs_comp_sucs0([H|T], File) :-
    findall(S,successor(H,S),SucSet),
    eccs_sys_if_then_else( SucSet = [], true,
    	eccs_store_in_database(File, suc, H, _, SucSet, [])),
    eccs_comp_sucs0(T, File).

% successor(X,Y) is true if X is the immediate successor of Y.

successor(X,Y) :-
    eccs_get_from_database(subsumes, X, Y),
	\+ ( eccs_get_from_database(subsumes, X,I),
	     \+ equivalent(X,I),
	     eccs_get_from_database(subsumes, I,Y),
	     \+ equivalent(I,Y) ),
	\+ equivalent(X,Y).

% compute the immediate predecessor relation.

eccs_comp_pres :-
    eccs_db_retractall(pre, _, _),
    eccs_file_of_type(File, morph_tables),
    eccs_get_from_database(table_ids, table_ids, IdSet),
    eccs_comp_pres0(IdSet, File).

eccs_comp_pres0([], _).
eccs_comp_pres0([H|T], File) :-
    findall(P,successor(P,H),PreSet),
    eccs_sys_if_then_else( PreSet = [], true,
	eccs_store_in_database(File, pre, H, _, PreSet, [])),
    eccs_comp_pres0(T, File).

% compute the equivalence relation over tables.  again, we don't
% currently do anything here if two tables are equivalent.  in fact,
% this information is not used.  it would be helpful to issue a
% warning to the user.  such situations arise when the entry
% conditions are equivalent but the morphology/lexical rule
% specifications are different.

eccs_comp_equiv :-
    eccs_db_retractall(equiv, _, _),
    eccs_get_from_database(table_ids, table_ids, IdSet),
    eccs_file_of_type(File, morph_tables),
    eccs_comp_equiv0(IdSet, File).

:- dynamic equiv/1.

eccs_comp_equiv0([], _).
eccs_comp_equiv0([H|T], File) :-
    \+ ( eccs_get_from_database(equiv, _, E),
	eccs_member(H,E) ), !,
    findall(P,equivalent(P,H),EquivSet),
    eccs_sys_if_then_else( EquivSet = [], true, eccs_store_in_database(File, equiv, equiv, [H|EquivSet])),
    eccs_comp_equiv0(T, File).
eccs_comp_equiv0([_|T], File) :-
	eccs_comp_equiv0(T, File).

% two tables are equivalent if they subsume each other and they are
% not identical.

equivalent(X,Y) :-
    eccs_get_from_database(subsumes, X,Y),
    eccs_get_from_database(subsumes, Y,X),
    eccs_not_eq(X,  Y).
