/*
File:	/mnt/jo/Protolexicon/qtablesubsumption.pl
Date:	Wed Apr 27 15:38:08 1988
By:	Mike Reape

Subsidiary functions for table subsumption etc.
*/

% eccs_u_sub_con(X,Y,X0,Y0) is true if the disjunctive or negative constraints in
% X subsume those in Y.  X0 is the reduced version of X and Y0 is the
% reduced version of Y.

% first simplify the two constraint lists (calculate disjunction
% intersections, etc.).  then check that X's negative constraints
% subsume Y's.  then check that X's disjunctive constraints subsume Y's.

eccs_u_sub_con(X,Y) :-
	simplify_constraints(X,XN,XD),
	simplify_constraints(Y,YN,YD),
	eccs_u_sub_neg(XN,YN),
	eccs_u_sub_disj(XD,YD).

% separate the constraint list into two separate negative and
% disjunctive constraint lists.

separate_neg_and_disj(X0,XN,XD) :-
	eccs_flatten(X0,X),
	separate_neg_and_disj0(X,XN,XD).

separate_neg_and_disj0([],[],[]).
separate_neg_and_disj0([Neg|C],[Neg|N],D) :-
	Neg = c(Node, not(_, L)),
	separate_neg_and_disj0(C,N,D).
separate_neg_and_disj0([Disj|C],N,[Disj|D]) :-
	Disj = c(Node, or(_, L)),
	separate_neg_and_disj0(C,N,D).

% eccs_u_sub_neg(X,Y) is true if the negative constraints in X subsume
% those in Y.

% the basic idea here is that for X to subsume Y, any negative
% constraint in X must also occur in Y.  however, Y can contain negative
% constraints that X does not.

eccs_u_sub_neg([], []) :- !.		% trivially okay
eccs_u_sub_neg([], YN) :- !.		% XN is a subset of YN
eccs_u_sub_neg(XN, []) :- !, fail.	% XN has constraints that YN doesn't, fail.
eccs_u_sub_neg(XN, YN) :-		% see that each XNi is in some YNi
	eccs_u_sub_neg0(XN, YN).

eccs_u_sub_neg0([], YN).
eccs_u_sub_neg0([H1|T1], YN) :- !,
	eccs_u_sub_neg1(H1, YN),
	eccs_u_sub_neg0(T1, YN).

eccs_u_sub_neg1(c(Node, not(_, L)), R) :- !,	% unpack 
	eccs_u_sub_neg2(Node,L,R).

% check that each element of the first list occurs somewhere in the
% list of constraints in the second list.

eccs_u_sub_neg2(_,[],_) :- !.
eccs_u_sub_neg2(Node,[H|T],L) :-
	eccs_u_sub_neg3(Node,H,L),
	eccs_u_sub_neg2(Node,T,L).

% check that the first argument appears in one of the negative
% constraints in the list in the second argument.  if there is no
% corresponding negative constraint in the second argument then there
% is no subsumption relation.  i.e., the subsumed argument does not
% contain that constraint.

eccs_u_sub_neg3(Node,Y,X) :-
	eccs_member(c(Node2, not(_, L)), X),
	Node2 == Node,
	eccs_member(Y,L).

% eccs_u_sub_disj(X,Y) is true if the disjunctive constraints in X
% subsume those in Y.

% the basic idea here is that for X to subsume Y, any disjunctive
% constraint in Y must also occur in X.  i.e., X must have more disjunctions 
% than Y but if X has any then Y must as well because Y = [] is
% equivalent to the disjunction over everything.  but, if Y has any
% and X = [] that's okay because it means that X is a disjunction over
% everything.

eccs_u_sub_disj([], []) :- !.	% trivially okay
eccs_u_sub_disj([], YD) :- !.	% XN is a disjunction of everything so okay
eccs_u_sub_disj(XD, []) :- !, fail.	% YN is a disjunction of everything, fail.
eccs_u_sub_disj(XD, YD) :-		% see that each YNi is in some XNi
	eccs_u_sub_disj0(XD, YD).

eccs_u_sub_disj0(XD, []).
eccs_u_sub_disj0(XD, [H1|T1]) :- !,
	eccs_u_sub_disj1(XD, H1),
	eccs_u_sub_disj0(XD, T1).

eccs_u_sub_disj1(R, c(Node, or(_, L))) :- !,
	eccs_u_sub_disj2(Node,L,R).

% check that each element of the first list occurs somewhere in the
% list of constraints in the second list.

eccs_u_sub_disj2(_,[],_) :- !.
eccs_u_sub_disj2(Node,[H|T],L) :-
	eccs_u_sub_disj3(Node,H,L),
	eccs_u_sub_disj2(Node,T,L).

% check that the first argument appears in one of the disjunctive
% constraints in the list in the second argument.  if there is no
% corresponding disjunctive constraint in the second argument then
% there is a subsumption relation.  i.e., the subsuming argument does
% not constrain that variable at all.

eccs_u_sub_disj3(Node,Y,X) :-
	( eccs_member(c(Node2, or(_, L)), X),
	  Node2 == Node ->
	  eccs_member(Y,L) ;
	  true ).

% simplify a set of disjunctions by creating a new disjunction of the
% intersection of their disjuncts.

disjunction_intersection([],[]).
disjunction_intersection([c(V,or(_,L))|T0],[c(V,or(_,Int))|T]) :-
    remove_disjs(V,T0,Disjs,Rest),
    eccs_intersection([L|Disjs],Int),
    eccs_not_eq(Int,  []),
    disjunction_intersection(Rest,T).

% get all the disjuncts of a variable V.

remove_disjs(V,[],[],[]).
remove_disjs(V,[c(V1,or(_,L1))|T0],[L1|Disjs],Rest) :-
    V == V1,
    remove_disjs(V,T0,Disjs,Rest).
remove_disjs(V,[c(V1,or(_,L1))|T0],Disjs,[c(V1,or(_,L1))|Rest]) :-
    V \== V1,
    remove_disjs(V,T0,Disjs,Rest).

% simplify a set of negations by creating a new negation of the union
% of the elements of the negation.

negation_union([],[]).
negation_union([c(V,not(_,L))|T0],[c(V,not(_,Un))|T]) :-
    remove_negs(V,T0,Negs,Rest),
    union([L|Negs],Un),
    negation_union(Rest,T).

% get all the negative disjuncts of a variable V.

remove_negs(V,[],[],[]).
remove_negs(V,[c(V1,not(_,L1))|T0],[L1|Negs],Rest) :-
    V == V1,
    remove_negs(V,T0,Negs,Rest).
remove_negs(V,[c(V1,not(_,L1))|T0],Negs,[c(V1,not(_,L1))|Rest]) :-
    V \== V1,
    remove_negs(V,T0,Negs,Rest).

% simplify a set of negations and disjunctions be subtracting the
% negative disjuncts from the positive disjuncts.

negation_subtraction([],[],Disjs,Disjs).
negation_subtraction([c(V,not(_,L))|T0],T,Disjs0,Disjs) :-
    remove_disj(V,Disjs0,List0,Disjs1), !,
    subtract(List0,L,List),
    negation_subtraction(T0,T,[c(V,or(_,List))|Disjs1],Disjs).
negation_subtraction([c(V,not(_,L))|T0],[c(V,not(_,L))|T],Disjs0,Disjs) :-
    negation_subtraction(T0,T,Disjs0,Disjs).

% remove a disjunct constraint on V from the list.

remove_disj(V,[c(V0,or(_,List))|T],List,T) :-
    V == V0, !.
remove_disj(V,[c(V0,or(_,L0))|T0],List,[c(V0,or(_,L0))|T]) :-
    remove_disj(V,T0,List,T).

% simplify a set of constraints.

simplify_constraints(X,XN,XD) :-
    separate_neg_and_disj(X,XN0,XD0),
    disjunction_intersection(XD0,XD1),
    negation_union(XN0,XN1),
    negation_subtraction(XN1,XN,XD1,XD).

% if a disjunction contains only one element, then instantiate the
% corresponding variable to that element and eliminate the constraint.

elim_identities([],[]) :- !.
elim_identities([c(V0,or(_,[L0]))|T0],T) :- !,
    V0=L0,
    elim_identities(T0,T).
elim_identities([c(V0,or(_,[L0|L]))|T0],[c(V0,or(_,[L0|L]))|T]) :-
    elim_identities(T0,T).

% adapted from jo's pimple code.
% check the satisfiability of a set of constraints.  if any of the
% constraints are unsatisfiable, then fail.  if any of the constraints
% have been satisfied, then eliminate the constraint.

eccs_q_morph_check_constraints(X,Y) :-
    eccs_q_morph_check_constraints(X,Y,[]).

eccs_q_morph_check_constraints([], Y, Y) :- !.
eccs_q_morph_check_constraints([H|T], R0, R) :- !,
	eccs_q_morph_check_constraint(H, R0, R1),
	eccs_q_morph_check_constraints(T, R1, R).

eccs_q_morph_check_constraint(c(Node, Cons), [c(Node, Cons)|R], R) :-
	eccs_sys_var(Node), !.
eccs_q_morph_check_constraint(c(Node, Cons), R, R) :-
	eccs_sys_atomic(Node), !,
	eccs_q_morph_check_constraint0(c(Node, Cons)).
eccs_q_morph_check_constraint(c('$VAR'(N), Cons), [c('$VAR'(N), Cons)|R], R).
eccs_q_morph_check_constraint(c([_|_], not(_,_)), R, R).

eccs_q_morph_check_constraint0(c(Node, or(Real, L))) :- !,
	Node = Real, 
	eccs_memberchk(Real, L).
eccs_q_morph_check_constraint0(c(Node, not(Real, L))) :- !,
	Node = Real, 
	\+ eccs_memberchk(Real, L).

