;;; This file contains all (hopefully) of the pathname specific constants
;;; and variables and most of the implementation dependent definitions.
;;; The hope is that installing Picasso or porting it to a new
;;; Common LISP will require lots of changes to this file, and very
;;; few elsewhere.
;;; Necessary changes for porting can be found by searching for
;;; #+allegro, since these mark Allegro specific code segments.

;;; These macros are for allowing one set of source code to work
;;; with different versions of Common LISP.

(in-package "PT")

;; These are the constants and macros that allow Picasso to be installed
;; in any directory.  Change *picasso-home-default* to the directory
;; Picasso is installed in.

(defconstant *picasso-home-default* (namestring (pathname "~picasso/")))

;; this will be overridden on startup by the users PICASSOHOME
;; environment variable if there is one.  (this is here for compiling)
(defvar *picasso-home* *picasso-home-default*)

(defmacro picasso-path (path)
  `(concatenate 'string *picasso-home* ,path))

(defmacro load-picasso (path &rest x)
  `(load (concatenate 'string *picasso-home* ,path) ,@x))

;; These are the various pathname dependent declarations from other files.
;; Any declarations that use (picasso-path) are left in their place.
;; There are a number of object files (.o) that will need to recompiled
;; for machines other than SPARCstations, or might even be specific to the
;; hardware we use - these are included just to give examples of how Picasso
;; can make use of other equipment (e.g. video-disk players).

;; from headers/constants.cl

(defconstant *default-font-path* '("/usr/X11/lib/fonts/misc/"
                                   "/usr/X11/lib/fonts/75dpi/"
                                   "/usr/X11/lib/fonts/100dpi/"))

(defconstant *load-gif-object-location*
             (picasso-path "src/toolkit/resource/load-gif.o"))

(defconstant *map-sound-file* (picasso-path "src/widgets/sound/sound.o"))
(defconstant *map-sound-foreign-files* (list (picasso-path "src/widgets/sound/libaudio.a")))

;; from headers/hip.cl

(defvar $link-type-file (picasso-path "lib/hip/lib/link-types"))

(defvar *hyperdoc-info-directory* (picasso-path "lib/hip/hyperdocs/")
  "directory storing hyperdoc data")

;; from headers/cling.cl

(in-package "CLING")
(defconstant *libq-object-location* (PT::picasso-path "src/widgets/cling/libq.o"))
(defconstant *libq-foreign-files* '("/usr/cluster/ingres63/ingres/lib/libingres.a"))

;; from headers/sling.cl

(in-package "SLING")
(defconstant *libsq-object-location* (PT::picasso-path "src/widgets/sling/libsq.o"))
(defconstant *libsq-foreign-files* '("/usr/cluster/ingres63/ingres/lib/libingres.a"))

;; from headers/libpq.cl

(in-package "LIBPQ")
(defconstant *libpq-dot-o-file* (PT::picasso-path "src/widgets/libpq/libpq.o"))

(in-package "PT")
;; from headers/variables.cl

(defvar *picasso-defaults-path* (picasso-path "lib/picasso-defaults"))

;; widgets/etc/start-func.cl refers to /etc/hostname.le0 to get information
;; about the machine Picasso is running on.



;; These are the macros defined so that Allegro/Lucid specific functions
;; can be done with one function.

(defmacro remove-foreign-symbol (sym)
  `(#+allegro ff::remove-entry-point
    #+lucid unintern-foreign-symbol ,sym))

(defmacro intern-keyword (sym)
  `(intern (symbol-name ,sym) "KEYWORD"))

(defmacro get-environment (var)
  `(#+allegro excl::get-environ
    #+lucid environment-variable ,var))

(defmacro get-cur-directory ()
  `(#+allegro excl::current-directory
    #+lucid working-directory))

;; lucid run-program can't handle ~user/... pathnames, this resolves them
;; into a useable form.  Doesn't handle "~", works for "~/".
(defmacro resolve-pathname (path)
  #+allegro `,path
  #+lucid `(namestring (parse-namestring ,path)))

;; allegro wants command and arguments in one string.
;; lucid wants arguments as a list of strings.
;; shell-command takes a list of strings, combines into 1 string in Allegro.
(defmacro shell-command (command &rest args)
  #+allegro `(let ((result ,command))
	       (excl::run-shell-command 
		(dolist (elt ',args result) (setf result (concatenate 'string result " " elt)))))
  #+lucid `(run-program ,command :arguments ,args))

(defun command-line-args ()
  #+allegro (system:command-line-arguments)
  #+lucid (do* ((i 0 (1+ i))
	       (last-arg (command-line-argument i) (command-line-argument i))
	       (args nil))
	      ((equal last-arg nil) (reverse args))
	      (declare (fixnum i))
	      (setq args (cons last-arg args))))

(defmacro set-prompt (new-prompt)
  `(setq #+allegro tpl:*prompt*
	 #+lucid *prompt* ,new-prompt))

;; read-database is used to read the graphics data from the cimtool database
(defmacro read-database (stream eoferr eofval)
  `(read ,stream ,eoferr ,eofval #+allegro t #+lucid nil))
