/*
 *	bytecode.c
 *		Einlesen und relozieren des Bytecodes
 *
 */

#include <fcntl.h>
#ifdef STDC_HEADERS
#include <stdlib.h>
#endif
#include "sowam.h"
#include "af-area.h"
#include "bytecode.h"

int initial_symbol_count;

static void
init_symbols(af, names, n_af)
     struct ex_af_entry *af;
     char *names;
     long n_af;
{
  int i, index;
  struct ex_af_entry *p;

  init_builtin_symbols();

  for (p = af; p < af + n_af; p++)
    {
      index = lookup_symbol(&names[p->text_addr], (int)p->arity);
      if (p->code1_addr != -1)
	{
	  if (p->code2_addr != -1)
	    {
	      /* Funktion */
	      b_af_area[index].narrow_addr = b_code + p->code1_addr;
	      b_af_area[index].rewrite_addr = b_code + p->code2_addr;
	      b_af_area[index].pred_addr = NULL;
	    }
	  else
	    {
	      /* Praedikat */
	      b_af_area[index].narrow_addr = NULL;
	      b_af_area[index].rewrite_addr = NULL;
	      b_af_area[index].pred_addr = b_code + p->code1_addr;
	    }
	}
      else if (p->code2_addr != -1)
	{
	  /* darf nicht vorkommen */
#ifdef DEBUG
	  fprintf(stderr, "Nr. %d: %s/%d, code2_addr = %lx\n",
		  p-af, &names[p->text_addr], p->arity,
		  p->code2_addr);
#endif
	  fatal_error("illegal symbol specification");
	}
      else
	{
	  /* Konstruktor */
	  b_af_area[index].narrow_addr = NULL;
	  b_af_area[index].rewrite_addr = NULL;
	  b_af_area[index].pred_addr = NULL;
	}
      p->arity = index;		/* fuer Relozierung merken */

      b_af_area[index].op_type = p->op_type;
      b_af_area[index].prio = p->prio;
    }

  initial_symbol_count = af_ptr - b_af_area;
}

int
table_cmp(a, b)
     const void *a, *b;
{
  long l;
  /* Symbol-Index und -Stelligkeit gemeinsam vergleichen */
  l = GET_LONG(a) - GET_LONG(b);
  return l < 0 ? -1 : l > 0 ? 1 : 0;
}

/* Reloziert den Code anhand der Relocation-Tabelle reloc. In
   af[i].arity steht der neue Index des Symbols i (siehe oben) */

static void
relocate(reloc, n_rel, af)
     struct relocation *reloc;
     long n_rel;
     struct ex_af_entry *af;
{
  struct relocation *reloc_end = reloc + n_rel;
  int index, length, i;
  long address, swtab, p;

  while (reloc < reloc_end)
    {
      switch (reloc->tag)
	{
	case REL_SYM:
	  index = GET_WORD((long)b_code + reloc->param);
	  SET_WORD((long)b_code + reloc->param, af[index].arity);
	  break;

	case REL_CODE:
	  address = GET_LONG((long)b_code + reloc->param);
	  address = (long)(b_code + address);
	  SET_LONG((long)b_code + reloc->param, address);
	  break;
	  
	case REL_TABLE:
	  address = (long)b_code + reloc->param;
	  length = GET_WORD(address);
	  swtab = address + 2;
	  for (i = 0, p = swtab; i < length; i++, p += 8)
	    {
	      index = GET_WORD(p + 2);
	      SET_WORD(p + 2, af[index].arity);
	      address = GET_LONG(p + 4);
	      address = (long)(b_code + address);
	      SET_LONG(p + 4, address);
	    }

	  /* Switch-Tabelle sortieren */
	  qsort(swtab, length, 8, table_cmp);
	  break;

	default:
	  fatal_error("unknown relocation tag");
	}
      reloc++;
    }
}

void
read_bytecode(name)
     char *name;
{
  int fd;
  char file_name[1024];
  struct ex_af_entry *af_area;
  char *text_area;
  struct relocation *reloc_table;
  struct codeheader header;

  strcpy(file_name, name);
  strcat(file_name, ".byt");
  fd = open (file_name, O_RDONLY);
  if (fd < 0)
    {
      strcpy(file_name, name);
      fd = open(file_name, O_RDONLY);
    }
  if (fd < 0)
    error(name);

  if (read(fd, &header, sizeof(struct codeheader)) != sizeof(struct codeheader))
    fatal_error("illegal file type");

  if (strncmp(header.ch_magic, "FALF", 4))
    fatal_error("illegal file type");

  if (header.ch_version != bytecode_version)
    fatal_error("wrong bytecode version");

  b_code = (code_addr) xmalloc(header.ch_ca_max);
  af_area = (struct ex_af_entry *) xmalloc(header.ch_af_max);
  text_area = (char *) xmalloc(header.ch_ta_max);
  reloc_table = (struct relocation *) xmalloc(header.ch_rel_max);
  s_regs.p = b_code + header.ch_entry;

  if (read(fd, b_code, header.ch_ca_max) != header.ch_ca_max ||
      read(fd, af_area, header.ch_af_max) != header.ch_af_max ||
      read(fd, text_area, header.ch_ta_max) != header.ch_ta_max ||
      read(fd, reloc_table, header.ch_rel_max) != header.ch_rel_max)
    fatal_error("unexpected end of file");
  close (fd);

  af_max += header.ch_af_max / sizeof(struct ex_af_entry);
  init_symbols(af_area, text_area, header.ch_af_max / sizeof(struct ex_af_entry));
  relocate(reloc_table, header.ch_rel_max / sizeof(struct relocation), af_area);
}
