/*
 *
 *	sowam_c
 *		Bytecode-Interpreter fuer die SOWAM
 *
 *
 *	FILE
 *		interp.c
 *
 *	PURPOSE
 *		Interpreter fuer den eigelesenen Bytecode
 *		
 *
 *	AUTHORS
 *		Mike Wilhelm, Frank Lindert, Volker Siebert, Andreas Schwab
 *
 */

#ifdef __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif
#include "sowam.h"
#include "debug.h"

void sowam_setup()
{
  /* s_regs.p wird in read_bytecode gesetzt */
  s_regs.cp  = 0;
  s_regs.b   = 0;
  s_regs.r   = 0;
  s_regs.e   = 0;
  s_regs.t   = TRL_MAX;
  s_regs.h   = HP_MIN;
  s_regs.tfp = FAIL;
  s_regs.rfp = FAIL;
  s_regs.hr  = 0;
  s_regs.ts  = UNDEF;
  s_regs.ao  = UNDEF;
  s_regs.op  = (struct occ_stack *)OS_MIN;
  s_regs.op->occ_op = 0;
  s_regs.op->occ_pred = 0;
  s_regs.om  = s_regs.op->occ_os;
  s_regs.or  = OS_MAX;

  s_flags.os_valid = FALSE;
  s_flags.rw       = READ;

  s_glregs.hb = LS_MIN;
  s_glregs.sh = 0;
#ifdef DEBUG_BOX
  s_regs.p_box = 1;
  s_glregs.box_count = 2;
#endif
  s_glflags.error  = FALSE;

#ifdef STAT
  heap_used = HP_MIN;
  ls_used = LS_MIN;
  os_used = OS_MIN;
  trail_used = TRL_MAX;
  num_bos = 0;
#endif

  if (s_glflags.debug == 0)
    install_debug_handler();
}

static
#ifdef __STDC__
void trace(code_addr p, char *fmt,...)
#else
void trace(va_alist)
     va_dcl
#endif
{
  va_list args;
#ifndef __STDC__
  code_addr p;
  char *fmt;
#endif

#ifdef __STDC__
  va_start(args, fmt);
#else
  va_start(args);
  p = va_arg(args, code_addr);
  fmt = va_arg(args, char *);
#endif
  printf("L%d: ", LAB(p));
  vprintf(fmt, args);
  putchar('\n');
  va_end(args);
}

static
void illegal_opcode_error(p, oc)
     code_addr p;
     int oc;
{
#ifdef PILS
  char xyz[40];

  sprintf(xyz, "0x%x at %d", oc, LAB(p));
  label("illegal opcode:", xyz, "Result: NO");
#else
  extern char *program_name;

  fprintf(stderr, "%s: illegal opcode: 0x%x at %d\n", program_name,
	  oc, LAB(p));
#endif

  exit(1);
}

void interpret()
{
  int opcode;
  term *top_ls;
  long arg1,arg2,arg3,arg4;
  int occ_size,j,arity;
  term *t,term1;
  code_addr tp;
  struct environment *env;

  forever
    {
      opcode = *s_regs.p++;

      switch (MAJOR_OPCODE(opcode))
	{
	case 0:

	  switch (MINOR_OPCODE(opcode))
	    {
	    case 1:
	      /*
	       * allocate
	       */

	      TRACE0("allocate");

	      top_ls = TOP_LS();
	      CHECK_LS(top_ls + isizeof(struct environment) + 20);
	      env = (struct environment *)top_ls;
	      env->env_e = s_regs.e;
	      env->env_cp = s_regs.cp;
#ifdef DEBUG_BOX
	      env->env_cpb = s_regs.cp_box;
#endif
	      env->env_r = s_regs.r;
	      env->env_hr = s_regs.hr;
	      s_regs.e = env;
	      break;


	    case 2:
	      /*
	       * deallocate
	       */

	      TRACE0("deallocate");

	      env = ENVIRON;
	      s_regs.e = env->env_e;
	      s_regs.cp = env->env_cp;
#ifdef DEBUG_BOX
	      s_regs.cp_box = env->env_cpb;
#endif
	      s_regs.r = env->env_r;
	      s_regs.hr = env->env_hr;
	      break;


	    case 3:
	      /*
	       * allocate-occ
	       */

	      TRACE0("allocate-occ");

	      CHECK_OS(isizeof(struct occ_stack));

	      {
		struct occ_stack *oop = s_regs.op;

		s_regs.op = (struct occ_stack *)s_regs.om;
		s_regs.op->occ_op =
		  s_regs.op->occ_pred = oop;
		oop->occ_om = s_regs.om;
		oop->occ_ts = s_regs.ts;
		oop->occ_ao = s_regs.ao;
		oop->occ_or = s_regs.or;
		oop->occ_cp = FAIL;
		s_regs.om = s_regs.op->occ_os;
	      }
	      break;


	    case 4:
	      /*
	       * deallocate-occ
	       */

	      TRACE0("deallocate-occ");

	      if (s_regs.b && s_regs.b->bt_regs.op >= s_regs.op->occ_pred)
		{
		  struct occ_stack *oop = s_regs.op->occ_op;
		  /*
		   * logischer Vorgaenger durch Backtrackpunkt 
		   * geschuetzt -> kopieren
		   */
		  occ_size = OCC_SIZE(s_regs.op->occ_pred);
		  s_regs.om = s_regs.op->occ_os + occ_size;
		  CHECK_OS(0);

		  /*
		   * Occurences kopieren 
		   */
		  bcopy(s_regs.op->occ_pred, s_regs.op,
			(occ_size-1) * sizeof(term *) + sizeof(struct occ_stack));
		  /* 
		   * logischen Vorgaenger aktualisieren 
		   */
		  s_regs.op->occ_op = oop;
		}
	      else if (s_regs.op->occ_pred == s_regs.op->occ_op)
		{
		  /*
		   * obersten Occurence-Stack loeschen
		   */
		  s_regs.op = s_regs.op->occ_op;
		  s_regs.om = s_regs.op->occ_om;
		}
	      else
		internal_error("deallocate-occ");

	      s_regs.ao = s_regs.op->occ_ao;
	      s_regs.ts = s_regs.op->occ_ts;
	      s_regs.or = s_regs.op->occ_or;
	      break;


	    case 5:
	      /*
	       * proceed
	       */

	      TRACE0("proceed");

	      if (s_regs.cp == 0)
		{
		  print_time();
		  stop(YES);
		}
#ifdef DEBUG_BOX
	      s_regs.p_box = s_regs.cp_box;
#endif
	      s_regs.p = s_regs.cp;
	      break;


	    case 6:
	      /*
	       * unify-nil
	       */

	      TRACE0("unify-nil");

	      if (s_flags.rw == READ)
		{
		  t = s_glregs.sh;
		  DEREF_TERMP(t);
		  if (TAG(*t) == T_UNDEF)
		    {
		      if (trail_undef(t))
			{
			  SET_CONST(*t, NIL);
			  s_glregs.sh++;
			}
		    }
		  else if (TAG(*t) == T_AF && VAL(*t) == NIL)
		    s_glregs.sh++;
		  else
		    fail();
		}
	      else
		{
		  SET_CONST(*s_regs.h, NIL);
		  s_regs.h++;
		}
	      break;


	    case 7:
	      /*
	       * fail
	       */

	      TRACE0("fail");

	      fail();
	      break;


	    case 8:
	      /*
	       * stop
	       */

	      TRACE0("stop");

	      stop(YES);
	      break;


	    case 9:
	      /*
	       * pop-occ
	       */

	      TRACE0("pop-occ");

	      do
		{
		  POP_OCC();
		  DEREF_AO();
		}
	      while (s_regs.ao != UNDEF &&
		     (LOC(*s_regs.ao) != SKEL || !IS_FUNCTION(*s_regs.ao)));
	      break;


	    case 10:
	      /*
	       * execute-narrowing(ao)
	       * Ist der Occurrence-Stack leer,
	       * wird P := CP; sonst wird 'narrow' ausgefuehrt
	       */

	      TRACE0("execute-narrowing(ao)");

	      if (s_regs.ao == UNDEF)
		{
#ifdef DEBUG_BOX
		  s_regs.p_box = s_regs.cp_box;
#endif
		  s_regs.p = s_regs.cp;
		}
	      else
		{
#ifdef DEBUG_BOX
		  s_regs.p_box = s_glregs.box_count++;
#endif
		  if (s_glflags.debug)
		    debug(D_NARROW, s_regs.cp, 0);

		  /* Argumentregister setzen */
		  SET_TAG(term1, T_VAR);
		  SET_LOC(term1, SKEL);
		  arity = ARITY(VAL(*s_regs.ao));
		  for (j = 1; j <= arity; j++)
		    {
		      SET_VAL(term1, INDEX(s_regs.ao+j));
		      X_REG(j) = term1;
		    }
		  s_regs.p = AF_NARROW(VAL(*s_regs.ao));
#ifdef DEBUG
		  if (s_regs.p == 0)
		    internal_error("execute-narrowing(ao)");
#endif
		}
	      break;


	    case 11:
	      /*
	       * execute-rewriting(ao)
	       * Ist der Occurrence-Stack leer,
	       * wird P := CP; sonst wird die naechste
	       * Rewrite-Regel ausgefuehrt
	       */

	      TRACE0("execute-rewriting(ao)");

	      s_regs.rfp = s_regs.tfp = FAIL;
	      if (s_regs.ao == UNDEF)
		{
		  if (s_regs.op->occ_cp == FAIL)
		    {
		      s_regs.p = s_regs.cp;
#ifdef DEBUG_BOX
		      s_regs.p_box = s_regs.cp_box;
#endif
		    }
		  else
		    {
		      s_regs.p = s_regs.op->occ_cp;
#ifdef DEBUG_BOX
		      s_regs.p_box = s_regs.op->occ_cpb;
#endif
		    }
		}
	      else
		{
#ifdef DEBUG_BOX
		  s_regs.p_box = s_glregs.box_count++;
#endif
		  if (s_glflags.debug)
		    debug(D_REWRITE,
			  (s_regs.op->occ_cp == FAIL ?
			   s_regs.cp : s_regs.op->occ_cp), 0);

		  DEREF_AO();
		  SET_TAG(term1, T_VAR);
		  SET_LOC(term1, SKEL);
		  arity = ARITY(VAL(*s_regs.ao));
		  for (j = 1; j <= arity; j++)
		    {
		      SET_VAL(term1, INDEX(s_regs.ao+j));
		      X_REG(j) = term1;
		    }
		  s_regs.p = AF_REWRITE(VAL(*s_regs.ao));
#ifdef DEBUG
		  if (s_regs.p == 0)
		    internal_error("execute-rewriting(ao)");
#endif
		}
	      break;


	    case 12:
	      /*
	       * inner-reflection
	       */

	      TRACE0("inner-reflection");

	      TRAIL_AO();
	      CHANGE_ENV(*s_regs.ao);
	      POP_OCC();
	      if (s_regs.ao == UNDEF)
		{
#ifdef DEBUG_BOX
		  s_regs.p_box = s_regs.cp_box;
#endif
		  s_regs.p = s_regs.cp;
		}
	      else
		{
#ifdef DEBUG_BOX
		  s_regs.p_box = s_glregs.box_count++;
#endif
		  if (s_glflags.debug)
		    debug(D_NARROW, s_regs.cp, 0);

		  DEREF_AO();
		  /* Argumentregister setzen */
		  SET_TAG(term1, T_VAR);
		  SET_LOC(term1, SKEL);
		  arity = ARITY(VAL(*s_regs.ao));
		  for (j = 1; j <= arity; j++)
		    {
		      SET_VAL(term1, INDEX(s_regs.ao+j));
		      X_REG(j) = term1;
		    }
		  s_regs.p = AF_NARROW(VAL(*s_regs.ao));
		  if (s_regs.p == 0)
		    internal_error("reflection");
		}
	      break;


	    case 13:
	      /*
	       * put-nil-occ
	       */

	      TRACE0("put-nil-occ");

	      TRAIL_AO();
	      SET_CONST(*s_regs.ao, NIL);
	      break;


	    case 14:
	      /*
	       * put-list-occ
	       */

	      TRACE0("put-list-occ");

	      TRAIL_AO();
	      CHECK_HEAP(2);
	      SET_TERM(*s_regs.ao, T_LIST, s_regs.h);
	      break;


	    case 15:
	      /*
	       * trust-me-else-fail
	       */

	      TRACE0("trust-me-else-fail");

	      TRUST_ME_ELSE_FAIL();
	      break;


	    default:
	      illegal_opcode_error(s_regs.p-1,opcode);
	      break;

	    }
	  break;


	case 1:

	  switch (MINOR_OPCODE(opcode))
	    {
	    case 0:
	      /*
	       * r-trust-me-else-fail
	       */

	      TRACE0("r-trust-me-else-fail");

	      s_regs.re = s_regs.e;
	      if (s_regs.tfp != FAIL)
		s_regs.tfp = FAIL;
	      else
		s_regs.rfp = FAIL;
	      break;


	    case 1:
	      /*
	       * rebuild-occ-stack
	       */

	      TRACE0("rebuild-occ-stack");

#ifdef DEBUG
	      if (s_regs.rfp != FAIL || s_regs.tfp != FAIL)
		internal_error("rebuild-occ-stack");
#endif
	      s_regs.r = s_regs.e->env_r;
	      s_regs.hr = s_regs.e->env_hr;
	      if (s_flags.os_valid)
		{
		  /* Hilfsstack zurueckkopieren */
		  occ_size = s_regs.op->occ_op->occ_or - s_regs.or;
		  bcopy(s_regs.or, s_regs.om, occ_size * sizeof(term *));
		  s_regs.om += occ_size;
		}
	      else
		{
		  /* Neuen Occurence-Stack anlegen */
		  s_regs.om = s_regs.op->occ_os;
#ifdef STAT
		  num_bos++;
#endif
		  build_occ_stack(s_regs.ts);
		}
	      s_regs.or = s_regs.op->occ_op->occ_or;
	      do
		{
		  POP_OCC();
		  DEREF_AO();
		}
	      while (s_regs.ao != UNDEF &&
		     (LOC(*s_regs.ao) != SKEL || !IS_FUNCTION(*s_regs.ao)));
	      break;


	    case 2:
	      /*
	       * write-nil
	       */

	      TRACE0("write-nil");

	      SET_CONST(*s_regs.h, NIL);
	      s_regs.h++;
	      break;


	    case 3:
	      /*
	       * read-nil
	       */

	      TRACE0("read-nil");

	      t = s_glregs.sh;
	      DEREF_TERMP(t);
	      if (TAG(*t) == T_AF && VAL(*t) == NIL)
		s_glregs.sh++;
	      else
		fail();
	      break;

	    case 4:
	      /*
	       * invalid-os
	       */

	      TRACE0("invalid-os");

	      if (s_regs.or < s_regs.op->occ_op->occ_or)
		s_flags.os_valid = FALSE;
	      break;


	    case 5:
	      /*
	       * push-act-occ
	       * entspricht push-occ-ao
	       */

	      TRACE0("push-act-occ");

	      PUSH_OS_ENTRY(s_regs.ao);
	      break;


	    case 6:
	      /*
	       * l-trust-me-else-fail
	       */

	      TRACE0("l-trust-me-else-fail");

	      s_regs.b = *(struct backtrack **)&Y_REG(1);
	      s_regs.tfp = s_regs.b->bt_regs.tfp;
	      s_regs.rfp = s_regs.b->bt_regs.rfp;
	      s_regs.om = s_regs.b->bt_regs.om;
	      s_regs.op = s_regs.b->bt_regs.op;
	      s_regs.re = s_regs.b->bt_regs.re;
	      s_regs.t = s_regs.b->bt_regs.t;
#ifdef DEBUG_BOX
	      s_regs.tfp_box = s_regs.b->bt_regs.tfp_box;
	      s_regs.rfp_box = s_regs.b->bt_regs.rfp_box;
#endif
	      TRUST_ME_ELSE_FAIL();
	      break;


	    case 7:
	      /*
	       * call-rewriting(ao)
	       */

	      TRACE0("call-rewriting(ao)");

	      s_regs.rfp = s_regs.tfp = FAIL;
	      s_flags.os_valid = TRUE;
	      if (s_regs.ao != UNDEF)
		{
#ifdef DEBUG_BOX
		  s_regs.op->occ_cpb = s_regs.p_box;
		  s_regs.p_box = s_glregs.box_count++;
#endif
		  if (s_glflags.debug)
		    debug(D_REWRITE, s_regs.p, 0);

		  SET_TAG(term1, T_VAR);
		  SET_LOC(term1, SKEL);
		  arity = ARITY(VAL(*s_regs.ao));
		  for (j = 1; j <= arity; j++)
		    {
		      SET_VAL(term1, INDEX(s_regs.ao+j));
		      X_REG(j) = term1;
		    }
		  s_regs.r = TOP_LS();
		  s_regs.hr = s_regs.h;
		  s_regs.op->occ_cp = s_regs.p;
		  s_regs.p = AF_REWRITE(VAL(*s_regs.ao));
#ifdef DEBUG
		  if (s_regs.p == 0)
		    internal_error("call-rewriting(ao)");
#endif
		}
	      break;


	    case 8:
	      /*
	       * reject
	       */

	      TRACE0("reject");

	      t = s_regs.ts;
	      DEREF_TERMP(t);
	      if (TAG(*t) == T_AF && VAL(*t) == EQ_SYM)
		reject_terms(t[1], t[2]);
	      break;


	    case 9:
	      /*
	       * copy-pop-occ
	       */

	      TRACE0("copy-pop-occ");

	      do
		{
		  *--s_regs.or = s_regs.ao;
		  POP_OCC();
		  DEREF_AO();
		}
	      while (s_regs.ao != UNDEF &&
		     (LOC(*s_regs.ao) != SKEL || !IS_FUNCTION(*s_regs.ao)));
	      break;


	    case 10:
	      /*
	       * reflection
	       */

	      TRACE0("reflection");

	      t = s_regs.ts;
	      DEREF_TERMP(t);
	      if (TAG(*t) == T_AF && VAL(*t) == EQ_SYM)
		unify(t[1], t[2]);
	      break;


	    default:
	      illegal_opcode_error(s_regs.p-1,opcode);
	      break;

	    }
	  break;


	case 2:
	  arg1 = OP2ARG(opcode);
	  switch (MINOR_OPCODE(opcode))
	    {
	    case 0:
	      /*
	       * write-void
	       */

	      TRACE1("write-void(%d)",arg1);

	      CHECK_HEAP(arg1);
	      while (arg1-- > 0)
		{
		  SET_ETERM(*s_regs.h, T_UNDEF, s_regs.h);
		  s_regs.h++;
		}
	      break;


	    case 1:
	      /*
	       * put-nil-x
	       */

	      TRACE1("put-nil(X%d)",arg1);

	      SET_CONST(X_REG(arg1), NIL);
	      break;


	    case 2:
	      /*
	       * get-nil
	       */

	      TRACE1("get-nil(X%d)",arg1);

	      t = &X_REG(arg1);
	      DEREF_TERMP(t);
	      if (TAG(*t) == T_UNDEF)
		{
		  if (trail_undef(t))
		    SET_CONST(*t, NIL);
		}
	      else if (TAG(*t) != T_AF || VAL(*t) != NIL)
		fail();
	      break;


	    case 3:
	      /*
	       * put-list-x
	       */

	      TRACE1("put-list(X%d)",arg1);

	      CHECK_HEAP(2);
	      SET_TERM(X_REG(arg1), T_LIST, s_regs.h);
	      break;


	    case 4:
	      /*
	       * get-list
	       */

	      TRACE1("get-list(X%d)",arg1);

	      t = &X_REG(arg1);
	      DEREF_TERMP(t);
	      if (TAG(*t) == T_UNDEF)
		{
		  if (trail_undef(t))
		    {
		      CHECK_HEAP(2);
		      SET_TERM(*t, T_LIST, s_regs.h);
		      s_flags.rw = WRITE;
		    }
		}
	      else if (TAG(*t) == T_LIST)
		{
		  s_flags.rw = READ;
		  s_glregs.sh = &REF(*t);
		}
	      else
		fail();
	      break;


	    case 5:
	      /*
	       * unify-void
	       */

	      TRACE1("unify-void(%d)",arg1);

	      if (s_flags.rw == READ)
		s_glregs.sh += arg1;
	      else
		{
		  CHECK_HEAP(arg1);
		  while (arg1-- > 0)
		    {
		      SET_ETERM(*s_regs.h, T_UNDEF, s_regs.h);
		      s_regs.h++;
		    }
		}
	      break;


	    case 6:
	      /*
	       * put-unsafe-value-occ-y
	       */

	      TRACE1("put-unsafe-value-occ(Y%d)", arg1);

	      term1 = Y_REG(arg1);
	      DEREF_TERM(term1);
	      if (TAG(term1) == T_UNDEF && VAL(term1) > INDEX(s_regs.e))
		{
		  if (trail_undef(&REF(term1)))
		    {
		      TRAIL_AO();

		      /* neue Variable auf den Heap schreiben */
		      SET_TERM(*s_regs.ao, T_UNDEF, s_regs.ao);

		      /* Y-Register, auf das deref_term(arg1) zeigt setzen */
		      SET_REF(REF(term1), s_regs.ao);
		      SET_TAG(REF(term1), T_VAR);
		    }
		}
	      else
		{
		  TRAIL_AO();
		  CHANGE_ET(term1);
		  *s_regs.ao = term1;
		}
	      break;

		
	    case 7:
	      /*
	       * unify-variable-x
	       */

	      TRACE1("unify-variable(X%d)",arg1);

	      if (s_flags.rw == READ)
		{
		  term1 = *s_glregs.sh++;
		  CHANGE_ET(term1);
		  X_REG(arg1) = term1;
		}
	      else
		{
		  SET_TERM(*s_regs.h, T_UNDEF, s_regs.h);
		  SET_TERM(X_REG(arg1), T_VAR, s_regs.h);
		  s_regs.h++;
		}
	      break;


	    case 8:
	      /*
	       * write-variable-x
	       */

	      TRACE1("write-variable(X%d)",arg1);

	      SET_TERM(*s_regs.h, T_UNDEF, s_regs.h);
	      SET_TERM(X_REG(arg1), T_VAR, s_regs.h);
	      s_regs.h++;
	      break;


	    case 9:
	      /*
	       * unify-variable-y
	       */

	      TRACE1("unify-variable(Y%d)",arg1);

	      if (s_flags.rw == READ)
		{
		  term1 = *s_glregs.sh++;
		  CHANGE_ET(term1);
		  Y_REG(arg1) = term1;
		}
	      else
		{
		  SET_TERM(*s_regs.h, T_UNDEF, s_regs.h);
		  SET_TERM(Y_REG(arg1), T_VAR, s_regs.h);
		  s_regs.h++;
		}
	      break;


	    case 10:
	      /*
	       * unify-value-x
	       */

	      TRACE1("unify-value(X%d)",arg1);

	      if (s_flags.rw == READ)
		unify(*s_glregs.sh++,X_REG(arg1));
	      else
		{
		  term1 = X_REG(arg1);
		  DEREF1_TERM(term1);
		  CHANGE_ET(term1);
		  *s_regs.h++ = term1; /* Occur-check!!! */
		}
	      break;


	    case 11:
	      /*
	       * write-value-x
	       */

	      TRACE1("write-value(X%d)",arg1);

	      term1 = X_REG(arg1);
	      DEREF1_TERM(term1);
	      CHANGE_ET(term1);
	      *s_regs.h++ = term1; /* Occur-check!!! */
	      break;


	    case 12:
	      /*
	       * unify-value-y
	       */

	      TRACE1("unify-value(Y%d)",arg1);

	      if (s_flags.rw == READ)
		unify(*s_glregs.sh++,Y_REG(arg1));
	      else
		{
		  term1 = Y_REG(arg1);
		  DEREF1_TERM(term1);
		  CHANGE_ET(term1);
		  *s_regs.h++ = term1; /* Occur-check!!!! */
		}
	      break;


	    case 13:
	      /*
	       * unify-local-value-x
	       */

	      TRACE1("unify-local-value(X%d)",arg1);

	      if (s_flags.rw == READ)
		unify(*s_glregs.sh++,X_REG(arg1));
	      else
		{
		  CHECK_HEAP(1);
		  t = s_regs.h;
		  term1 = X_REG(arg1);
		  DEREF_TERM(term1);
		  if (TAG(term1) == T_UNDEF && IS_LS_ADDR(VAL(term1)))
		    {
		      if (trail_undef(&REF(term1)))
			{
			  SET_TERM(*t, T_UNDEF, s_regs.h);
			  REF(term1) = *t;
			  CHANGE_ET(REF(term1));
			  X_REG(arg1) = REF(term1);
			  s_regs.h++;
			}
		    }
		  else
		    {
		      CHANGE_ET(term1);
		      X_REG(arg1) = *t = term1;	/* Occur-check!!!! */
		      s_regs.h++;
		    }
		}
	      break;


	    case 14:
	      /*
	       * write-local-value-x
	       */

	      TRACE1("write-local-value(X%d)",arg1);

	      CHECK_HEAP(1);
	      t = s_regs.h;
	      term1 = X_REG(arg1);
	      DEREF_TERM(term1);
	      if (TAG(term1) == T_UNDEF && IS_LS_ADDR(VAL(term1)))
		{
		  if (trail_undef(&REF(term1)))
		    {
		      SET_TERM(*t, T_UNDEF, s_regs.h);
		      REF(term1) = *t;
		      CHANGE_ET(REF(term1));
		      X_REG(arg1) = REF(term1);
		      s_regs.h++;
		    }
		}
	      else
		{
		  CHANGE_ET(term1);
		  X_REG(arg1) = *t = term1; /* Occur-check!!! */
		  s_regs.h++;
		}
	      break;


	    case 15:
	      /*
	       * unify-local-value-y
	       */

	      TRACE1("unify-local-value(Y%d)",arg1);

	      if (s_flags.rw == READ)
		unify(*s_glregs.sh++,Y_REG(arg1));
	      else
		{
		  CHECK_HEAP(1);
		  t = s_regs.h;
		  term1 = Y_REG(arg1);
		  DEREF_TERM(term1);
		  if (TAG(term1) == T_UNDEF && IS_LS_ADDR(VAL(term1)))
		    {
		      if (trail_undef(&REF(term1)))
			{
			  SET_TERM(*t, T_UNDEF, s_regs.h);
			  REF(term1) = *t;
			  CHANGE_ET(REF(term1));
			  s_regs.h++;
			}
		    }
		  else
		    {
		      CHANGE_ET(term1);
		      *t = term1; /* Occur-check!!! */
		      s_regs.h++;
		    }
		}
	      break;


	    default:
	      illegal_opcode_error(s_regs.p-1,opcode);
	      break;

	    }
	  break;


	case 3:
	  arg1 = OP2ARG(opcode);
	  switch (MINOR_OPCODE(opcode))
	    {
	    case 0:
	      /*
	       * put-var-occ-x
	       */

	      TRACE1("put-var-occ(X%d)",arg1);

	      TRAIL_AO();

	      /* X-Register setzen */
	      SET_TERM(X_REG(arg1), T_VAR, s_regs.ao);

	      SET_TERM(*s_regs.ao, T_UNDEF, s_regs.ao);
	      break;


	    case 1:
	      /*
	       * put-var-occ-y
	       */

	      TRACE1("put-var-occ(Y%d)",arg1);

	      TRAIL_AO();

	      /* Y-Register setzen */
	      SET_TERM(Y_REG(arg1), T_VAR, s_regs.ao);

	      SET_TERM(*s_regs.ao, T_UNDEF, s_regs.ao);
	      break;


	    case 2:
	      /*
	       * put-value-occ-x
	       */

	      TRACE1("put-value-occ(X%d)",arg1);

	      TRAIL_AO();
	      *s_regs.ao = X_REG(arg1);
	      break;


	    case 3:
	      /*
	       * put-value-occ-y
	       */

	      TRACE1("put-value-occ(Y%d)",arg1);

	      TRAIL_AO();
	      *s_regs.ao = Y_REG(arg1);
	      break;


	    case 4:
	      /*
	       * load-occ-x
	       */

	      TRACE1("load-occ(X%d)",arg1);

	      s_regs.ao = &REF(X_REG(arg1));
	      break;


	    case 5:
	      /*
	       * set-begin-of-term-x
	       */

	      TRACE1("set-begin-of-term(X%d)",arg1);

	      s_flags.os_valid = TRUE;
	      s_regs.ts = &REF(X_REG(arg1));
	      break;


	    case 6:
	      /*
	       * push-occ-x
	       */

	      TRACE1("push-occ(X%d)",arg1);

	      PUSH_OS_ENTRY(&REF(X_REG(arg1)));
	      break;


	    case 7:
	      /*
	       * write-variable-y
	       */

	      TRACE1("write-variable(Y%d)",arg1);

	      SET_TERM(*s_regs.h, T_UNDEF, s_regs.h);
	      SET_TERM(Y_REG(arg1), T_VAR, s_regs.h);
	      s_regs.h++;
	      break;


	    case 8:
	      /*
	       * write-value-y
	       */

	      TRACE1("write-value(Y%d)",arg1);

	      term1 = Y_REG(arg1);
	      DEREF_TERM(term1);
	      CHANGE_ET(term1);
	      *s_regs.h++ = term1;
	      break;


	    case 9:
	      /*
	       * write-local-value-y
	       */

	      TRACE1("write-local-value(Y%d)",arg1);

	      CHECK_HEAP(1);
	      t = s_regs.h;
	      term1 = Y_REG(arg1);
	      DEREF_TERM(term1);
	      if (TAG(term1) == T_UNDEF && IS_LS_ADDR(VAL(term1)))
		{
		  if (trail_undef(&REF(term1)))
		    {
		      SET_TERM(*t, T_UNDEF, s_regs.h);
		      REF(term1) = *t;
		      CHANGE_ET(REF(term1));
		      s_regs.h++;
		    }
		}
	      else
		{
		  CHANGE_ET(term1);
		  *t = term1;	/* Occur-check!!! */
		  s_regs.h++;
		}
	      break;


	    case 11:
	      /*
	       * match-list
	       */

	      TRACE1("match-list(X%d)",arg1);

	      t = &X_REG(arg1);
	      DEREF_TERMP(t);
	      if (TAG(*t) == T_LIST)
		s_glregs.sh = &REF(*t);
	      else
		fail();
	      break;


	    case 12:
	      /*
	       * load-occ-y
	       */

	      TRACE1("load-occ(Y%d)",arg1);

	      s_regs.ao = &REF(Y_REG(arg1));
	      break;


	    case 13:
	      /*
	       * push-occ-y
	       */

	      TRACE1("push-occ(Y%d)",arg1);

	      PUSH_OS_ENTRY(&REF(Y_REG(arg1)));
	      break;


	    case 14:
	      /*
	       * write-and-ask
	       */

	      TRACE1("write-and-ask(X%d)",arg1);

	      print_time();
	      write_and_ask(arg1);
	      break;


	    case 15:
	      /*
	       * built-in
	       */

	      TRACE1("built-in(%d)", arg1);
	      built_in(arg1);
	      break;


	    default:
	      illegal_opcode_error(s_regs.p-1,opcode);
	      break;

	    }
	  break;


	case 4:
	  arg1 = OP2ARG(opcode);
	  switch (MINOR_OPCODE(opcode))
	    {
	    case 0:
	      /*
	       * set-begin-of-term-y
	       */

	      TRACE1("set-begin-of-term(Y%d)",arg1);

	      s_flags.os_valid = TRUE;
	      s_regs.ts = &REF(Y_REG(arg1));
	      break;


	    case 1:
	      /*
	       * read-void
	       */

	      TRACE1("read-void(%d)",arg1);

	      s_glregs.sh += arg1;
	      break;

	    case 2:
	      /*
	       * read-variable-x
	       */

	      TRACE1("read-variable(X%d)",arg1);

	      term1 = *s_glregs.sh++;
	      CHANGE_ET(term1);
	      X_REG(arg1) = term1;
	      break;


	    case 3:
	      /*
	       * read-variable-y
	       */

	      TRACE1("read-variable(Y%d)",arg1);

	      term1 = *s_glregs.sh++;
	      CHANGE_ET(term1);
	      Y_REG(arg1) = term1;
	      break;

	    case 4:
	      /*
	       * read-value-x
	       */

	      TRACE1("read-value(X%d)",arg1);

	      match(*s_glregs.sh++,X_REG(arg1));
	      break;


	    case 5:
	      /*
	       * read-value-y
	       */

	      TRACE1("read-value(Y%d)",arg1);

	      match(*s_glregs.sh++,Y_REG(arg1));
	      break;


	    case 6:
	      /*
	       * match-nil
	       */

	      TRACE1("match-nil(X%d)",arg1);

	      t = &X_REG(arg1);
	      DEREF_TERMP(t);
	      if (TAG(*t) != T_AF || VAL(*t) != NIL)
		fail();
	      break;


	    case 7:
	      /*
	       * put-list-y
	       */

	      TRACE1("put-list(Y%d)",arg1);

	      CHECK_HEAP(2);
	      SET_TERM(Y_REG(arg1), T_LIST, s_regs.h);
	      break;


	    default:
	      illegal_opcode_error(s_regs.p-1,opcode);
	      break;

	    }
	  break;


	case 5:
	  arg1 = LONG_ARG(s_regs.p);
	  s_regs.p += 2;

	  switch (MINOR_OPCODE(opcode))
	    {
	    case 0:
	      /*
	       * execute
	       * Der Programmzeiger P wird auf den Code des
	       * Praedikates gesetzt
	       */

	      /* arg1 wird im Macro in zwei Argumente aufgeteilt */
	      TRACE1P("execute(%s/%d)",arg1);

#ifdef DEBUG_BOX
	      s_regs.p_box = s_glregs.box_count++;
#endif
	      if (s_glflags.debug)
		debug(D_CALL, s_regs.cp, CODE(arg1));
	      s_regs.p = CODE(arg1);
	      break;


	    case 1:
	      /*
	       * retry_me_else
	       */

	      TRACE1L("retry-me-else(L%d)",arg1);

	      BACKB->bt_regs.p = CODE(arg1);
	      break;


	    case 2:
	      /*
	       * r-retry-me-else
	       */

	      TRACE1L("r-retry-me-else(L%d)",arg1);

	      s_regs.re = s_regs.e;
	      if (s_regs.tfp == FAIL)
		{
		  s_regs.rfp = CODE(arg1);
#ifdef DEBUG_BOX
		  s_regs.rfp_box = s_regs.p_box;
#endif
		}
	      else
		{
		  s_regs.tfp = CODE(arg1);
#ifdef DEBUG_BOX
		  s_regs.tfp_box = s_regs.p_box;
#endif
		}
	      break;


	    case 3:
	      /*
	       * retry
	       */

	      TRACE1L("retry(L%d)",arg1);

	      BACKB->bt_regs.p = s_regs.p;
	      s_regs.p = CODE(arg1);
	      break;


	    case 4:
	      /*
	       * r-retry
	       */

	      TRACE1L("r-retry(L%d)",arg1);

	      s_regs.re = s_regs.e;
	      s_regs.tfp = s_regs.p;
#ifdef DEBUG_BOX
	      s_regs.rfp_box = s_regs.p_box;
#endif
	      s_regs.p = CODE(arg1);
	      break;


	    case 5:
	      /*
	       * trust
	       */

	      TRACE1L("trust(L%d)",arg1);
#if 0
	      if (s_glflags.debug)
		debug(D_REDO, s_regs.cp, s_regs.p);
#endif
	      s_regs.b = BACKB->bt_regs.b;
	      s_regs.p = CODE(arg1);
	      break;


	    case 6:
	      /*
	       * r-trust
	       */

	      TRACE1L("r-trust(L%d)",arg1);

	      s_regs.re = s_regs.e;
	      s_regs.p = CODE(arg1);
	      s_regs.tfp = FAIL;
	      break;


	    case 7:
	      /*
	       * unify-constant
	       */

	      TRACE1C("unify-constant(%s/%d)",AF_NAME(arg1),ARITY(arg1));

	      if (s_flags.rw == READ)
		{
		  term1 = *s_glregs.sh++;
		  DEREF_TERM(term1);
		  if(TAG(term1) == T_UNDEF)
		    {
		      if (trail_undef(&REF(term1)))
			SET_CONST(REF(term1), arg1);
		    }
		  else if (TAG(term1) != T_AF ||
			   polymorph_unequal(VAL(term1),arg1))
		    fail();
		}
	      else
		{
		  SET_CONST(*s_regs.h, arg1);
		  s_regs.h++;
		}
	      break;


	    case 8:
	      /*
	       * put-struct-occ
	       */

	      TRACE1C("put-struct-occ(%s/%d)",AF_NAME(arg1),ARITY(arg1));

	      TRAIL_AO();
	      CHECK_HEAP(ARITY(arg1));
	      SET_TERM(*s_regs.ao, T_STRUCT, s_regs.h);
	      SET_CONST(*s_regs.h, arg1);
	      s_regs.ao = s_regs.h++;
	      break;


	    case 9:
	      /*
	       * put-const-occ
	       */

	      TRACE1C("put-const-occ(%s/%d)",AF_NAME(arg1),ARITY(arg1));

	      TRAIL_AO();
	      SET_CONST(*s_regs.ao, arg1);
	      break;


	    case 10:
	      /*
	       * execute-rewriting
	       * Der Programmzeiger P wird auf den Rewrite-Code der
	       * Funktion gesetzt
	       */

	      /* arg1 wird im Macro in zwei Argumente aufgeteilt */
	      TRACE1P("execute-rewriting(%s/%d)",arg1);

	      s_regs.rfp = s_regs.tfp = FAIL;
#ifdef DEBUG_BOX
	      s_regs.p_box = s_glregs.box_count++;
#endif
	      if (s_glflags.debug)
		debug(D_REWRITE_A,
		      (s_regs.op->occ_cp == FAIL ?
		       s_regs.cp : s_regs.op->occ_cp), CODE(arg1));
	      s_regs.p = CODE(arg1);
	      break;


	    case 11:
	      /* 
	       * put-function-occ
	       * Falls an der akuellen Occurrence eine ungebundene
	       * Variable ist, wird die uebergebene Funktion zusammen
	       * mit den Argumenten eingesetzt.
	       */

	      TRACE1C("put-function-occ(%s/%d)",AF_NAME(arg1),ARITY(arg1));

	      if (TAG(*s_regs.ao) == T_UNDEF)
		{
		  CHECK_HEAP(ARITY(arg1));
		  SET_TERM(*s_regs.ao, T_STRUCT, s_regs.h);
		  SET_CONST(*s_regs.h, arg1);
		  s_regs.ao = s_regs.h++;
		  for (j = 1; j <= ARITY(arg1); j++)
		    *s_regs.h++ = X_REG(j);
		}
	      break;


	    case 12:
	      /*
	       * call-rewriting
	       */

	      /* arg1 wird im Macro in zwei Argumente aufgeteilt */
	      TRACE1P("call-rewriting(%s/%d)",arg1);

	      s_regs.rfp = s_regs.tfp = FAIL;
	      s_flags.os_valid = TRUE;
#ifdef DEBUG_BOX
	      s_regs.op->occ_cpb = s_regs.p_box;
	      s_regs.p_box = s_glregs.box_count++;
#endif
	      if (s_glflags.debug)
		debug(D_REWRITE_A, s_regs.p, CODE(arg1));

	      s_regs.r = TOP_LS();
	      s_regs.hr = s_regs.h;
	      s_regs.op->occ_cp = s_regs.p;
	      s_regs.p = CODE(arg1);
	      break;


	    case 13:
	      /*
	       * write-constant
	       */

	      TRACE1C("write-constant(%s/%d)",AF_NAME(arg1),ARITY(arg1));

	      SET_CONST(*s_regs.h, arg1);
	      s_regs.h++;
	      break;


	    case 14:
	      /*
	       * r-try-me-else
	       */

	      TRACE1L("r-try-me-else(L%d)",arg1);

	      s_regs.re = s_regs.e;
	      if (s_regs.rfp != FAIL)
		{
		  s_regs.tfp = CODE(arg1);
#ifdef DEBUG_BOX
		  s_regs.tfp_box = s_regs.p_box;
#endif
		}
	      else
		{
		  s_regs.rfp = CODE(arg1);
#ifdef DEBUG_BOX
		  s_regs.rfp_box = s_regs.p_box;
#endif
		}
	      break;


	    case 15:
	      /*
	       * r-try
	       */

	      TRACE1L("r-try(L%d)",arg1);

	      s_regs.re = s_regs.e;
	      s_regs.tfp = s_regs.p;
#ifdef DEBUG_BOX
	      s_regs.tfp_box = s_regs.p_box;
#endif
	      s_regs.p = CODE(arg1);
	      break;


	    default:
	      illegal_opcode_error(s_regs.p-3,opcode);
	      break;

	    }
	  break;


	case 6:
	  arg1 = LONG_ARG(s_regs.p);
	  s_regs.p += 2;

	  switch (MINOR_OPCODE(opcode))
	    {
	    case 0:
	      /*
	       * read-constant
	       */

	      TRACE1C("read-constant(%s/%d)",AF_NAME(arg1),ARITY(arg1));

	      term1 = *s_glregs.sh++;
	      DEREF_TERM(term1);
	      if (TAG(term1) != T_AF || polymorph_unequal(VAL(term1),arg1))
		fail();
	      break;


	    default:
	      illegal_opcode_error(s_regs.p-3,opcode);
	      break;

	    }
	  break;


	case 7:
	  arg2 = OP2ARG(opcode);
	  arg1 = LONG_ARG(s_regs.p);
	  s_regs.p += 2;

	  switch (MINOR_OPCODE(opcode))
	    {
	    case 0:
	      /*
	       * call
	       */

	      /* arg1 wird im Macro in zwei Argumente aufgeteilt */
	      TRACE2P ("call(%s/%d,%d)",arg1,arg2);

#ifdef DEBUG_BOX
	      s_regs.cp_box = s_regs.p_box;
	      s_regs.p_box = s_glregs.box_count++;
#endif
	      if (s_glflags.debug)
		debug(D_CALL, s_regs.p, CODE(arg1));

	      s_regs.cp = s_regs.p;
	      s_regs.p = CODE(arg1);
	      break;


	    case 1:
	      /*
	       * call-narrowing-ao
	       */

	      TRACE1("call-narrowing(ao,%d)",arg2);

	      s_flags.os_valid = TRUE;
	      if (s_regs.ao != UNDEF)
		{
#ifdef DEBUG_BOX
		  s_regs.cp_box = s_regs.p_box;
		  s_regs.p_box = s_glregs.box_count++;
#endif
		  if (s_glflags.debug)
		    debug(D_NARROW, s_regs.p, 0);
		  SET_TAG(term1, T_VAR);
		  SET_LOC(term1, SKEL);
		  arity = ARITY(VAL(*s_regs.ao));
		  for (j = 1; j <= arity; j++)
		    {
		      SET_VAL(term1, INDEX(s_regs.ao+j));
		      X_REG(j) = term1;
		    }
		  s_regs.cp = s_regs.p;
		  s_regs.p = AF_NARROW(VAL(*s_regs.ao));
		  if (s_regs.p == 0)
		    internal_error("call-af");
		}
	      break;


	    case 2:
	      /*
	       * call-rewriting-ao
	       */

	      TRACE1("call-rewriting(ao,%d)",arg2);

	      s_regs.rfp = s_regs.tfp = FAIL;
	      s_flags.os_valid = TRUE;
#ifdef DEBUG_BOX
	      s_regs.cp_box = s_regs.p_box;
	      s_regs.p_box = s_glregs.box_count++;
#endif
	      if (s_glflags.debug)
		debug(D_REWRITE, s_regs.p, 0);

	      SET_TAG(term1, T_VAR);
	      SET_LOC(term1, SKEL);
	      arity = ARITY(VAL(*s_regs.ao));
	      for (j = 1; j <= arity; j++)
		{
		  SET_VAL(term1, INDEX(s_regs.ao+j));
		  X_REG(j) = term1;
		}
	      s_regs.r = &Y_REG(arg2+1); /* = TOP_LS(); */
	      s_regs.hr = s_regs.h;
	      s_regs.cp = s_regs.p;
	      s_regs.op->occ_cp = FAIL;
	      s_regs.p = AF_REWRITE(VAL(*s_regs.ao));
	      break;


	    case 3:
	      /*
	       * try-me-else
	       */

	      TRACE2L("try-me-else(L%d,%d)",arg1,arg2);

	      top_ls = TOP_LS();
	      PUSH_B(arg2,top_ls);
	      BACKB->bt_regs.p = CODE(arg1);
	      s_regs.tfp = FAIL;
	      s_regs.rfp = FAIL;
	      break;


	    case 4:
	      /*
	       * match-structure
	       */

	      TRACE2C("match-structure(%s/%d,X%d)",AF_NAME(arg1),ARITY(arg1),arg2);

	      term1 = X_REG(arg2);
	      DEREF_TERM(term1);
	      if (TAG(term1) == T_STRUCT && VAL(REF(term1)) == arg1)
		s_glregs.sh = &REF(term1) + 1;
	      else
		fail();
	      break;


	    case 5:
	      /*
	       * try
	       */

	      TRACE2L("try(L%d,%d)",arg1,arg2);

	      top_ls = TOP_LS();
	      PUSH_B(arg2,top_ls);
	      BACKB->bt_regs.p = s_regs.p;
	      s_regs.p = CODE(arg1);
	      break;


	    case 6:
	      /*
	       * match-constant
	       */

	      TRACE2C("match-constant(%s/%d,X%d)",AF_NAME(arg1),ARITY(arg1),arg2);

	      t = &X_REG(arg2);
	      DEREF_TERMP(t);
	      if (TAG(*t) != T_AF || VAL(*t) != arg1)
		fail();
	      break;


	    case 7:
	      /*
	       * call-rewriting
	       */

	      /* arg1 wird im Macro in zwei Argumente aufgeteilt */
	      TRACE2P("call-rewriting(%s/%d,%d)",arg1,arg2);

	      s_regs.rfp = s_regs.tfp = FAIL;
	      s_flags.os_valid = TRUE;
#ifdef DEBUG_BOX
	      s_regs.cp_box = s_regs.p_box;
	      s_regs.p_box = s_glregs.box_count++;
#endif
	      if (s_glflags.debug)
		debug(D_REWRITE_A, s_regs.p, CODE(arg1));

	      s_regs.r = &Y_REG(arg2+1); /* = TOP_LS(); */
	      s_regs.hr = s_regs.h;
	      s_regs.cp = s_regs.p;
	      s_regs.op->occ_cp = FAIL;
	      s_regs.p = CODE(arg1);
	      break;


	    case 9:
	      /*
	       * put-structure-x
	       */

	      TRACE2C("put-structure(%s/%d,X%d)",AF_NAME(arg1),ARITY(arg1),arg2);

	      CHECK_HEAP(ARITY(arg1)+1);
	      SET_TERM(X_REG(arg2), T_STRUCT, s_regs.h);
	      SET_CONST(*s_regs.h, arg1);
	      s_regs.h++;
	      break;


	    case 10:
	      /*
	       * put-structure-y
	       */

	      TRACE2C("put-structure(%s/%d,Y%d)",AF_NAME(arg1),ARITY(arg1),arg2);

	      CHECK_HEAP(ARITY(arg1)+1);
	      SET_TERM(Y_REG(arg2), T_STRUCT, s_regs.h);
	      SET_CONST(*s_regs.h, arg1);
	      s_regs.h++;
	      break;


	    case 11:
	      /*
	       * get-structure
	       */

	      TRACE2C("get-structure(%s/%d,X%d)",AF_NAME(arg1),ARITY(arg1),arg2);

	      term1 = X_REG(arg2);
	      DEREF_TERM(term1);
	      if (TAG(term1) == T_UNDEF)
		{
		  if (trail_undef(&REF(term1)))
		    {
		      CHECK_HEAP(ARITY(arg1)+1);
		      SET_TERM(REF(term1), T_STRUCT, s_regs.h);
		      SET_CONST(*s_regs.h, arg1);
		      s_flags.rw = WRITE;
		      s_regs.h++;
		    }
		}
	      else if (TAG(term1) == T_STRUCT && VAL(REF(term1)) == arg1)
		{
		  s_glregs.sh = &REF(term1) + 1;
		  s_flags.rw = READ;
		}
	      else
		fail();
	      break;


	    case 13:
	      /*
	       * put-constant
	       */

	      TRACE2C("put-constant(%s/%d,X%d)",AF_NAME(arg1),ARITY(arg1),arg2);

	      SET_CONST(X_REG(arg2), arg1);
	      break;


	    case 14:
	      /*
	       * get-constant
	       */

	      TRACE2C("get-constant(%s/%d,X%d)",AF_NAME(arg1),ARITY(arg1),arg2);

	      t = &X_REG(arg2);
	      DEREF_TERMP(t);
	      if (TAG(*t) == T_UNDEF)
		{
		  if (trail_undef(t))
		    SET_CONST(*t, arg1);
		}
	      else if (TAG(*t) != T_AF || VAL(*t) != arg1)
		fail();
	      break;


	    default:
	      illegal_opcode_error(s_regs.p-3,opcode);
	      break;
	    }
	  break;


	case 8:
	  arg2 = *s_regs.p++;
	  arg1 = OP2ARG(arg2);
	  arg2 = OP1ARG(arg2);

	  switch (MINOR_OPCODE(opcode))
	    {
	    case 1:
	      /*
	       * match-value-x
	       */

	      TRACE2("match-value(X%d,X%d)", arg1, arg2);

	      match(X_REG(arg2), X_REG(arg1));
	      break;


	    case 2:
	      /*
	       * match-value-y
	       */

	      TRACE2("match-value(Y%d,X%d)", arg1, arg2);

	      match(X_REG(arg2), Y_REG(arg1));
	      break;


	    case 3:
	      /*
	       * put-variable-y
	       */

	      TRACE2("put-variable(Y%d,X%d)", arg1, arg2);

	      SET_TERM(term1, T_UNDEF, &Y_REG(arg1));

	      /* X- und Y-Register setzen */
	      Y_REG(arg1) = term1;

	      SET_TAG(term1, T_VAR);
	      X_REG(arg2) = term1;
	      break;


	    case 4:
	      /*
	       * put-value-x
	       */

	      TRACE2("put-value(X%d,X%d)", arg1, arg2);

	      X_REG(arg2) = X_REG(arg1);
	      break;


	    case 5:
	      /*
	       * put-value-y
	       */

	      TRACE2("put-value(Y%d,X%d)", arg1, arg2);

	      X_REG(arg2) = Y_REG(arg1);
	      CHANGE_ET(X_REG(arg2));
	      break;


	    case 6:
	      /*
	       * get-variable-x
	       */

	      TRACE2("get-variable(X%d,X%d)", arg1, arg2);

	      term1 = X_REG(arg2);
	      /* DEREF_TERM(term1);
	      CHANGE_ET(term1); */
	      X_REG(arg1) = term1;
	      break;


	    case 7:
	      /*
	       * get-variable-y
	       */

	      TRACE2("get-variable(Y%d,X%d)", arg1, arg2);

	      term1 = X_REG(arg2);
	      /* DEREF_TERM(term1);
	      CHANGE_ET(term1); */
	      Y_REG(arg1) = term1;
	      break;


	    case 8:
	      /*
	       * get-value-x
	       */

	      TRACE2("get-value(X%d,X%d)", arg1, arg2);

	      unify(X_REG(arg1), X_REG(arg2));
	      break;


	    case 9:
	      /*
	       * get-value-y
	       */

	      TRACE2("get-value(Y%d,X%d)", arg1, arg2);

	      unify(Y_REG(arg1), X_REG(arg2));
	      break;


	    case 10:
	      /*
	       * put-unsafe-value-y
	       */

	      TRACE2("put-unsafe-value(Y%d,X%d)", arg1, arg2);

	      term1 = Y_REG(arg1);
	      DEREF_TERM(term1);
	      if (TAG(term1) == T_UNDEF && VAL(term1) > INDEX(s_regs.e))
		{
		  if (trail_undef(&REF(term1)))
		    {
		      CHECK_HEAP(1);

		      /* neue Variable auf den Heap schreiben */
		      SET_TERM(*s_regs.h, T_UNDEF, s_regs.h);

		      /* Y_Register, auf das deref_term(arg1) zeigt setzen */
		      SET_REF(REF(term1), s_regs.h);
		      SET_TAG(REF(term1), T_VAR);

		      /* X-Register setzen */
		      X_REG(arg2) = REF(term1);

		      s_regs.h++;
		    }
		}
	      else
		{
		  CHANGE_ET(term1);
		  X_REG(arg2) = term1;
		}
	      break;


	    case 11:
	      /*
	       * put-variable-x
	       */

	      TRACE2("put-variable(X%d,X%d)", arg1, arg2);

	      CHECK_HEAP(1);

	      SET_TERM(term1, T_VAR, s_regs.h);

	      /* X-Register setzen */
	      X_REG(arg1) = X_REG(arg2) = term1;

	      SET_TAG(term1, T_UNDEF);
	      *s_regs.h++ = term1;
	      break;


	    case 12:
	      /*
	       * put-unsafe-value-x
	       */

	      TRACE2("put-unsafe-value(X%d,X%d)", arg1, arg2);

	      term1 = X_REG(arg1);
	      DEREF_TERM(term1);
	      if (TAG(term1) == T_UNDEF && VAL(term1) > INDEX(s_regs.e))
		{
		  if (trail_undef(&REF(term1)))
		    {
		      CHECK_HEAP(1);

		      /* neue Variable auf den Heap schreiben */
		      SET_TERM(*s_regs.h, T_UNDEF, s_regs.h);

		      /* Y_Register, auf das deref_term(arg1) zeigt setzen */
		      SET_REF(REF(term1), s_regs.h);
		      SET_TAG(REF(term1), T_VAR);

		      /* X-Register setzen */
		      X_REG(arg2) = REF(term1);

		      s_regs.h++;
		    }
		}
	      else
		{
		  CHANGE_ET(term1);
		  X_REG(arg2) = term1;
		}
	      break;


	    default:
	      illegal_opcode_error(s_regs.p-2,opcode);
	      break;
	    }
	  break;


	case 10:
	  arg1 = *s_regs.p++;

	  switch (MINOR_OPCODE(opcode))
	    {
	    case 0:
	      /*
	       * switch-on-structure
	       */

	      TRACE2T("switch-on-structure(%d,TABLE)", arg1);

	      if (TAG(X_REG(1)) != T_STRUCT || TAG(REF(X_REG(1))) != T_AF)
		internal_error("switch-on-structure");
	      term1 = REF(X_REG(1));
	      goto binsearch;


	    case 1:
	      /*
	       * switch-on-constant
	       */

	      TRACE2T("switch-on-constant(%d,TABLE)", arg1);

	      if (TAG(X_REG(1)) != T_AF || ARITY(VAL(X_REG(1))) != 0)
		internal_error("switch-on-constant");
	      term1 = X_REG(1);

	    binsearch:
	      {
		code_addr a, b;
		long dir;
		a = s_regs.p;
		b = s_regs.p + 4 * (arg1 - 1);
		forever
		  {
		    if (a > b)
		      {
			fail();
			break;
		      }
		    tp = a + (b - a) / 8 * 4;
		    if ((dir = VAL(term1) - LONG_ARG(tp)) == 0)
		      {
			s_regs.p = CODE(LONG_ARG(tp+2));
			break;
		      }
		    else if (dir < 0)
		      b = tp - 4;
		    else
		      a = tp + 4;
		  }
	      }
	      break;


	    default:
	      illegal_opcode_error(s_regs.p-1, opcode);
	      break;

	    }
	  break;


	case 12:
	  arg1 = LONG_ARG(s_regs.p);
	  arg2 = LONG_ARG(s_regs.p+2);
	  arg3 = LONG_ARG(s_regs.p+4);
	  arg4 = LONG_ARG(s_regs.p+6);

	  switch (MINOR_OPCODE(opcode))
	    {
	    case 0:
	      /*
	       * switch-on-term
	       */

	      TRACE4L("switch-on-term(L%d,L%d,L%d,L%d)", arg1, arg2, arg3, arg4);

	      DEREF_TERM(X_REG(1));
	      switch (TAG(X_REG(1)))
		{
		case T_UNDEF:
		  CHANGE_ET(X_REG(1));
		  s_regs.p = CODE(arg1);
		  break;

		case T_AF:
		  s_regs.p = CODE(arg2);
		  break;

		case T_LIST:
		  s_regs.p = CODE(arg3);
		  break;

		case T_STRUCT:
		  s_regs.p = CODE(arg4);
		  break;

		default:
		  internal_error("switch-on-term");
		  break;
		}
	      break;


	    default:
	      illegal_opcode_error(s_regs.p,opcode);
	      break;
	    }
	  break;


	case 14:
	  arg1 = LONG_ARG(s_regs.p);
	  arg3 = s_regs.p[2];
	  arg2 = OP2ARG(arg3);
	  arg3 = OP1ARG(arg3);
	  s_regs.p += 3;

	  switch (MINOR_OPCODE(opcode))
	    {
	    case 0:
	      /*
	       * l-try-me-else
	       * wie "try-me-else", nur zusaetzlich s_regs.b sichern
	       */

	      TRACE3L("l-try-me-else(L%d,%d,%d)",arg1,arg2,arg3);

	      if ((term *)s_regs.e < (term *)s_regs.b)
		top_ls = (term *)(s_regs.b + 1);
	      else if (s_regs.e == 0)
		top_ls = LS_MIN;
	      else
		top_ls = &Y_REG(arg3+1);
	      PUSH_B(arg2,top_ls);
	      BACKB->bt_regs.p = CODE(arg1);

	      /* Cutpunkt setzen */
	      *(struct backtrack **)&Y_REG(1) = s_regs.b;
	      s_regs.tfp = FAIL;
	      s_regs.rfp = FAIL;
	      break;

	    default:
	      illegal_opcode_error(s_regs.p-1, opcode);
	      break;
	    }
	  break;


	case MAJOR_OPCODE(DEBUG_OPCODE):
	  debug_return(--s_regs.p);
	  break;


	default:
	  illegal_opcode_error(s_regs.p-1, opcode);
	  break;
	}
    }
}
